#! /usr/bin/env false

use v6.d;

use File::Which;
use Pod::To::Anything::Subsets;
use Pod::To::Anything;
use String::Fold;
use Terminal::ANSIColor;
use Text::BorderedBlock;

unit class Pod::To::Rakuman does Pod::To::Anything;

my @footnotes;
my $tab-size = %*ENV<RAKUMAN_TAB_SIZE>.?Int // 8;
my $line-length = %*ENV<RAKUMAN_LINE_LENGTH>.?Int // 79;

multi method render (Pod::Block::Code:D $pod --> Str) {
	my $block = Text::BorderedBlock.new;
	my $body;

	if ($pod.config<lang>:exists && %*ENV<RAKUMAN_PYGMENTIZE> // '' ne '0' && which('pygmentize')) {
		my @cmd = «
			pygmentize
			-s
			-l "{$pod.config<lang>}"
		»;

		my $pygmentize = run « pygmentize », :in, :out;

		$pygmentize.in.say($pod.contents.join(''));
		$pygmentize.in.close;

		$body = $pygmentize.out.slurp(:close);
	}

	if (!$body) {
		$body = $pod.contents.join('');
	}

	self.indentify(
		$block.render(
			$body,
			:header($pod.config<name> // ''),
		),
		:!fold
	) ~ "\n\n"
}

multi method render (Pod::Block::Declarator:D $pod --> Str) { self.traverse($pod) }

multi method render (Pod::Block::Named::Author:D $pod --> Str) {
	self.colorize('bold', 'Author')
		~ "\n"
		~ self.indentify(self.unpod($pod))
		~ "\n\n"
}

multi method render (Pod::Block::Named::Name:D $pod --> Str) {
	self.colorize('bold', 'Name')
		~ "\n"
		~ self.indentify(self.unpod($pod))
		~ "\n\n"
}

multi method render (Pod::Block::Named::Subtitle:D $pod --> Str) { self.traverse($pod) }
multi method render (Pod::Block::Named::Title:D $pod --> Str) { self.traverse($pod) }

multi method render (Pod::Block::Named::Version:D $pod --> Str) {
	self.colorize('bold', 'Version')
		~ "\n"
		~ self.indentify(self.unpod($pod))
		~ "\n\n"
}

multi method render (Pod::Block::Named::Pod:D $document --> Str) {
	@footnotes = ();

	my $body = self.traverse($document);

	if (@footnotes) {
		$body ~= "\n\n"
			~ '═' x 25;

		for @footnotes.kv -> $index, $note {
			$body ~= "\n"
				~ self.indentify(self.superscript-digits($index + 1), 0)
				~ ' '
				~ $note
		}
	}

	$body
}

multi method render (Pod::Block::Para:D $pod --> Str) {
	self.indentify(self.traverse($pod)) ~ "\n\n"
}

multi method render (Pod::Block::Named:D $pod --> Str) {
	self.colorize('bold', $pod.name)
		~ "\n"
		~ self.indentify(self.traverse($pod))
		~ "\n\n"
}

multi method render (Pod::Defn:D $pod --> Str) {
	color('italic')
		~ self.indentify($pod.term)
		~ color('reset')
		~ "\n"
		~ self.indentify(self.traverse($pod), 2)
		~ "\n\n"
}

multi method render (Pod::Block::Table:D $pod --> Str) { self.traverse($pod) }

multi method render (Pod::FormattingCode::B:D $pod --> Str) {
	color('bold') ~ self.traverse($pod) ~ color('reset')
}

multi method render (Pod::FormattingCode::C:D $pod --> Str) {
	self.traverse($pod)
}

multi method render (Pod::FormattingCode::E:D $pod --> Str) { self.traverse($pod) }

multi method render (Pod::FormattingCode::I:D $pod --> Str) {
	color('italic') ~ self.traverse($pod) ~ color('reset')
}

multi method render (Pod::FormattingCode::K:D $pod --> Str) { self.traverse($pod) }

multi method render (Pod::FormattingCode::L:D $pod --> Str) {
	@footnotes
		.append($pod.meta.map({
			$_ ~~ Str ?? $_ !! self.traverse($_)
		}));

	color('underline blue')
		~ self.traverse($pod)
		~ color('reset')
		~ self.superscript-digits(@footnotes.elems)
}

multi method render (Pod::FormattingCode::N:D $pod --> Str) {
	@footnotes.append(self.traverse($pod));

	self.superscript-digits(@footnotes.elems);

	''
}

multi method render (Pod::FormattingCode::P:D $pod --> Str) { self.traverse($pod) }
multi method render (Pod::FormattingCode::R:D $pod --> Str) { self.traverse($pod) }
multi method render (Pod::FormattingCode::T:D $pod --> Str) { self.traverse($pod) }

multi method render (Pod::FormattingCode::U:D $pod --> Str) {
	color('underline') ~ self.traverse($pod) ~ color('reset')
}

multi method render (Pod::FormattingCode::V:D $pod --> Str) { self.traverse($pod) }
multi method render (Pod::FormattingCode::X:D $pod --> Str) { self.traverse($pod) }

multi method render (Pod::Heading:D $pod --> Str) {
	self.colorize('bold', self.unpod($pod)) ~ "\n"
}

multi method render (Pod::Item:D $pod --> Str) {
	my $indent-size = $tab-size * ($pod.level - 1);
	my ($first, @rest) = self
		.indentify('- ' ~ self.traverse($pod), $pod.level)
		.lines
		;

	$first ~ "\n" ~ @rest».indent(2).join("\n") ~ (@rest ?? "\n" !! '') ~ "\n"
}

#
# Helper methods
#

method colorize (
	Str:D $definition,
	Str:D $content,
	--> Str
) {
	color($definition) ~ $content ~ color('reset')
}

method indentify (
	Str:D $prose is copy,
	Int:D $levels = 1,
	Bool:D :$fold = True,
	--> Str
) {
	my $size = $levels * $tab-size;

	if ($fold) {
		$prose .= &fold(:width($line-length - $size));
	}

	$prose.lines».indent($size).join("\n")
}

method superscript-digits (
	Str() $prose,
	--> Str
) {
	$prose
		.uninames
		.map(*.subst(/^DIGIT/, 'SUPERSCRIPT').uniparse)
		.join
}

=begin pod

=NAME    Pod::To::Rakuman
=AUTHOR  Patrick Spek <p.spek@tyil.work>
=VERSION 0.1.0

=head1 Synopsis

=head1 Description

=head1 Examples

=head1 See also

=end pod

# vim: ft=perl6 noet
