package Acme::MetaSyntactic::services;
use strict;
use Acme::MetaSyntactic::List;
our @ISA = qw( Acme::MetaSyntactic::List );
our $VERSION = '1.005';
__PACKAGE__->init();

our %Remote = (
    source  => 'http://www.insecure.org/nmap/data/nmap-services',
    extract => sub {
        return grep {/^\D/}
            map { s/[#\s].*//; y!-:.+/*!_!; s/__+/_/g; $_ }
            $_[0] =~ m!^(\S+)!gm;
    },
);

1;

=head1 NAME

Acme::MetaSyntactic::services - The services theme

=head1 DESCRIPTION

The names of the services usually found in F</etc/services>.

The first list came from my Debian system and was extracted with:

    perl -lane '$_=$F[0];y/-/_/;!/#/&&!$s{$_}++&&print' /etc/services

Which was then golfed down to:

    perl -lane '$_=$F[0];y/-/_/;!/#/&&$s{$_}++||print' /etc/services
    perl -lane '$_=$F[0];y/-/_/;/#/||$s{$_}++||print' /etc/services
    perl -ne 's/\s.*//;y/-/_/;/#/||$s{$_}++||print' /etc/services
    perl -pe 's/[#\s].*//;y/-/_/;$s{$_}++&&goto LINE' /etc/services
    perl -ne 's/[#\s].*//;y/-/_/;$s{$_}++||print' /etc/services

For version 0.68, this was golfed a little more, by using the symbol
table as the hash table it is:

    perl -ne 's/[#\s].*//;y/-/_/;$$_++||print' /etc/services

The reference list is given by the IANA, and available at
L<http://www.iana.org/assignments/port-numbers>.

A bigger services lists is used by B<nmap>:
L<http://www.insecure.org/nmap/data/nmap-services>.
This list is used to update the theme.

=head1 CONTRIBUTOR

Philippe "BooK" Bruhat.

=head1 CHANGES

=over 4

=item *

2017-06-12 - v1.005

Updated from the source web site in Acme-MetaSyntactic-Themes version 1.050.

=item *

2015-10-19 - v1.004

Updated from the source web site in Acme-MetaSyntactic-Themes version 1.048.

=item *

2015-06-08 - v1.003

Updated from the source web site in Acme-MetaSyntactic-Themes version 1.046.

=item *

2014-10-13 - v1.002

Updated from the source web site in Acme-MetaSyntactic-Themes version 1.043.

=item *

2014-08-18 - v1.001

Changed the source link to use the services list from nmap,
as the I<graffiti.com> site seems to have disappeared.

Updated from the source web site in Acme-MetaSyntactic-Themes version 1.041.

=item *

2012-05-07 - v1.000

Received its own version number in Acme-MetaSyntactic-Themes version 1.000.

=item *

2006-07-24

Updated with a link to the IANA list in Acme-MetaSyntactic version 0.84.

=item *

2006-04-03

Updated and made automatically updatable with the I<graffiti.com> list
in Acme-MetaSyntactic version 0.68.

This increased the theme size by a factor of 14.

=item *

2005-08-01

Introduced in Acme-MetaSyntactic version 0.33, published on August 1, 2005.

I got the idea for this list on IRC, on the day it was published.

=back

=head1 SEE ALSO

L<Acme::MetaSyntactic>, L<Acme::MetaSyntactic::List>.

=cut

__DATA__
# names
a13_an
a14
a15
a16_an_an
a17_an_an
a1_bs
a1_msc
a21_an_1xbs
a25_fap_fgw
a26_fap_fgw
a27_ran_ran
a3_sdunode
a4_sdunode
aairnet_1
aairnet_2
aal_lm
aamp
aap
aas
abacus_remote
abarsd
abatemgr
abatjss
abbaccuray
abb_escp
abb_hw
abbs
abcsoftware
abcvoice_port
about
abr_api
abr_secure
abyss
acap
acas
accel
accelenet
accelenet_data
accessbuilder
accessnetwork
ac_cluster
accord_mgc
acc_raid
acctopus_cc
acctopus_st
accu_lmgr
accuracer
accuracer_dbms
acd_pm
ace_client
ace_proxy
ace_server
ace_svr_prop
aci
acis
acl_manager
acmaint_dbd
acmaint_transd
acme
acms
acmsoda
acnet
acp
acp_conduit
acp_discovery
acplt
acp_policy
acp_port
acp_proto
acptsys
acr_nema
ac_tech
acter
actifio_c2c
actifioreplic
actifioudsagent
activememory
active_net
activesync
activesync_notify
actnet
ada_cip
adap
adaptecmgr
adapt_sna
adcp
adi_gxp_srvprt
admd
admdog
admeng
admind
admins_lms
adobeserver_1
adobeserver_2
adobeserver_3
adobeserver_4
adobeserver_5
adrep
ads
ads_c
adtech_test
adtempusclient
advant_lm
adws
aed_512
aegate
aequus
aequus_alt
aero
aeroflight_ads
aeroflight_ret
aes_discovery
aesop
aes_x170
af
afesc_mc
affiliate
afore_vdp_disc
afp
afrog
afs
afs3_bos
afs3_callback
afs3_errors
afs3_fileserver
afs3_kaserver
afs3_prserver
afs3_rmtsys
afs3_update
afs3_vlserver
afs3_volser
aftmux
agcat
agentsease_db
agentview
agentx
agpolicy
agps_port
agri_gateway
agriserver
agslb
ah_esp_encap
aiagent
aibkup
aicc_cmi
aic_np
aic_oncrpc
aigairserver
ailith
aimpp_hello
aimpp_port_req
aipn_auth
aipn_reg
aironetddp
airport_admin
airs
airshot
airsync
aises
aja_ntv4_disc
ajp12
ajp13
aker_cdp
alaris_disc
alarm
alarm_clock_c
alarm_clock_s
alchemy
alesquery
alfin
alias
alljoyn
alljoyn_stm
allpeers
allstorcns
almobile_system
alpes
alpha_sms
alphatech_lm
alta_ana_lm
altalink
alta_smp
altav_remmgt
altav_tunnel
altbsdp
altcp
altovacentral
altova_lm
altserviceboot
amahi_anywhere
amanda
amandaidx
amberon
ambit_lm
amc
amcs
amdsched
amicon_fpsu_ra
amicon_fpsu_s
amidxtape
amiganetfs
aminet
amp
ampify
ampl_lic
ampl_tableproxy
ampr_info
ampr_inter
ampr_rcmd
amqp
amqps
ams
amt
amt_blc_port
amt_cnf_prot
amt_esd_prot
amt_redir_tcp
amt_redir_tls
amt_soap_http
amt_soap_https
amx_axbnet
amx_icsp
amx_rms
amx_webadmin
amx_weblinx
analogx
and_lm
anet
anet_b
anet_h
anet_l
anet_m
anoto_rendezv
an_pcp
ansanotify
ansatrader
ans_console
an_signaling
ansoft_lm_1
ansoft_lm_2
answersoft_lm
ansysli
ansys_lm
ansyslmd
anthony_data
antidotemgrsvr
anynetgateway
aocp
aodv
aol
aol_1
aol_2
aol_3
ap
apani1
apani2
apani3
apani4
apani5
apc_2160
apc_2161
apc_2260
apc_3052
apc_3506
apc_5454
apc_5455
apc_5456
apc_6547
apc_6548
apc_6549
apc_7845
apc_7846
apc_9950
apc_9951
apc_9952
apc_agent
apc_necmp
apcupsd
apdap
apertus_ldp
apex_edge
apex_mesh
aplx
apm_link
apocd
apogeex_port
apollo_admin
apollo_cc
apollo_data
apollo_gms
apollo_relay
apollo_status
apparenet_as
apparenet_tps
apparenet_ts
apparenet_ui
appiq_mgmt
apple_imap_admin
apple_iphoto
apple_licman
appleqtc
appleqtcsrvr
apple_sasl
appleugcontrol
apple_vpns_rp
apple_xsrvr_admin
appliance_cfg
applix
applus
applusservice
appman_server
appserv_http
appserv_https
appss_lm
appswitch_emp
appworxsrv
aprigo_cs
apri_lm
apsolab_col
apsolab_cols
apsolab_data
apsolab_rpc
apsolab_tag
apsolab_tags
apt_cacher
apwi_disc
apwi_imserver
apwi_rxserver
apwi_rxspooler
apw_registry
apx500api_1
apx500api_2
arbortext_lm
arcisdms
arcp
arcpd
arcserve
ardt
ardus_cntl
ardus_mtrns
ardusmul
ardus_trns
ardusuni
areaguard_neo
arena_server
arepa_cas
arepa_raft
argis_ds
argis_te
aria
ariel1
ariel2
ariel3
aries_kfinder
ariliamulti
aritts
arkivio
armadp
armagetronad
armcenterhttp
armcenterhttps
armi_server
armtechdaemon
arns
array_manager
ars_master
ars_vista
artifact_msg
aruba_server
asa
asam
asap_sctp
asap_sctp_tls
asap_tcp
asap_tcp_tls
asci_val
ascomalarm
asc_slmd
asctrl_agent
as_debug
asdis
asf_rmcp
asgcypresstcps
asgenf
asi
asihpi
asipregistry
asip_webadmin
asmp
asmps
asnaacceler8db
asoki_sma
aspcoordination
aspeclmd
aspen_services
aspentec_lm
asprovatalk
asr
as_servermap
assoc_disc
assuria_ins
assuria_slm
assyst_dr
astergate_disc
asterix
astrolink
astromed_main
at_3
at_5
at_7
at_8
atc_appserver
atc_lm
at_echo
atex_elmd
athand_mmp
ati_ip_to_ncpe
atlinks
atls
atmp
atmtcp
atm_uhas
atm_zip_office
at_nbp
at_rtmp
ats
atsc_mh_ssc
attachmate_g32
attachmate_s2s
attachmate_uts
att_mt_sms
atul
at_zis
audio_activmail
audiojuggler
audionews
audit
auditd
audit_transfer
aura
auriga_router
auris
aurora
aurora_balaena
aurora_cmgr
aurp
auth
authentx
autobuild
autocueds
autocuelog
autocuesmi
autocuetime
autodesk_lm
autodesk_nlm
autonoc
autopac
autotrac_acp
availant_mgr
avantageb2b
avanti_cdp
avauthsrvprtcl
avdecc
av_emb_config
avenue
avenyo
avesterra
avian
avi_nms
avinstalldisc
aviva_sna
avocent_adsap
avocentkvm
avocent_proxy
avsecuremgmt
avtp
avt_profile_1
avt_profile_2
awacs_ice
awg_proxy
aws_brf
axio_disc
axis_wimp_port
axon_lm
axon_tunnel
axysbridge
ayiya
azeti
azeti_bd
aztec
b2_license
b2n
b2_runtime
babel
backburner
BackOrifice
backroomnet
backupedge
backupexec
backup_express
bacnet
bacula_dir
bacula_fd
bacula_sd
badm_priv
badm_pub
bakbonenetvault
balour
bandwiz_system
banyan_net
banyan_rpc
banyan_vip
barracuda_bbs
base
batman
bb
bbars
bbn_mmc
bbn_mmx
bccp
bcinameservice
bcs
bcs_broker
bcs_lmserver
bcslogc
bctp
bctp_server
bdir_priv
bdir_pub
bdp
beacon_port
beacon_port_2
bears_01
bears_02
beeyond
beeyond_media
beorl
BESApi
beserver_msg_q
bess
bex_webadmin
bex_xr
beyond_remote
bfd_control
bfd_echo
bfd_lag
bfd_multi_ctl
bf_game
bflckmgr
bf_master
bftp
bgmp
bgp
bgpd
bgs_nsi
bh611
bhevent
bhfhs
bhmds
biap_mp
biff
bigbrother
biimenu
bilobit
bim_pem
binderysupport
bingbang
binkp
bintec_admin
bintec_capi
bintec_tapi
biolink_auth
bioserver
biotic
bip
bis_sync
bis_web
bitcoin
bitforestsrv
bitspeer
bittorrent_tracker
blackboard
blackice_alerts
blackice_icecap
blackjack
blaze
bl_idm
blizwow
blockade
blockade_bpsp
blocks
blp2
blp3
blp4
blp5
blueberry_lm
bluectrlproxy
bluelance
blwnkl_port
bmap
bmc_ar
bmc_ctd_ldap
bmc_data_coll
bmc_ea
bmc_gms
bmc_grx
bmc_jmx_port
bmc_messaging
bmc_net_adm
bmc_net_svc
bmc_onekey
bmcpatrolagent
bmc_patroldb
bmcpatrolrnvu
bmc_perf_agent
bmc_perf_mgrd
bmc_perf_sd
bmc_reporting
bmdss
bmpp
bnet
bnetfile
bnetgame
bngsync
b_novative_ls
bnt_manager
bo2k
board_roar
board_voip
boe_cachesvr
boe_cms
boe_eventsrv
boe_filesvr
boe_pagesvr
boe_processsvr
boe_resssvr1
boe_resssvr2
boe_resssvr3
boe_resssvr4
boe_was
boinc
boks
boks_clntd
boks_servc
boldsoft_lm
bolt
bones
boomerang
boosterware
bootclient
bootserver
borland_dsj
boscap
bosswave
bounzza
boxbackupstore
boxp
bpcd
bpcp_poll
bpcp_trap
bpdbm
bpjava_msvc
bpmd
bprd
brain
brcd
brcd_vr_req
br_channel
brcm_comm_port
brdptc
bre
brf_gw
bridgecontrol
brightcore
brlp_0
brlp_1
brlp_2
brlp_3
broker_service
brp
bruce
brutus
brvread
bsfserver_zn
bsfsvr_zn_ssl
bspne_pcc
bsquare_voip
btpp2audctr1
btpp2sectrans
btprjctrl
btrieve
bts_appserver
bts_x73
btx
buddy_draw
bues_service
bullant_rap
bullant_srap
busboy
buschtrommel
business
busycal
bv_agent
bvcdaemon_port
bvcontrol
bv_ds
bveapi
bv_is
bv_queryengine
bv_smcsrv
bvtsonar
bwnfs
bxp
bytex
bzflag
bzr
c1222_acse
c3
ca_1
ca_2
caaclang2
caacws
ca_audit_da
ca_audit_ds
cableport_ax
cab_protocol
cabsm_comm
caci_lm
cacp
cacsambroker
cadabra_lm
cadencecontrol
cadis_1
cadis_2
cadkey_licman
cadkey_tablet
cadlock
cadsi_lm
cadsisvr
cadview_3d
caerpc
caevms
caicci
caiccipc
ca_idms
caids_sensor
caistoragemgr
cajo_discovery
cal
caldsoft_backup
caller9
call_logging
call_sig_trans
calltrax
callwaveiam
camac
cambertx_lm
camp
can_dch
canditv
candp
candrp
canex_watch
can_ferret
can_ferret_ssl
canit_store
canna
can_nds
can_nds_ssl
canocentral0
canocentral1
canon_bjnp1
canon_bjnp2
canon_bjnp3
canon_bjnp4
canon_capt
canon_cpp_disc
canon_mfnp
canto_roboflow
capfast_lmd
capioverlan
capmux
caps_lm
capwap_control
capwap_data
car
CarbonCopy
cardax
cardbox
cardbox_http
cardifftec_back
CardWeb_IO
carracho
carrius_rshell
cart_o_rama
cas
casanswmgmt
cas_mapi
casp
caspssl
casrmagent
castorproxy
catalyst
catchpole
caupc_remote
cautcpd
cawas
ca_web_update
cba8
cbos_ip_port
cbp
cbserver
cbt
ccag_pib
cce3x
cce4x
ccmad
ccmail
ccmcomm
ccm_port
ccmrmi
ccnx
ccowcmr
ccp
ccproxy_ftp
ccproxy_http
ccs_software
ccss_qmm
ccss_qsm
cc_tracking
cctv_port
ccu_comm_1
ccu_comm_2
ccu_comm_3
cd3o_protocol
cdbroker
cdc
cddbp_alt
cdfunc
cdid
cdl_server
cdn
cds
cecsvc
cedros_fds
cefd_vmp
celatalk
centerline
centra
ceph
cequint_cityid
cera_bcm
cernsysmgmtagt
cert_initiator
cert_responder
cesdcdman
cesdcdtrn
cesdinv
cfdptkt
cfengine
cfs
cft_0
cft_1
cft_2
cft_3
cft_4
cft_5
cft_6
cft_7
cfw
cgi_starapi
cgms
cgn_config
cgn_stat
chargen
charsetmgr
checkoutdb
checkpoint_rtm
checksum
chevinservices
childkey_ctrl
childkey_notif
chimera_hwm
chip_lm
chipper
c_h_it_port
chmd
choiceview_agt
choiceview_clt
chromagrafx
chshell
ci3_software_1
ci3_software_2
cichild_lm
cichlid
cifs
cimple
cimplex
cim_rs
cimtrak
cindycollab
cinegrfx_elmd
cinegrfx_lm
ciphire_data
ciphire_serv
circle_x
cis
cisco_avp
ciscocsdb
cisco_fna
cisco_ipsla
cisco_net_mgmt
ciscopop
cisco_redu
cisco_sccp
cisco_snat
cisco_sys
cisco_tdp
cisco_tna
cisco_vpath_tun
cisco_wafs
cis_secure
citadel
citrixadmin
citrix_ica
citrixima
citriximaclient
citrix_rtmp
citrixupp
citrixuppg
citynl
citysearch
cl_1
clariion_evr01
classic
cl_db_attach
cl_db_remote
cl_db_request
cleanerliverc
clearcase
clearvisn
clever_ctrace
cleverdetect
clever_tcpip
client_ctrl
client_wakeup
cloanto_lm
cloanto_net_1
close_combat
cloudcheck
cloudsignaling
clp
cluster_disc
clusterxl
clutild
clvm_cfg
cm
cma
cmadmin
cmc_port
cmip_agent
cmip_man
cmmdriver
cmtp_mgt
cnap
cnckadserver
cncp
cnhrp
cnrp
cnrprotocol
cns_srv_port
coap
coaps
coauthor
cockroach
codaauth2
codasrv
codasrv_se
CodeMeter
codemeter_cmwan
codima_rtp
cogitate
cognex_dataman
cognex_insight
cognima
cogsys_lm
coherence
coldfusion_auth
collaber
collaborator
colubris
com_bardac_dw
combox_web_acc
comcam_io
commandport
commerce
commlinx_avl
commonspace
commplex_link
commtact_http
commtact_https
community
comotionback
comotionmaster
compaqdiag
compaq_evm
compaq_https
compaq_scp
compaq_wcp
composit_server
compressnet
compx_lockview
comscm
con
concert
conclave_cpp
concomp1
concurrent_lm
condor
conductor
conf
conference
conferencetalk
config_port
confluent
connect
connect_client
connected
connection
connect_proxy
connect_server
connendp
connlcli
conspiracy
consul_insight
contamac_icm
contclientms
contentserver
continuus
controlit
controlone_con
converge
coord_svr
cops
cops_tls
copy
copycat
cora
corba_iiop
corba_iiop_ssl
corbaloc
corelccam
corelvideo
corel_vncadmin
corerjd
cosir
cosmocall
couchdb
courier
covia
cp_cluster
cpdi_pidas_cm
cpdlc
cplscrambler_al
cplscrambler_in
cplscrambler_lg
cppdp
cpqrpm_agent
cpqrpm_server
cpq_tasksmart
cpq_wbem
cps
cpscomm
cpsp
cp_spxdpy
cp_spxrpts
cp_spxsvr
cpudpencap
cqg_netlan
cqg_netlan_1
creativepartnr
creativeserver
cresco_control
crestron_cip
crestron_cips
crestron_ctp
crestron_ctps
crinis_hb
crip
crmsbits
cronus
crs
cruise_config
cruise_diags
cruise_enum
cruise_swroute
cruise_update
crusecontrol
cr_websystems
cryptoadmin
crystalenterprise
crystalreports
cs_auth_svr
csbphonemaster
csccfirewall
csccredir
cscp
csc_proxy
csdm
csdmbase
csd_mgmt_port
csd_monitor
csi_lfap
csi_sgwp
cslg
cslistener
cs_live
csms
csms2
csnet_ns
csnotify
csoauth
csoft1
csoft_plusclnt
csoft_prev
csoftragent
cspclmulti
cspmlockmgr
cspmulti
cspuni
csregagent
cs_remote_db
csrpc
cssc
cs_services
cssp
cst_port
csvr
csvr_proxy
csvr_sslproxy
ct2nmcs
ctcd
ctdb
ctdhercules
ctdp
ctechlicensing
ctf
ctiprogramload
cti_redwood
ctisystemmsg
ctlptc
ctp
ctp_state
ctsd
ctt_broker
ctx_bridge
ctxlic
ctxs_vpp
cucme_1
cucme_2
cucme_3
cucme_4
cuelink
cuelink_disc
cuillamartin
cumulus
cumulus_admin
cuseeme
custix
cvc
cvc_hostd
cvd
cvmmon
cvmon
cvspserver
cvsup
cwmp
cxtp
cxws
cyaserv
cybercash
cyborg_systems
cybro_a_bus
cyc
cycleserv
cycleserv2
cylink_c
cymtec_port
cypress
cypress_stat
cytel_lm
d2000kernel
d2000webserver
d2dconfig
d2ddatatrans
d2k_datamover1
d2k_datamover2
d2k_tapestry1
d2k_tapestry2
d3winosfi
daap
dab_sti_c
dai_shell
daishi
dali_port
damewaremobgtwy
dandv_tester
danf_ak2
daqstream
darcorp_lm
dashpas_port
dasp
datacaptor
data_insurance
datalens
data_port
datascaler_ctl
datascaler_db
datasurfsrv
datasurfsrvsec
datex_asn
datusorb
davsrc
davsrcs
dawn
dayliteserver
daylitetouch
dayna
daytime
dbabble
dbase
dbbrowse
dbcontrol_agent
dbcontrol_oms
dbdb
dberegister
dbisamserver1
dbisamserver2
db_lsp
db_lsp_disc
dbm
dbref
dbreporter
dbsa_lm
dbstar
dbsyncarbiter
dc
dca
dcap
dccm
dccp_udp
d_cinema_csp
d_cinema_rrp
dcp
dcs
dcs_config
dcsl_backup
dcsoftware
dctp
dcutility
d_data
d_data_control
dddp
ddgn
ddi_tcp_2
ddi_tcp_3
ddi_tcp_4
ddi_tcp_6
ddi_tcp_7
ddi_udp_1
ddi_udp_2
ddi_udp_3
ddi_udp_4
ddi_udp_5
ddi_udp_6
ddi_udp_7
ddm_dfm
ddm_rdb
ddm_ssl
ddns_v3
ddrepl
ddt
de_cache_query
decap
decauth
decbsrv
dec_dlm
decladebug
dec_mbadmin
dec_mbadmin_h
dec_notes
dectalk
decvms_sysmgt
dei_icda
delibo
dell_eql_asm
dellpwrappks
dell_rm_port
dellwebadmin_1
dellwebadmin_2
delos_dms
delta_mcp
denali_server
de_noc
deos
deploymentmap
derby_repli
descent3
de_server
deskshare
desktop_dna
deskview
deslogin
deslogind
de_spot
devbasic
device
device2
deviceshare
dey_keyneg
dey_sapi
d_fence
dfn
dfoxserver
dfserver
dgi_serv
dgpf_exchg
dhanalakshmi
dhcpc
dhcp_failover
dhcp_failover2
dhcps
dhcpv6_client
dhcpv6_server
dhct_alerts
dhct_status
dhe
diagd
diagmond
diagnose_proc
dialogic_elmd
dialog_port
dialpad_voice1
dialpad_voice2
diameter
diameters
diamondport
di_ase
dic_aida
dicom
dicom_iscl
dicom_tls
dict
dict_lookup
di_drm
dif_port
digiman
digital_notary
digital_vrc
digivote
di_msg
direcpc_dll
direcpc_si
direcpc_video
direct
directnet
directplay
directplay8
directplaysrvr
directv_catlg
directvdata
directv_soft
directv_tick
directv_web
dirgis
dis
discard
disclose
discovery_port
discp_client
discp_server
display
distcc
distccd
distinct
distinct32
dist_upgrade
di_traceware
dixie
dj_ice
dj_ilm
dka
dkmessenger
dl_agent
dlep
dlip
dlms_cosem
dlpx_sp
dls
dls_mon
dls_monitor
dlsrap
dlsrpn
dlswpn
dmaf_caster
dmaf_server
dmdocbroker
DMExpress
dmidi
dmod_workspace
dmp
dmt
dn6_nlm_aud
dn6_smm_red
dna
dna_cml
dnap
dnc_port
dnet_keyproxy
dnet_tstproxy
dnox
dnp
dnp_sec
dns2go
dnsix
dns_llq
dnx
doc1lm
docent
doceri_ctl
docker
doc_server
docstor
documentum
documentum_s
doglms
doip_data
domain
domain_s
domaintime
domiq
donnyworld
doom
dossier
down
downtools
downtools_disc
dpap
dpcp
dpi_proxy
dpkeyserv
dpm
dpm_acm
dpm_agent
dproxy
dpserve
dpserveadmin
dpsi
dragonfly
DragonIDSConsole
drip
driveappserver
drizzle
drm_production
drmsfsd
drmsmc
drp
drwcs
ds_admin
dsatp
dsc
ds_clnt
dsdn
dserver
dsETOS
dsf
dsfgw
dslremote_mgmt
ds_mail
dsmcc_ccp
dsmcc_config
dsmcc_download
dsmcc_passthru
dsmcc_session
dsmeter_iatc
dsmipv6
dsm_scm_target
d_s_n
dsom_server
dsp
dsp3270
dssiapi
ds_slp
ds_srv
ds_srvr
ds_user
dsx_agent
dsx_monitor
dtag_ste_sb
dta_systems
dtk
dt_mgmtsvc
dtn1
dtn_bundle_tcp
dtn_bundle_udp
dtp
dtp_dia
dtp_net
dtpt
dts
dtserver_port
dtspc
dtspcd
dtv_chan_req
dt_vra
dvapps
dvbservdsc
dvcprov_port
dvl_activemail
dvr_esm
dvs
dvt_data
dvt_system
dwf
dwmsgserver
dwnmshttp
dwr
dxadmind
dx_instrument
dxmessagebase1
dxmessagebase2
dxspider
dyna_access
dyna_lm
dynamic3d
dynamid
dyniplookup
dyn_site
dzdaemon
dzoglserver
e3consultants
ea
ea1
eapol_relay
eapsp
easl
easyengine
easy_soft_mux
eba
ebinsite
e_builder
echo
echonet
ecmp
ecmport
ecnp
ecolor_imager
ecomm
ecovisiong6_1
ecp
ecsqdmn
ecwcfg
edb_server1
edb_server2
edbsrvr
e_design_net
e_design_web
editbench
edix
edm_adm_notify
edm_manager
edm_mgr_cntrl
edm_mgr_sync
edm_stager
edm_std_notify
edonkey
e_dpnet
edtools
eenet
efb_aci
efcp
efidiningport
efi_lm
efi_mg
eforward
efr
efs
eftp
egptlm
egs
ehome_ms
ehp_backup
ehs
ehs_ssl
ehtp
eicon_server
eicon_slp
eicon_x25
eims_admin
eis
eisp
eisport
eklogin
ekshell
elad
elan
elanlm
elatelink
elcn
elcsd
elektron_admin
elfiq_repl
eli
elipse_rec
Elite
ellpack
elm_momentum
elpro_tunnel
els
elvin_client
elvin_server
elxmgmt
em7_secom
ema_sent_lm
embl_ndt
emb_proj_cmd
embrace_dp_c
embrace_dp_s
emcads
emce
emc_gateway
emc_pp_mgmtsvc
emcrmirccd
emcsymapiport
emc_vcas_tcp
emc_vcas_udp
emc_xsw_dcache
emc_xsw_dconfig
e_mdu
emfis_cntl
emfis_data
emgmsg
emperion
empire_empuma
empowerid
emprise_lls
emprise_lsc
emp_server1
emp_server2
ems
emsd_port
emwavemsg
emwin
encore
encrypted_admin
encrypted_llrp
e_net
enfs
enl
enl_name
enpc
enpp
enrp
enrp_sctp
enrp_sctp_tls
ent_engine
entexthigh
entextlow
entextmed
entextnetwk
entextxid
entomb
entp
entrust_aaas
entrust_aams
entrust_ash
entrustmanager
entrust_sps
entrusttime
eor_game
eoss
epc
epicon
epl_slp
epmd
epncdp2
epnsdp
ep_nsp
eportcomm
eportcommdata
epp
eppc
ep_pcp
ept_machine
eq3_update
eq_office_4940
eq_office_4941
eq_office_4942
equationbuilder
ergolight
eristwoguns
erpc
erp_scale
erunbook_agent
erunbook_server
esbroker
escp_ip
escvpnet
es_elmd
eserver_pap
esimport
esinstall
esip
esl_lm
esmagent
esmmanager
esnm_zoning
esp
espeech
espeech_rtp
esp_encap
esp_lm
esps_portal
esri_sde
esro_emsdp
esro_gen
essbase
essp
essweb_gw
estamp
etb4j
etc_control
etcd_client
etcd_server
etebac5
etftp
ethercat
EtherNetIP_1
EtherNetIP_2
ethoscan
etlservicemgr
etp
ets
eudora_set
evb_elm
event_listener
event_port
everydayrc
evm
ev_services
evtp
evtp_data
ewall
ewcappsrv
ewctsp
ewdgs
ew_disc_cmd
ewinstaller
ew_mgmt
ewnn
e_woa
exapt_lmgr
exasoftport1
exbit_escp
exce
excerpt
excerpts
excw
exec
exlm_agent
exoconfig
exoline_tcp
exoline_udp
exonet
exp1
exp2
expresspay
extensis
extensisportfolio
eye2eye
eyelink
eyetv
ezmeeting
ezmeeting_2
ezmessagesrv
ezproxy
ezproxy_2
ezrelay
f5_globalsite
f5_iquery
facelink
facilityview
fac_restore
facsys_ntp
facsys_router
fagordnc
fairview
famdc
farenet
fastlynx
fast_rem_serv
fasttrack
fatpipe
fatserv
fax
faxcomservice
faximum
faxportwinport
faxstfx_port
fazzt_admin
fazzt_ptp
fc_cli
fc_faultnotify
fcip_port
fcis
fcis_disc
fcmsys
fcopy_server
fcopys_server
fcp
fcp_addr_srvr2
fcp_cics_gw1
fcp_srvr_inst1
fcp_srvr_inst2
fcp_udp
fc_ser
fdtracks
fdt_rcatp
febooti_aw
feitianrockey
femis
ferrari_foam
ff_annunc
ff_fms
ff_lr_port
ffserver
ff_sm
fg_fps
fg_gip
fg_sysupdate
fhc
fhsp
fibotrader_com
filecast
filemaker
filemq
filenet_powsrm
filenet_rpc
filenet_tms
filesphere
filex_lport
find
findviatv
finger
finisar
finle_lm
fintrx
fiorano_msgsvc
fiorano_rtrsvc
firefox
firemonrcc
firepower
firewall1_rdp
firstcall42
first_defense
fis
fisa_svc
fiveacross
fjappmgrbulk
fjcp
fjdmimgr
fjdocdist
fj_hdnet
fjhpjp
fjicl_tep_a
fjicl_tep_b
fjicl_tep_c
fjinvmgr
fjippol_cnsl
fjippol_polsvr
fjippol_port1
fjippol_port2
fjippol_swrly
fjitsuappmgr
fjmpcm
fjmpjps
fjmpss
fjsv_gssagt
fjsvmpor
fjswapsnp
fksp_audit
flamenco_proxy
flashfiler
flashmsg
flcrs
flexlm
flex_lm
flexlm0
flexlm1
flexlm10
flexlm2
flexlm3
flexlm4
flexlm5
flexlm6
flexlm7
flexlm8
flexlm9
flirtmitmir
fln_spx
florence
flr_agent
flukeserver
fly
fmp
fmpro_fdal
fmpro_v6
fmsas
fmsascon
fmtp
fmwp
fnet_remote_ui
fodms
foliocorp
font_service
foresyte_clear
foresyte_sec
fortisphere_vm
fotogcad
found
fpitp
fpo_fns
fprams
frc_hicp
frc_hp
frc_lp
frc_mp
frcs
freeciv
freezexservice
fronet
fryeserv
fs_agent
fsc_port
fse
fs_mgmt
fsportmap
fs_qos
fsr
fs_rh_srv
fs_server
ftnmtp
ftp
ftp_agent
ftp_data
ftp_proxy
ftps
ftps_data
ftranhc
ftrapid_1
ftrapid_2
ft_role
ftsrv
ftsync
fujitsu_dev
fujitsu_dtc
fujitsu_dtcns
fujitsu_mmpdc
fujitsu_neat
funk_dialout
funk_license
funk_logger
funkproxy
fuscript
futrix
fw1_mc_fwmodule
fw1_mc_gui
fw1_or_bgmp
fw1_secureremote
fxaengine_net
fxp
fxuptp
fyre_messanger
g2tag
g5m
gacp
gadgetgate1way
gadgetgate2way
gadugadu
gaia
galaxy4d
galaxy7_data
galaxy_network
galaxy_server
galileo
galileolog
gamegen1
gamelobby
gamesmith_port
gammafetchsvr
gandalf_lm
gap
garcon
garmin_marine
gat_lmd
gbjd816
gbmt_stars
gbs_smp
gbs_stp
gc_config
gcm_app
gcmonitor
gcp_rphy
gcsp
gdbremote
gdoi
gdomap
gdp_port
gdrive_sync
gds_adppiw_db
gds_db
gearman
gemini_lm
geneous
generalsync
geneve
genie
genie_lm
genisar_port
geniuslm
genrad_mux
genstat
geognosis
geognosisman
geolocate
gerhcs
gf
ggf_ncp
ggz
ghvpn
giga_pocket
gilatskysurfer
ginad
giop
giop_ssl
gist
git
gkrellm
glbp
gld
glishd
globalcatLDAP
globalcatLDAPssl
global_cd_port
global_dtserv
global_wlink
globe
globecast_id
globmsgsvc
glogger
glrpc
gmmp
gmrupdateserv
gntp
gnunet
gnutella
gnutella2
goahead_fldup
gog_multiplayer
goldleaf_licman
golem
go_login
gopher
gotodevice
gpfs
gppitnp
gprs_cube
gprs_data
gprs_sig
gpsd
gradecam
graphics
grasp
grcmp
grcp
gre_in_udp
gre_udp_dtls
grf_port
grid
grid_alt
gridgen_elmd
griffin
gris
groove
groove_dpp
groupwise
grubd
gsakmp
gsi
gsidcap
gsiftp
gsigatekeeper
gsmp
gsms
gsmtap
gss_http
gss_xlicen
g_talk
gtaua
gtegsc_lm
gte_samp
gtp_control
gt_proxy
gtp_user
gtrack_ne
gtrack_server
gue
guibase
guttersnex
gvcp
gv_pf
gv_us
gw
gw_asv
gw_call_port
gwen_sonya
gwha
gw_log
gxs_data_port
gxtelmd
h2250_annex_g
h225gatedisc
h248_binary
h263_video
h2gf_w_2m
h323callsigalt
h323gatedisc
h323gatestat
h323hostcall
h323hostcallsc
h323q931
hacl_cfg
hacl_gs
hacl_hb
hacl_local
hacl_monitor
hacl_poll
hacl_probe
hacl_qs
hacl_test
ha_cluster
hagel_dump
haipe_discover
haipe_otnk
halflife
hao
hap
harp
hart_ip
hassle
hawk
hbci
hb_engine
hcp_wismar
hctl
hdap
hddtemp
hde_lcesrvr_1
hde_lcesrvr_2
hdl_srv
healthd
health_polling
health_trap
heartbeat
heathview
hecmtl_db
helix
hello
hello_port
hems
here_lm
heretic2
hermes
herodotus_net
hexarc
hexen2
hfcs
hfcs_manager
hhb_gateway
hhb_handheld
hicp
hid
high_criteria
hillrserv
hinp
hiperscan_id
hip_nat_t
hippad
hiq
hislip
hivep
hivestor
hkp
hks_lm
hl7
hlibmgr
hlserver
hmmp_ind
hmmp_op
hncp_dtls_port
hncp_udp_port
hnm
hnmp
homeportal_web
homesteadglory
honyaku
hostname
hosts2_ns
hotline
hotu_chat
houdini_lm
houston
hp_3000_telnet
hp_alarm_mgr
hp_clic
hp_collector
hp_dataprotect
hp_device_disc
hpdevms
hpe_dp_bura
hp_hcip
hp_hcip_gwy
hpidsadmin
hpidsagent
hpiod
hp_managed_node
hp_nnm_data
hpnpd
hpocbus
hpoms_ci_lstn
hpoms_dps_lstn
hp_pdl_datastr
hpppssvr
hppronetman
hp_pxpib
hp_san_mgmt
hp_sca
hp_sci
hp_sco
hp_server
hp_sessmon
hpssd
hpssmgmt
hpss_ndapi
hp_status
hpstgmgr
hpstgmgr2
hpvirtctrl
hpvirtgrp
hpvmmagent
hpvmmcontrol
hpvmmdata
hpvroom
hp_webadmin
hp_webqosdb
hrd_ncs
hri_port
hrpd_ith_at_an
hsl_storm
hs_port
hsrp
hsrpv6
htcp
htrust
http
http_alt
http_mgmt
http_proxy
http_rpc_epmap
https
https_alt
https_wmap
http_wmap
httpx
htuilsrv
hub_open_net
hughes_ap
husky
hybrid
hybrid_pop
hydap
hydra
hylafax
hypercube_lm
hyper_g
hyperip
hyperscsi_port
hyperwave_isp
i3_sessionmgr
iad1
iad2
iad3
iadt
iadt_disc
iadt_tls
iafdbase
iafserver
ianywhere_dbns
iapp
ias_admind
ias_auth
iascontrol
iascontrol_oms
iasd
ias_neighbor
ias_paging
ias_reg
ias_session
iatp_highpri
iatp_normalpri
iax
ibar
iberiagames
ibm3494
ibm_abtact
ibm_app
ibm_cics
ibm_db2
ibm_db2_admin
ibm_dial_out
ibm_diradm
ibm_diradm_ssl
ibm_dt_2
ibm_mgr
ibm_mqisdp
ibm_mqseries
ibm_mqseries2
ibm_pps
ibm_res
ibm_rsyscon
ibm_ssd
ibm_wrless_lan
ibp
ibprotocol
ibridge_data
ibridge_mgmt
ibus
icabrowser
icad_el
icap
icb
iccrushmore
iceedcp_rx
iceedcp_tx
ice_location
ice_router
ice_slocation
ice_srouter
icg_bridge
icg_iprelay
icg_swp
ici
iclcnet_locate
iclcnet_svinfo
iclid
iclpv_dm
iclpv_nlc
iclpv_nls
iclpv_pm
iclpv_sas
iclpv_sc
iclpv_wsm
icl_twobase1
icl_twobase10
icl_twobase2
icl_twobase3
icl_twobase4
icl_twobase5
icl_twobase6
icl_twobase7
icl_twobase8
icl_twobase9
icmpd
icms
icon_discover
iconp
iconstructsrv
icp
icpp
icpps
icpv2
icq
ics
icshostsvc
icslap
ida_agent
idac
ida_discover1
ida_discover2
idcp
ideafarm_door
ideafarm_panic
ideesrv
iden_ralp
ident
identify
idfp
idig_mux
idmaps
idmgratm
idonix_metanet
idotdist
idp
idp_infotrieve
idps
idrs
idtp
idware_router
idxp
iec_104
iec_104_sec
ieee11073_20701
ieee_mih
ieee_mms
ieee_mms_ssl
iee_qfx
ies_lm
ifcp_port
ife_icorp
ifor_protocol
ifsf_hb_port
ifsp
igcp
igi_lm
igo_incognito
igrid
igrs
ii_admin
iiimsf
iims
iiop
IIS
IISrpc_or_vat
iiw_port
ild
ill
ilss
imagepump
imagequery
imap
imap3
imap4_ssl
imaps
imdocsvc
imgames
imink
imip
imip_channels
immedianet_bcn
imoguia_port
impera
imprs
imqbrokerd
imqstomp
imqstomps
imqtunnel
imqtunnels
imsldoc
imsp
imtc_map
imtc_mcs
imyx
incognitorv
incp
index_net
index_pc_wb
indi
indigo_server
indigo_vbcp
indigo_vrmi
indura
indx_dds
indy
inedo
i_net_2000_npr
inetfs
infi_async
infiniswitchcl
influence
infobright
infocrypt
infoexch
infolibria
infoman
infomover
informatik_lm
informer
infoseek
infotos
infowave
ingreslock
ingres_net
ininmessaging
iniserve_port
initlsmsad
innosys
innosys_acl
inova_ip_disco
inovaport1
inovaport2
inovaport3
inovaport4
inovaport5
inovaport6
insis
insitu_conf
inspect
instantia
inst_discovery
instl_bootc
instl_boots
intecom_ps1
intecom_ps2
intecourier
integral
integra_sme
integrius_stp
intelliadm_disc
intellistor_lm
intel_rci
intel_rci_mp
intelsync
interact
interactionweb
interbase
interhdl_elmd
interintelli
intermapper
interpathpanel
intersan
interserver
intersys_cache
interwise
interworld
intraintra
intrastar
int_rcv_cntrl
intrepid_ssl
intrinsa
intu_ec_client
intu_ec_svcdisc
intuitive_edge
intv
invision
invision_ag
invokator
ioc_sea_lm
io_dist_data
io_dist_group
ionixnetmon
iop
ipass
ip_blf
ipcd
ipcd3
ipcore
ipcs_command
ipcserver
ipdcesgbs
ipdd
ipdr_sp
ipdtp_port
ipether232port
ipfix
ipfixs
ipfltbcst
iphone_sync
iph_policy_adm
iph_policy_cli
iposplanet
ipp
ip_provision
ip_qsig
ipr_dglt
ipt_anri_anri
ipulse_ics
ipx
iqnet_port
iqobject
iqrm
iqserver
ique
iracinghelper
iRAPP
irc
ircs
ircs_u
ircu
irdg_post
irdmi
irdmi2
irisa
iris_beep
iris_lwz
iris_xpc
iris_xpcs
ironmail
ironstorm
irp
irtrans
is99c
is99s
isa100_gci
isakmp
isbconference1
isbconference2
iscape
ischat
iscsi
isdc
isdd
isdninfo
isg_uda_server
isi_gl
isi_irp
isis
isis_am
isis_ambc
isis_bcast
islc
ismaeasdaqlive
ismaeasdaqtest
ismc
ismserver
isnetserv
isns
isode_dua
isoft_p2p
iso_ill
iso_ip
isoipsigport_1
isoipsigport_2
isomair
iso_tp0
iso_tp0s
iso_tsap
iso_tsap_c2
ispipes
ispmmgr
isqlplus
isrp_port
issa
issc
iss_console_mgr
issd
iss_mgmt_ssl
iss_realsec
iss_realsecure
isysg_lm
ita_agent
itach
itactionserver1
itactionserver2
italk
ita_manager
itap_ddtp
itelserverport
item
itinternet
itm_lm
itm_mccs
itm_mcell_s
itm_mcell_u
ito_e_gui
itose
itscomm_ns
itu_bicc_stc
itv_control
itwo_server
iua
iuhsctpassoc
ivcollector
ivecon_port
ivmanager
ivocalize
ivsd
ivs_database
ivs_insertion
ivs_video
iwb_whiteboard
iwec
iwg1
iwlistener
iw_mmogame
iwserver
i_zipqd
izm
j_ac
jacobus_lm
jaleosnd
jamlink
jamserverport
janus_disc
jaugsremotec_1
jaugsremotec_2
jaus
java_or_OTGfileshare
jaxer_manager
jaxer_web
jaxflow
jaxflow_data
jboss_iiop
jboss_iiop_ssl
jbroker
jcp
jdatastore
jdl_dbkitchen
jdmn_port
jdp_disc
jediserver
jeol_nsddp_1
jeol_nsddp_2
jeol_nsddp_3
jeol_nsddp_4
jeol_nsdtp_1
jeol_nsdtp_2
jeol_nsdtp_3
jeol_nsdtp_4
jerand_lm
jesmsjc
jetcmeserver
jetdirect
jetform
jetformpreview
jetstream
jibe_eb
jini_discovery
j_lan_p
jlicelmd
j_link
jmact3
jmact5
jmact6
jmb_cds1
jmb_cds2
jmevt2
jmq_daemon_1
jmq_daemon_2
jms
joaJewelSuite
jomamqmonitor
joost
journee
jpegmpeg
jprinter
jps
jstel
jt400
jt400_ssl
jtag_server
jtnetd_server
jtnetd_status
jute
juxml_port
jvclient
jvl_mactalk
jvserver
jwalkserver
jwclient
jwpc
jwpc_bin
jwserver
k3software_cli
k3software_svr
ka0wuc
kademlia
kali
kana
kar2ouche
ka_sddp
kastenchasepad
kastenxpipe
kauth
k_block
kca_service
kdm
kdnet
kentrox_prot
kerberos
kerberos_adm
kerberos_master
kerberos_sec
kermit
keyserver
keyshadow
keysrvr
kfserver
kftp
kftp_data
kfxaclicensing
kingdomsonline
kingfisher
kink
kiosk
kip
kis
kitim
kjtsiteserver
klio
klogin
kme_trap_port
kmip
kmscontrol
knet_cmp
knetd
kofax_svr
konshus_lm
konspire2b
kopek_httphead
kpasswd
kpasswd5
k_patentssensor
kpdp
kpn_icw
kpop
krb524
krb5gatekeeper
krb_prop
krbupdate
kryptolan
kshell
ksysguard
ktelnet
ktickets_rest
kti_icad_srvr
kuang2
kv_agent
kvm_via_ip
kv_server
kwdb_commn
kwtc
kx
kyoceranetdev
kz_migr
l2c_control
l2c_data
l2c_disc
l2f
L2TP
l3_exprt
l3_hawk
l3_hbmon
l3_ranger
l3t_at_an
l5nas_parchan
labrat
l_acoustics
laes_bf
lam
la_maint
lan900_remote
landesk_cba
landesk_rc
landmarks
lanmessenger
lanner_lm
lanrevagent
lanrevserver
lanschool
lanserver
lansource
lansurveyor
lansurveyorxml
lanyon_lantern
laplink
launchbird_lm
lavenir_lm
lazy_ptop
lbc_control
lbc_measure
lbc_sync
lbc_watchdog
lbf
lbm
lcm_server
lcs_ap
ldap
ldap_admin
ldaps
ldapssl
ldgateway
ldoms_mgmt
ldoms_migr
ldp
lds_distrib
lds_dump
ldss
ldxp
lecroy_vicp
leecoposserver
legent_1
legent_2
leoip
lgtomapper
lhtp
libelle
liberty_lm
licensedaemon
LiebDevMgmt_A
LiebDevMgmt_C
LiebDevMgmt_DM
light
limnerpressure
link
linkname
linktest
linktest_s
linogridengine
linuxconf
linx
lionhead
lipsinc
lipsinc1
lisp_cons
lisp_control
lisp_data
lispworks_orb
listcrt_port
listcrt_port_2
listen
listmgr_port
litecoin
livelan
livestats
ljk_login
lkcmserver
llm_csv
llmnr
llm_pass
llrp
llsurfup_http
llsurfup_https
lmcs
lmdp
lm_dta
lm_instmgr
lm_mon
lmp
lm_perfworks
lms
lmsocialserver
lm_sserver
lm_webwatcher
lm_x
lnvalarm
lnvconsole
lnvmailmon
lnvmaps
lnvpoller
lnvstatus
loadsrv
loaprobe
localinfosrvr
lockd
lockstep
locus_con
locus_disc
locus_map
lofr_lm
logcabin
login
lonewolf_lm
lontalk_norm
lontalk_urgnt
lonworks
lonworks2
lorica_in
lorica_in_sec
lorica_out
lorica_out_sec
lot105_ds_upd
lotusmtap
lotusnotes
lpar2rrd
lpcp
lpdg
lpsrecommender
lrp
lrs_paging
ls3
ls3bcast
LSA_or_nterm
lsdp
lsi_bobcat
lsi_raid_mgmt
lsnr
lsp_ping
lsp_self_ping
lstp
ltctcp
ltcudp
ltp
ltp_deepspace
lumimgrd
lupa
lutap
lutcp
lv_auth
lv_ffx
lv_frontpanel
lvision_lm
lv_jc
lv_not
lv_pici
lxi_evntsvc
lyskom
m2ap
m2mservices
m2pa
m2ua
m3ap
m3da
m3ua
m4_network_as
macbak
macon
macromedia_fcs
mac_srvr_admin
madcap
madge_ltd
magaya_network
magbind
magenta_logic
magiccontrol
magicnotes
magicom
magpie
mailbox
mailbox_lm
mailprox
mailq
maincontrol
mainsoft_lm
maitrd
man
manage_exec
manet
manyone_http
manyone_xml
mao
mapper_mapethd
mapper_nodemgr
mapper_ws_ethd
mapx
marathontp
marcam_lm
markem_dcp
martalk
masc
masqdialer
matahari
matip_type_a
matip_type_b
matrix_vnet
max
maxim_asics
MaxumSP
maybe_fw1
maybe_veritas
maytagshuffle
mbap
mbap_s
mbg_ctrl
mbl_battd
mbus
mc2studios
mc3ss
mcagent
mc_appserver
mc_brk_srv
mc_client
mc_comm
mccwebsvr_port
mcer_port
mcftp
mc_gt_srv
mciautoreg
mcidas
mck_ivpip
mcns_sec
mcns_tel_ret
mcntp
mcp
mcp_port
mcreport
mcs_calypsoicf
mcs_fastmail
mcs_mailsvr
mcs_messaging
mctet_gateway
mctet_jserv
mctet_master
mctfeed
mctp
mdap_port
mdbs_daemon
md_cg_http
mdc_portmapper
mdm
mdns
mdnsresponder
mdtp
mecomm
med_ci
medevolve
med_fsp_rx
med_fsp_tx
media_agent
mediabox
mediacntrlnfsd
mediaspace
mediat
mediavault_gui
medimageportal
med_ltp
med_net_svc
med_ovw
med_supp
meetingmaker
megaco_h248
megardsvr_port
megaregsvrport
memcache
memcachedb
menandmice_dns
menandmice_lpm
menandmice_mon
menandmice_noh
menandmice_upg
mentaclient
mentaserver
mercantile
mercury_disc
meregister
mesavistaco
messageasap
messageservice
metaagent
metaconsole
meta_corp
metaedit_mu
metaedit_se
metaedit_ws
metagram
metalbend
metasage
metastorm
metasys
metatude_mds
meter
metricadbc
metrics_pas
mevent
mfcobol
mfserver
mftp
mgcp_callagent
mgcp_gateway
mgcs_mfp_port
mgemanagement
mgesupervision
mgxswitch
miami_bcast
mib_streaming
mice
micom_pfs
microcom_sbp
micromuse_lm
micromuse_ncps
micromuse_ncpw
microsan
microsoft_ds
microtalon_com
microtalon_dis
midnight_tech
mikey
mil_2045_47001
miles_apart
mimer
mindarray_ca
mindfilesys
mindprint
minecraft
minger
minilock
mini_sql
minivend
minotaur_sa
mipi_debug
mipv6tls
mira
miroconnect
mirrtex
mit_dov
miteksys_lm
mit_ml_dev
miva_mqs
mkm_discovery
mlchat_proxy
mle
mles
mloadd
mlsn
ml_svnet
mmacomm
mma_discovery
mm_admin
mmaeds
mmcal
mmcals
mmcc
mmpft
mnet_discovery
mni_prot_rout
mnotes
mnp_exchange
mns_mail
m_oap
mobileanalyzer
mobile_file_dl
mobileip_agent
mobile_p2p
mobilip_mn
MobilitySrv
mobrien_chat
moldflow_lm
molly
mon
mondex
monetra
monetra_admin
mongod
monitor
monkeycom
monp
montage_lm
mortgageware
mosaicsyssvc1
mosaixcc
MOS_aux
MOS_lower
mosmig
MOS_soap
MOS_soap_opt
MOS_upper
mount
mountd
movaz_ssc
moy_corp
mpc_lifenet
mpfoncl
mpfwsas
mphlpdmc
mpidcagt
mpidcmgr
mpl_gprs_port
mpls_pm
mpls_udp
mpls_udp_dtls
mpm
mpm_flags
mpm_snd
mpnjsc
mpnjsocl
mpnjsomb
mpnjsomg
mpnjsosv
mpp
mppolicy_mgr
mppolicy_v5
mpshrsv
mps_raft
mpsserver
mpsysrmsvr
mptn
mqe_agent
mqe_broker
mqtt
mrip
mrm
mrssrendezvous
ms_alerter
msantipiracy
ms_cluster_net
msdfsr
msdp
msdtc
msdts1
msexch_routing
msfrs
msft_dpm_cert
msft_gc
msft_gc_ssl
msfw_array
msfw_control
msfw_replica
msfw_s_storage
msfw_storage
msg_auth
msgclnt
msg_icp
msgsrvr
msgsys
mshvlm
msiccp
msi_cps_rm
ms_ilm
ms_ilm_sts
msims
msi_selectplay
ms_la
ms_licensing
msl_lmd
ms_lsa
msmq
msmq_mgmt
msnp
ms_olap1
ms_olap2
ms_olap3
ms_olap4
msolap_ptp2
msp
msp_os
msql
ms_rome
msrp
msrpc
msr_plugin_port
ms_rule_engine
ms_shuttle
ms_sideshow
ms_smlbiz
ms_sna_base
ms_sna_server
ms_sql2000
ms_sql_m
ms_sql_s
msss
ms_s_sideshow
ms_streaming
ms_theater
mstmg_sstp
ms_v_worlds
ms_wbt_server
msync
mtcevrunqman
mtcevrunqss
mti_tcs_comm
mtl8000_matrix
mtn
mtp
mtportmon
mtport_regist
mtqp
mtrgtrans
mt_scaleserver
mtsserver
multicast_ping
multidropper
multiling_http
multiplex
multip_msg
mumps
munin
mupdate
murmur
murray
murx
muse
musiconline
must_backplane
must_p2p
mvel_lm
mvs_capacity
mvx_lm
m_wnn
mxi
mxit
mxodbc_connect
mxomss
mxxrlogin
myblast
mylex_mapd
mylxamport
mynahautostart
myq_termlink
myrtle
mysql
mysql_cluster
mysql_cm_agent
mysql_im
mysql_proxy
mysqlx
mythtv
mzca_action
mzca_alert
n1_fwp
n1_rmgmt
n2h2server
n2nremote
n2receive
naap
nacagent
nacnl
na_er_tip
na_localise
namemunge
nameserver
namp
nani
napster
nas
nasmanager
nas_metering
natdataservice
nati_dstp
nati_logos
nati_svrloc
nati_vi_server
nat_pmp
nat_pmp_status
nat_t_ike
nattyserver
natuslink
navbuddy
nav_data_cmd
navegaweb_port
navisphere
navisphere_sec
nav_port
nbd
nbdb
nbt_pc
nbt_wol
nburn_id
nbx_au
nbx_cc
nbx_dir
nbx_ser
ncacn_ip_tcp
ncadg_ip_udp
ncconfig
ncd_conf
ncd_conf_tcp
ncd_diag
ncd_diag_tcp
ncdloadbalance
ncdmirroring
ncd_pref
ncd_pref_tcp
nced
ncl
ncld
ncp
ncpm_ft
ncpm_hip
ncpm_pm
ncr_ccl
ncu_1
ncu_2
ncube_lm
ncxcp
ndl_aas
ndl_ahp_svc
ndl_als
ndl_aps
ndl_tcp_ois_gw
ndm_agent_port
ndmp
ndmps
ndm_requester
ndm_server
ndn
ndnp
ndsauth
ndsconnect
ndsp
nds_sso
ndtp
neckar
necp
nec_raidplus
nei_management
neo4j
neod1
neod2
neoiface
neon24x7
nerv
nesh_broker
nessus
nessus_xmlrpc
nest_protocol
net2display
net8_cman
netadmin
netagent
netangel
netapp_icdata
netapp_icmgmt
netapp_sync
netarx
netaspi
netassistant
netattachsdmp
netbackup
netbill_auth
netbill_cred
netbill_keyrep
netbill_prod
netbill_trans
netbios_dgm
netbios_ns
netbios_ssn
netblox
netbookmark
netboot_pxe
netbus
netcabinet_com
netcelera
netchat
netcheque
netclip
netcomm1
netcomm2
netconf_beep
netconf_ch_ssh
netconf_ch_tls
netconfsoapbeep
netconfsoaphttp
netconf_ssh
netconf_tls
netcp
netdb_export
net_device
neteh
neteh_ext
netgw
netinfo
netinfo_local
netiq
netiq_endpoint
netiq_endpt
netiq_mc
netiq_ncap
netiq_qcheck
netiq_voipa
netlabs_lm
netmagic
netmap_lm
netml
netmo_default
netmo_http
netmon
netmount
netmpi
netnews
netobjects1
netobjects2
neto_dcs
netopia_vo1
netopia_vo2
netopia_vo3
netopia_vo4
netopia_vo5
netop_rc
netops_broker
netop_school
neto_wol_server
netperf
netplan
netplay_port1
netplay_port2
netport_id
net_projection
netprowler_manager
netprowler_manager2
netprowler_sensor
netrcs
netrek
netrisk
netrix_sftm
netrjs_1
netrjs_2
netrjs_3
netrjs_4
netrockey6
netsaint
netsc_dev
netsc_prod
netscript
netserialext1
netserialext2
netserialext3
netserialext4
netspeak_acd
netspeak_cps
netspeak_cs
netspeak_is
netstat
netsteward
net_steward
netsupport
netsupport2
nettest
nettgain_nms
netuitive
netvenuechat
netview_aix_1
netview_aix_10
netview_aix_11
netview_aix_12
netview_aix_2
netview_aix_3
netview_aix_4
netview_aix_5
netview_aix_6
netview_aix_7
netview_aix_8
netview_aix_9
netviewdm1
netviewdm2
netviewdm3
netwall
netware_csp
netware_ip
netwatcher_db
netwatcher_mon
netwave_ap_mgmt
netwkpathengine
networklens
networklenss
netx_agent
netxms_agent
netxms_mgmt
netxms_sync
netx_server
neveroffline
newacct
newbay_snc_mc
newgenpay
newheights
newlixconfig
newlixengine
newlixreg
newoak
new_rwho
news
newton_dock
newwavesearch
nexentamv
nexgen
nexgen_aux
nexstorindltd
nextstep
nexus_portal
nfa
nfapi
nfoldman
nfs
nfsd_keepalive
nfsd_status
NFS_or_IIS
nfsrdma
ng_control
ng_umds
nhci
nhserver
nicelink
nicetec_mgmt
nicetec_nmsvc
ni_ftp
nifty_hmi
nilinkanalyst
nim
ni_mail
nimaux
nimbusdb
nimbusdbctrl
nimcontroller
nimgtw
nimhub
nimreg
nimrod_agent
nimsh
nimspooler
nim_vdrshell
nim_wan
ninaf
niobserver
nip
niprobe
nirp
nitrogen
ni_visa_remote
njenet_ssl
nkd
nlg_data
nlogin
nls_tl
nmap
nmasoverip
nm_asses_admin
nm_assessor
nmc_disc
nmea_0183
nmea_onenet
nm_game_admin
nm_game_server
nmmp
nms
nmsd
nms_dpnss
nmsigport
nmsp
nmsserver
nms_topo_serv
nnp
nnsp
nntp
noaaport
noadmin
noagent
nod_client
nod_provider
noit_transport
nokia_ann_ch1
nokia_ann_ch2
nomad
nomdb
norton_lambert
notateit
notateit_disc
noteit
noteshare
noteza
notezilla_lan
notify
notify_srvr
novar_alarm
novar_dbase
novar_global
novastorbakcup
novation
novell_ipx_cmd
novell_lu6_2
novell_zen
nowcontact
npdbgmngr
npds_tracker
npep_messaging
nping_echo
npmp
npmp_gui
npmp_local
npmp_trap
npp
nppmp
npqes_test
npsp
nq_ap
nqs
nrcabq_lm
nrpe
ns
nsc_ccs
ns_cfg_server
nsc_posa
nsdeepfreezectl
nsesrvr
nsiiops
nsjtp_ctrl
nsjtp_data
nsp
nsrexecd
nsrp
nss
nssagentmgr
nssalertmgr
ns_server
nssocketport
nss_routing
nsstp
nst
nsw_fe
nta_ds
ntalk
nta_us
ntp
ntz_p2p_storage
ntz_tracker
nuauth
nucleus
nucleus_sand
nufw
nupaper_ss
nusrp
nut
nuts_bootp
nuts_dem
nuxsl
nvc
nvcnet
nvd
nvm_express
nvmsgd
nw_license
nxlmd
o2server_port
oap
oap_admin
oap_s
oa_system
obex
objcall
objective_dbc
objectmanager
obrpd
observium_agent
ocbinder
oceansoft_lm
oce_snmp_trap
ock
oc_lm
ocs_amu
ocs_cmu
ocserver
octopus
octopustentacle
odbcpathway
odette_ftp
odette_ftps
odeumservlink
odi_port
odmr
odn_castraq
odnsp
odsi
oem_agent
oemcacao_jmxmp
oemcacao_rmi
oemcacao_websvc
officelink2000
office_tools
ofsd
oftep_rpc
ogs_client
ogs_server
ohimsrv
ohmtrigger
ohsc
oi_2000
oidocsvc
oidsr
oirtgsvc
olhost
olsr
olsv
omabcastltkm
omad
oma_dcdocbs
oma_ilp
oma_ilp_s
oma_mlp
oma_mlp_s
oma_rlp
oma_rlp_s
omasgport
oma_ulp
omfs
omginitialrefs
omhs
omid
omnilink_port
omnisky
omnivision
omnivisionesx
oms
omscontact
omsdk
omserv
oms_nonsecure
omstopology
omviagent
omviserver
onbase_dds
onehome_help
onehome_remote
onep_tls
onesaf
onmux
onpsocket
onscreen
ontime
ontobroker
oob_ws_http
opalis_rbt_ipc
opalis_rdv
opalis_robot
opc_job_start
opc_job_track
opcon_xps
opcua_tcp
opcua_tls
opcua_udp
opencm
opencore
opendeploy
openflow
openhpid
openit
openmail
openmailg
openmailns
openmailpxy
openmanage
openmath
opennl
opennl_voice
openport
openqueue
openremote_ctrl
openstack_id
opentable
opentrac
openvms_sysipc
openvpn
openwebnet
opequus_server
opi_sock
op_probe
opsec_cvp
opsec_ela
opsec_lea
opsec_omi
opsec_sam
opsec_uaa
opsec_ufp
opsession_clnt
opsession_prxy
opsession_srvr
opsmessaging
opsmgr
opsview_envoy
opswagent
opswmanager
optech_port1_lm
optika_emedia
optilogic
optimanet
optima_vnet
optiwave_lm
optocontrol
optohost002
optohost003
optohost004
opus_services
oracle
oracleas_https
oracle_em1
oracle_em2
oracle_ms_ens
oraclenames
oraclenet8cman
oracle_oms
oracle_vp1
oracle_vp2
ora_lm
ora_oap
orasrv
orbiter
orbix_cfg_ssl
orbix_config
orbixd
orbix_locator
orbix_loc_ssl
orbplus_iiop
ordinox_dbase
ordinox_server
origo_sync
orion
orion_rmi_reg
ortec_disc
osaut
osdcp
os_licman
osm_appsrvr
osm_oev
osmosis_aeea
osp
ospfd
ospf_lite
osu_nms
osvr
osxwebadmin
otlp
otmp
otp
otpatch
ott
ottp
otv
outlaws
ovalarmsrv
ovalarmsrv_cmd
ovbus
oveadmgr
overnet
ovhpas
ovladmgr
ov_nnm_websrv
ovobs
ovrimosdbman
ovsam_d_agent
ovsam_mgmt
ovsdb
ovsessionmgr
ovtopmd
ovwdb
owamp_control
owms
owserver
p25cai
p2pcommunity
p2pgroup
p2pq
p2p_sip
p4p_portal
p6ssmc
pace_licensed
pacerforum
pacmand
pacom
padl2sim
pads
pafec_lm
paging_port
pago_services1
pago_services2
palace_1
palace_2
palace_3
palace_4
palace_5
palace_6
palcom_disc
pammratc
pammrpc
pana
panagolin_ident
panasas
pando_pub
pando_sec
pangolin_laser
paradym_31
paragent
parallel
park_agent
parliant
parsec_game
parsec_master
parsec_peer
partimage
passgo
passgo_tivoli
password_chg
passwrd_policy
patrol
patrol_coll
patrol_ism
patrol_mq_gm
patrol_mq_nm
patrol_snmp
patrolview
pawserv
paycash_online
paycash_wbp
pay_per_view
payrouter
pcanywhere
pcanywheredata
pcanywherestat
pcc_image_port
pcc_mfp
pcduo
pcduo_old
pcep
pciarray
pcia_rxp_b
pcihreq
pcle_infex
pclemultimedia
pcm
pcmail_srv
pcmk_remote
pc_mta_addrmap
pcnfs
pcoip
pcoip_mgmt
pconnectmgr
pcptcpservice
pcs_pcw
pcs_sf_ui_man
pcsync_http
pcsync_https
pc_telecommute
pctrader
pcttunnell
pda_data
pd_admin
pda_gate
pdap
pdap_np
pda_sys
pdb
pdefmns
pdnet
pdp
pdps
pdrncs
pds
pdtp
pduncs
pearldoc_xact
peerbook_port
peerenabler
peerwire
pegasus
pegasus_ctl
pegboard
pehelp
pe_mike
pentbox_sim
peocoll
peoctlr
peport
perfd
perf_port
perimlan
periscope
permabit_cs
perrla
persona
personal_link
personalos_001
personnel
pfcp
pftp
pfu_prcallback
pgbouncer
pgps
ph
pharmasoft
pharos
philips_vc
phoenix_rpc
phonebook
phonex_port
photuris
phrelay
phrelaydbg
piccolo
pichat
picknfs
picodbc
pictrography
pim_port
pim_rp_disc
pinghgl
ping_pong
pip
pipes
pipe_server
piranha1
piranha2
pirp
pit_vpn
pixelpusher
pjj_player
pjlink
pk
pkagent
pk_electronics
pkix_3_ca_ra
pkix_cmc
pkix_timestamp
pksd
pktcable_cops
pktcablemmcops
pkt_krb_ipsec
plato
plato_lm
playsta2_app
playsta2_lob
plbserve_port
plcy_net_svcs
plethora
plex
plgproxy
pluribus
plysrv_http
plysrv_https
pmas
pmcd
pmcdproxy
pm_cmdsvr
pmcp
pmcs
pmd
pmdfmgt
pmdmgr
pmip6_cntl
pmip6_data
pmsm_webrctl
pmwebapi
pnaconsult_lm
pnbs
pnbscada
pnet_conn
pnet_enc
p_net_local
p_net_remote
pn_requester
pn_requester2
pnrp_port
pns
polestar
policyserver
polipo
pon_ictp
pop2
pop3
pop3pw
pop3s
popup_reminders
portgate_auth
postgresql
pov_ray
PowerAlert_nsa
powerburst
powerchute
powerchuteplus
powerclientcsf
powerexchange
powergemplus
powerguardian
poweronnud
powerschool
powwow_client
powwow_server
ppactivation
ppcontrol
ppp
ppsms
ppsuitemsg
pptconference
pptp
pq_lic_mgmt
pqsflows
pqsp
pra_elmd
prat
prchat_server
prchat_user
precise_comm
precise_i3
precise_sft
precise_vip
predatar_comms
prelude
presence
presonus_ucnet
press
prex_tcp
prgp
primaserver
princity_agent
printer
printer_agent
printopia
print_srv
priority_e_com
prism_deploy
prismiq_plugin
privateark
privatechat
privatewire
priv_dial
priv_file
privilege
priv_mail
privoxy
priv_print
priv_rje
priv_term
priv_term_l
prizma
prm_nm
prm_nm_np
prm_sm
prm_sm_np
prnrequest
prnstatus
proactivate
proactivesrvr
proaxess
procos_lm
prodigy_intrnet
productinfo
pro_ed
profile
profilemac
profinet_cm
profinet_rt
profinet_rtm
progistics
programmar
prolink
proofd
propel_msgsys
proremote
proshare1
proshare2
proshareaudio
prosharedata
proshare_mc_1
proshare_mc_2
prosharenotify
prosharerequest
prosharevideo
prospero
proxim
proxima_lm
proxy_gateway
proxy_plus
prp
prRegister
prsvp
ps_ams
psbserver
PS_Capture_Pro
pscl_mgt
pscribe
pscupd
psdbserver
pserver
psi_ptt
pslicser
psl_management
pslserver
psmond
psprserver
pspserver
psrserver
pss
pssc
PS_Server
pt2_discover
ptcnameservice
ptk_alink
ptp
ptp_event
ptp_general
pt_tls
publiqare_sync
pulseaudio
pulsonixnls
pumpkindb
puparp
puppet
purenoise
purityrpc
pushns
pvaccess
pvsw
pvsw_inet
pvuniwien
pvxpluscs
pvxplusio
pwdgen
pwdis
pwgippfax
pwgpsi
pwgwims
pwrsevent
pxc_epmap
pxc_ntfy
pxc_pin
pxc_roid
pxc_sapxom
pxc_splr
pxc_splr_ft
pxc_spvr
pxc_spvr_ft
pyrrho
pythonds
q3ade
q55_pcc
qadmifevent
qadmifoper
qaz
qbdb
qb_db_server
qbikgdp
qcp
qdb2service
qencp
qfp
qft
qftest_lookup
qip_audup
qip_login
qip_msgd
qip_qdhcp
qke_llc_v3
qmqp
qmtps
qmvideo
qnts_orb
qnxnetman
qo_secure
qotd
qotps
qpasa_agent
qptlmd
qrh
qsc
qsm_gui
qsm_proxy
qsm_remote
qsnet_assist
qsnet_cond
qsnet_nucl
qsnet_trans
qsnet_workst
qsoft
qtms_bootstrap
qtp
qt_serveradmin
quaddb
quailnet
quake
quake2
quake3
quakeworld
quantastor
quartus_tcl
quasar_server
qubes
quest_agent
quest_data_hub
questdb2_lnchr
quest_disc
questnotify
quest_vista
queueadm
quickbooksrds
quicksuite
quicktime
quosa
quotad
qvr
qwave
raadmin
racf
radacct
radan_http
radclientport
radec_corp
radio
radio_bc
radio_sm
radius
radius_acct
radius_dynauth
radix
radmin
radmind
radmin_port
radpdf
rads
radsec
radware_rpm
radware_rpm_s
radwiz_nms_srv
raid_ac
raid_am
raid_cc
raid_cd
raid_cs
raid_sf
ramp
rancher
rap
rapi
rapidbase
rapidmq_center
rapidmq_reg
rapido_ip
rap_ip
rap_listen
rap_service
raqmon_pdu
rasadv
ratio_adp
ratl
ravehd
raven_rdp
raven_rmp
raventbs
raventdm
raw_serial
raxa_mgmt
razor
rbakcup1
rbakcup2
rblcheckd
rbr_debug
rbr_discovery
rbsystem
rbt_smc
rbt_wanopt
rcc_host
rcip_itu
rcp
rcst
rcts
rda
rdb_dbs_disp
RDCenter
rdc_wh_eos
rdgs
rdlap
rdmnet_ctrl
rdm_tfs
rdrmshc
rds
rds2
rds_ib
rds_ip
re101
reachout
realm_rusd
realsecure
realserver
rebol
recipe
re_conn_proto
recvr_rc
redis
redstone_cpss
redstorm_diag
redstorm_find
redstorm_info
redstorm_join
redwood_chat
reftek
registrar
relief
rellpack
re_mail_ck
remcap
remctl
remographlm
remoteanything
remote_as
remote_collab
remotedeploy
remotefs
remote_kis
remoteware_un
remote_winsock
rendezvous
repcmd
repliweb
repscmd
repsvc
res
resacommunity
rescap
resorcs
resource_mgr
responselogic
responsenet
res_sap
restconf_ch_tls
resvc
retp
retrospect
rets
RETS_or_BackupExec
rets_ssl
reversion
rexecj
rfa
rfb
rfe
rfid_rp1
rfio
rfmp
rfx_lm
rgtp
rhp_iibp
rib_slm
ricardo_lm
rich_cp
rid
ridgeway1
ridgeway2
rightbrain
rimf_ps
rimsl
ripd
ripng
ripngd
ris
ris_cm
rise
rjcdb_vcards
rje
rkb_oscs
rkinit
rlm
rlm_admin
rlp
rlzdbase
rmc
rmiactivation
rmiaux
rmiregistry
rmlnk
rmonitor
rmonitor_secure
rmopagt
rmpp
rmt
rmtserver
rna
rna_lm
rndc
rnm
rnmap
rnrp
robcad_lm
robix
roboeda
roboer
robotraconteur
robot_remote
roce
rockwell_csp1
rockwell_csp2
rocrail
roketz
rootd
rothaga
route
routematch
roverlog
rpasswd
rpc2portmap
rpcbind
rpi
rpki_rtr
rpki_rtr_tls
rplay
rp_reputation
rprt
rrac
rrdp
rrh
rrifmm
rrilwm
rrimwm
rrirtr
rrisat
rrp
rsap
rscd
rsc_robot
rscs
rsf_1
rsftp
rsh_spx
rsip
rsisysaccess
rsms
rsmtp
rsom
rs_pias
rsqlserver
rs_rmi
rs_status
rsvd
rsvp_encap_1
rsvp_encap_2
rsvp_tunnel
rsync
rt_classmanager
rtcm_sc104
rtc_pm_port
rt_devicemapper
rtelnet
rt_event
rt_event_s
rt_helper
rtimeviewer
rtip
rt_labtracker
rtmp
rtmp_port
rtnt_1
rtnt_2
rtps_dd_mt
rtps_dd_ut
rtps_discovery
rtraceroute
rtsclient
rt_sound
rtsp
rtsp_alt
rtsps
rtsserv
rt_viewer
rugameonline
rusb_sys_port
rushd
rvs_isdn_dcp
rwhois
rwp
rww
rxapi
rxe
rxmon
s102
s1_control
s3db
s8_client_port
sabams
sabarsd
sabp_signal
sac
sacred
sae_urn
safetynetp
saft
sage_best_com1
sage_best_com2
sagectlpanel
sagxtsds
sah_lm
saiseh
sai_sentlm
salient_dtasrv
salient_mux
salient_usrmgr
samba_swat
samd
sa_msg_port
samsung_disc
samsung_unidex
sanavigator
sane_port
sanity
santak_ups
santools
sapcomm
sapeps
saphostctrl
saphostctrls
saposs
saprouter
sapv1
saratoga
saris
sas_1
sas_2
sas_3
sasg
sasggprs
sasp
sas_remote_hlp
satvid_datalnk
sauterdongle
savant
sbackup
sbcap
s_bfd
sbi_agent
sbl
sbook
scaleft
scan_change
scanstat_1
sccip_media
scc_security
sceanics
scenccs
scenidm
scientia_sdb
scientia_ssdb
scinet
scintilla
scipticslsrvr
sco_aip
sco_dtmgr
scohelp
scoi2odialog
sco_inetmgr
scol
sco_peer_tta
scoremgr
sco_sysmgr
scotty_disc
scotty_ft
sco_websrvrmg3
sco_websrvrmgr
scp
scp_config
scpi_raw
scpi_telnet
scrabble
screencast
scriptview
scscp
scservp
scte104
scte30
sctp_tunneling
scup
scup_disc
scx_proxy
sd
sdadmind
sdbproxy
sd_capacity
sdclient
sd_data
sddp
sde_discovery
sd_elmd
sdfunc
sdhelp
sdl_ets
sdlog
sdmmp
sdnskmp
sdo
sdo_ssh
sdo_tls
sdp_id_port
sdp_portmapper
sdproxy
sdr
sdreport
sd_request
sds
sds_admin
sdsc_lm
sdserv
sdserver
sdt
sdt_lmd
sdtvwcam
sdxauthd
seagull_ais
seagulllms
seaodbc
search
search_agent
seaview
seclayer_tcp
seclayer_tls
sec_ntb_clnt
sec_pc2fax_srv
secrmmsafecopya
sec_t4net_clt
sec_t4net_srv
secure_aux_bus
secure_cfg_svr
secureidprop
secure_mqtt
securenetpro_sensor
secure_ts
securid
securitychase
seispoc
semantix
semaphore
send
senip
senomix01
senomix02
senomix03
senomix04
senomix05
senomix06
senomix07
senomix08
sentinel
sentinel_ent
sentinel_lm
sentinelsrm
sent_lm
sentlm_srv2srv
seosload
sep
seraph
sercomm_scadmin
sercomm_wlink
serialgateway
serialnumberd
serverdocs
server_find
servergraph
serverstart
serverview_as
serverview_asn
serverviewdbms
serverview_gf
serverview_icc
serverview_rm
serverwsd2
servexec
service_ctrl
servicemeter
servicetags
servistaitsm
servserv
servstat
sesi_lm
set
sflm
sf_lm
sflow
sfm_db_server
sfmsso
sfs_config
sfs_smp_net
sftdst_port
sftp
sftsrv
sftu
sgcip
sgcp
sge_execd
sge_qmaster
sgi_arrayd
sgi_dgl
sgi_dmfmgr
sgi_esphttp
sgi_eventmond
sgi_lk
sgi_soap
sgi_storman
sg_lm
sgmp
sgmp_traps
sgsap
shadowserver
shadowstream
shaperai
shareapp
sharp_server
shell
shiprush_d_ch
shiva_confsrvr
shivadiscovery
shivahose
shivasound
shockwave
shockwave2
shofarplayer
shrewd_control
shrinkwrap
sia_ctrl_plane
siam
sicct
sicct_sdp
siebel_ns
siemensgsm
sieve
sift_uft
sightline
sigma_port
signacert_agent
signal
signet_ctf
siipat
silc
silhouette
silkmeter
silkp1
silkp2
silkp3
silkp4
silverpeakcomm
silverpeakpeer
silverplatter
simba_cs
simbaexpress
simbaservices
simco
sim_control
simctlp
simon
simp_all
simplement_tie
simple_push
simple_push_s
simple_tx_rx
simplifymedia
simslink
sip
sip_tls
sis_emt
sitaradir
sitaramgmt
sitaraserver
sitewatch
sitewatch_s
sixchat
six_degrees
sixnetudr
sixtrak
sixxsconfig
sj3
skip_cert_recv
skip_cert_send
skip_mc_gikreq
skkserv
skronk
skytelnet
sky_transport
slc_ctrlrloops
slc_systemlog
slim_devices
slingshot
slinkysearch
slinterbase
slmap
slm_api
slnp
slp
slp_notify
slscc
slslavemon
slush
smaclmgr
smakynet
smallchat
smap
smar_se_port1
smar_se_port2
smartcard_port
smartcard_tls
smart_diagnose
smart_install
smart_lm
smartpackets
smartsdp
sma_spw
smauth_port
smbdirect
smc_admin
smc_http
smc_https
smc_jmx
smcluster
smile
smip
smip_agent
smntubootstrap
sm_pas_1
sm_pas_2
sm_pas_3
sm_pas_4
sm_pas_5
smpnameres
smpp
smpppd
smpte
sms_chat
smsd
smsp
smsqp
sms_rcinfo
sms_remctrl
sms_xfer
smtp
smtps
smux
smwan
snac
sna_cs
snagas
snap
snapd
snapenetio
snapp
snare
snaresecure
sncp
s_net
snet_sensor_mgmt
snews
snifferclient
snifferdata
snifferserver
snip_slave
snmp
snmpdtls
snmpdtls_trap
snmpssh
snmpssh_trap
snmp_tcp_port
snmptls
snmptls_trap
snmptrap
snpp
sns_admin
sns_agent
sns_channels
sns_credit
sns_dispatcher
sns_gateway
sns_protocol
sns_query
sns_quote
snss
sntlkeyssrvr
sntp_heartbeat
soagateway
soap_beep
soap_http
socalia
social_alarm
socks
socorfs
socp_c
socp_t
softaudit
softcm
softdataphone
softpc
softrack_meter
solaris_audit
solera_epmap
solera_lpn
solid_e_engine
solid_mux
solve
sometimes_rpc1
sometimes_rpc10
sometimes_rpc11
sometimes_rpc12
sometimes_rpc13
sometimes_rpc14
sometimes_rpc15
sometimes_rpc16
sometimes_rpc17
sometimes_rpc18
sometimes_rpc19
sometimes_rpc2
sometimes_rpc20
sometimes_rpc21
sometimes_rpc22
sometimes_rpc23
sometimes_rpc24
sometimes_rpc25
sometimes_rpc26
sometimes_rpc27
sometimes_rpc28
sometimes_rpc3
sometimes_rpc4
sometimes_rpc5
sometimes_rpc6
sometimes_rpc7
sometimes_rpc8
sometimes_rpc9
sonar
sonardata
soniqsync
sonus
sonuscallsig
sonus_logging
s_openmail
sophia_lm
sophos
sops
sor_update
sos
sossd_agent
sossd_collect
sossd_disc
sossecollector
soundsvirtual
spamassassin
spamtrap
spandataport
spc
spcsdlobby
spdp
spdy
spearway
spectardata
spectardb
spectraport
speedtrace
spg
sphinxapi
sphinxql
spice
spike
spiral_admin
splitlock
splitlock_gw
spmp
spock
spocp
spramsca
spramsd
spremotetablet
spsc
spss
spss_lm
sps_tunnel
spt_automation
sptx
spugna
spw_dialer
spw_dnspreload
sqdr
sqlexec
sqlexec_ssl
sqlnet
sql_net
sqlserv
sqlsrv
squid_htcp
squid_http
squid_ipc
squid_snmp
src
srcp
srdp
srmp
srp_feedback
srssend
sruth
srvc_registry
ss7ns
ssad
sscan
ssdispatch
ssdtp
sse_app_config
ssh
sshell
ssh_mgmt
ss_idi
ss_idi_disc
sslp
ssmc
ssm_cssps
ssm_cvs
ssmd
ssm_els
ssmpp
sso_control
sso_service
ssowatch
ssp
ssp_client
ssports_bcast
ssql
ssrip
ssr_servermgr
ssserver
ssslic_mgr
ssslog_mgr
sst
sstats
sstp_1
sstp_2
sstsys_lm
stanag_5066
starbot
starfish
stargatealerts
starquiz_port
stars
starschool
start_network
startron
stat_cc
stat_results
stat_scanner
statsci1_lm
statsci2_lm
statsrv
statusd
stdptc
stel
ste_smsc
stgxfws
sti_envision
stmf
stm_pproc
stone_design_1
stonefalls
storageos
storman
storview
streamcomm_ds
streetperfect
street_stream
streettalk
streletz
stresstester
strexec_d
strexec_s
stryker_com
stss
stt
sttunnel
stun
stun_p1
stun_p2
stun_p3
stun_port
stuns
stvp
stx
sua
submission
submit
submitserver
subntbcst_tftp
subseven
sugp
suitcase
suitjd
sum
su_mit_tg
sun_answerbook
sun_as_iiops
sun_as_iiops_ca
sun_as_jmxrmi
sun_as_jpda
sun_as_nodeagt
suncacao_csa
suncacao_jmxmp
suncacao_rmi
suncacao_snmp
suncacao_websvc
sunclustergeo
sunclustermgr
sun_dr
sunfm_port
sun_lm
sunlps_http
sun_manageconsole
sun_mc_grp
sunproxyadmin
sunscalar_dns
sunscalar_svc
sun_sea_port
sun_sr_admin
sun_sr_http
sun_sr_https
sun_sr_iiop
sun_sr_iiop_aut
sun_sr_iiops
sun_sr_jms
sun_sr_jmx
sun_user_https
SunVTS_RMI
sunwebadmin
sunwebadmins
supdup
supercell
supermon
supfiledbg
supfilesrv
support
supportassist
surebox
surf
surfcontrolcpa
surfpass
sur_meas
surveyinst
suucp
svbackup
svcloud
svdrp
svn
svnet
svnetworks
svrloc
svs_omagent
swa_1
swa_2
swa_3
swa_4
swarm
swdtp
swdtp_sv
sweetware_apps
swgps
swiftnet
swift_rvf
swismgr1
swismgr2
swispol
swistrap
swldy_sias
sw_orion
swrmi
swr_port
swtp_port1
swtp_port2
swx
swxadmin
swx_gate
sxmp
sxuptp
syam_agent
syam_smc
syam_webserver
sybase
sybaseanywhere
sybasedbsynch
sybasesrvmon
sychrond
sygatefw
symantec_av
symantec_sfdb
symantec_sim
symb_sb_port
symplex
synapse
synapse_nhttp
synapse_nhttps
synapsis_edge
sync_em7
synchromesh
synchronet_db
synchronet_rtc
synchronet_upd
synchronite
syncserver
syncserverssl
synctest
synel_data
synoptics_trap
synotics_broker
synotics_relay
sype_transport
syscomlan
syserverremote
sysinfo_sp
syslog
syslog_conn
syslog_tls
sysopt
sysorb
sysrqd
sysscanner
systat
systemics_sox
system_monitor
t128_gateway
t1distproc
t1distproc60
t1_e1_over_ip
t2_brm
t2_drm
t5_straton
tabula
tacacs
tacacs_ds
tacnews
tacticalauth
taep_as_svc
tag_pm
tag_ups_1
taiclock
talarian_mcast1
talarian_mcast2
talarian_mcast3
talarian_mcast4
talarian_mcast5
talarian_mqs
talarian_udp
taligent_lm
talikaserver
talk
talnet
talon_disc
talon_engine
talon_webserver
tal_pod
tam
tambora
tams
tapestry
tapeware
tappi_boxnet
tarantella
targus_getdata
targus_getdata1
targus_getdata2
targus_getdata3
tarp
taserver
taskman_port
taskmaster2000
tasp_net
taurus_wh
tbrpf
tcc_http
tcim_control
tclprodebugger
tcoaddressbook
tcoflashagent
tcoregagent
tcpdataserver
tcp_id_port
tcpmux
tcpnethaspsrv
tdaccess
tdmoip
td_postman
tdp_suite
td_replica
td_service
teamcoherence
teamviewer
tec5_sdctp
teedtap
tekpls
telaconsole
telefinder
telelpathattack
telelpathstart
teleniumdaemon
telesis_licman
telindus
tell
tellumat_nms
telnet
telnetcpcd
telnets
telops_lmd
tempest_port
tempo
tenebris_nts
tenfold
tentacle
teradataordbms
teredo
terminaldb
tesla_sys_msg
tetrinet
texai
texar
tftp
tftp_mcast
tftps
tgcconnect
tgp
thermo_calc
theta_lm
thingkit
thrp
thrtx
tht_treasure
ticf_1
ticf_2
tick_port
tidp
tiepie
tig
tigv2
tile_ml
timbuktu
timbuktu_srv1
timbuktu_srv2
timbuktu_srv3
timbuktu_srv4
time
timed
timeflies
timelot
timestenbroker
tinc
tinyfw
tinymessage
tip2
tip_app_server
tipc
tivoconnect
tivoli_npm
tksocket
tl1
tl1_lv
tl1_raw
tl1_raw_ssl
tl1_ssh
tl1_telnet
tl_ipcproxy
tlisrv
tmesis_upshot
tmi
tmo_icon_sync
tmophl7mts
tmosms0
tmosms1
tnETOS
tnmpv2
tnos_dp
tnos_dps
tnos_sp
tnp
tnp1_port
tnp_discover
tns_adv
tns_cml
tns_server
tn_timing
tn_tl_fd1
tn_tl_fd2
tn_tl_r1
tn_tl_r2
tn_tl_w1
tn_tl_w2
toad
toad_bi_appsrvr
tolfab
tolteces
tomato_springs
tonidods
topflow
topflow_ssl
topovista_data
topx
tor_control
tor_orport
tor_socks
tor_trans
toruxserver
touchnetplus
tpcsrvr
tpdu
tpip
tpmd
tqdata
traceroute
track
tragic
traingpsdata
tram
transact
transmit_port
trap
trap_daemon
trap_port
trap_port_mom
traversal
travsoft_ipx_t
trc_netpoll
trdp_md
trdp_pd
treehopper
trendchip_dcp
tributary
trident_data
trim
trim_event
trim_ice
trinity_dist
Trinoo_Bcast
Trinoo_Master
Trinoo_Register
triomotion
trip
tripe
tripwire
triquest_lm
trisoap
trispen_sra
tritium_can
trivnet1
trivnet2
trnsprntproxy
troff
trp
tr_rsrb_p1
tr_rsrb_p2
tr_rsrb_p3
tr_rsrb_port
truckstar
truecm
trusted_web
trustestablish
tsa
tsaf
tsb
tsb2
tscchat
tsdos390
tsepisp
tserver
tsilb
tsp
tsrmagt
tsspmap
ttat3lb
ttc
ttc_etap
ttc_etap_ds
ttc_etap_ns
ttcmremotectrl
ttc_ssl
ttg_protocol
ttlpriceproxy
ttl_publisher
ttnrepository
ttntspauto
ttyinfo
tunalyzer
tunatic
tungsten_http
tungsten_https
tunnel
tunstall_lwp
tunstall_pnc
turbonote_1
turbonote_2
tvbus
tvdumtray_port
tve_announce
tvnetworkvideo
tvpm
twamp_control
tw_auth_key
twcss
twds
twrpc
twsdss
uaac
uacs
uadtc
uaiact
uarps
ua_secureagent
ub_dns_control
ubroker
ubxd
ucentric_ds
ucontrol
ucs_isc
u_dbap
udpradio
udp_sr_port
udrawgraph
udrive
udt_os
uec
ufastro_instr
ufmp
ufsd
uis
ulistserv
ulp
ulpnet
ultimad
ultrabac
ultraseek_http
ultrex
uma
umeter
umm_port
ums
umsp
unbind_cluster
undo_lm
unet
unglue
unicall
unicontrol
unidata_ldm
unieng
unify
unify_adapter
unifyadmin
unify_debug
unihub_server
unikeypro
unimobilectrl
uniport
unisql
unisql_java
unisys_eportal
unisys_lm
unitary
unite
univ_appserver
universe_suite
univision
unix_status
unizensus
unknown
unot
uohost
uorb
updog
upgrade
upnotifyp
upnotifyps
upnp
ups
ups_engine
upsnotifyprot
ups_onlinet
upstriggervsw
urbisnet
urld_port
urm
us_cli
user_manager
us_gv
usicontentpush
us_srv
utcd
utime
utmpcd
utmpsd
utsftp
uucp
uucp_path
uucp_rlogin
uuidgen
v2g_secc
v5ua
vacdsm_app
vacdsm_sws
vad
valisys_lm
vantronix_mgmt
va_pacbase
vaprtm
varadero_0
varadero_1
varadero_2
vat
vatata
vat_control
vatp
vaultbase
vce
vchat
vcmp
vcnet_link_v10
vcom_tunnel
vcrp
vcs_app
vcscmd
vdab
vdmmesh
vdmplay
vemmi
ventoso
venus
venus_se
veracity
vergencecm
verismart
veritas_pbx
veritas_tcp1
veritas_ucl
veritas_udp1
veritas_vis1
veritas_vis2
veronica
versatalk
versa_tek
vestasdlp
vettcp
vfbp
vfmobile
vfo
vhd
via_ftp
vid
video_activmail
videobeans
videotex
videte_cipc
vidigo
vids_avtp
vieo_fe
vinainstall
vipera
vipera_ssl
vipremoteagent
virprot_lm
virtual_places
virtualtape
virtual_time
virtualuser
visd
visicron_vs
visinet_gui
vision_elmd
visionpyramid
vision_server
visitview
vista_4gl
vistium_share
visweather
vitalanalysis
viziblebrowser
vlsi_lm
vmnet
vmodem
vmpwscs
vmrdp
vmsvc
vmsvc_2
vmware_fdm
vnas
vnc
vnc_1
vnc_2
vnc_3
vnc_http
vnc_http_1
vnc_http_2
vnc_http_3
vnetd
vnsstr
vns_tp
vnwk_prapi
vnyx
vocaltec_admin
vocaltec_gold
vocaltec_hos
vocaltec_phone
vocaltec_wconf
vofr_gateway
voispeed_port
volley
v_one_spp
vop
vopied
voxelstorm
vp2p
vpa
vpac
vpad
vpa_disc
vpjp
vpm_udp
vpnz
vpp
vpps_qua
vpps_via
vpsipport
vpvc
vpvd
vqp
vrace
vrcommerce
vrml_multi_use
vrpn
vrt
vrtl_vmf_ds
vrtl_vmf_sa
vrtp
vrts_at_port
vrts_auth_port
vrts_ipcserver
vrts_registry
vrts_tdd
vrtstrapserver
vrxpservman
vsaiport
vsamredirector
vsat_control
vscp
vseconnector
vsiadmin
vsinet
vsi_omega
vsixml
vslmp
vsnm_agent
vspread
vs_server
vstat
vtp
vtr_emulator
vtsas
vts_rpc
vt_ssl
vtu_comms
vulture
vunknown
vvr_control
vvr_data
vx_auth_port
vxcrnbuport
vxlan
vxlan_gpe
vytalvaultbrtp
vytalvaultpipe
vytalvaultvsmp
waascluster
wacp
wafs
wago_io_system
wago_service
wag_service
wanscaler
wap_push
wap_push_http
wap_push_https
wap_pushsecure
wap_vcal
wap_vcal_s
wap_vcard
wap_vcard_s
wap_wsp
wap_wsp_s
wap_wsp_wtp
wap_wsp_wtp_s
warehouse
warehouse_sss
warmspotMgmt
waste
watchdoc
watchdoc_pod
watchdog_nt
watchme_7272
watcomdebug
watcom_sql
watershed_lm
watilapp
wbem_exp_https
wbem_http
wbem_https
wbem_rmi
wcbackup
wcpp
wcr_remlib
wdbrpc
weandsf
weave
web2host
webaccess
webdata
webemshttp
weblogin
webmachine
webmail_2
webmethods_b2b
webobjects
webphone
webpush
websm
websphere_snmp
webster
webtie
webyast
wello
westec_connect
westell_stats
wfc
wfm
wfremotertm
wg_netforce
wherehoo
whisker
who
whoami
whois
whosells
whosockami
WibuKey
wiegand
wifree
wilkenlistener
willy
wimaxasncp
wimd
wimsic
wincim
windb
winddlb
winddx
windlm
windows_icfw
winfs
wininstall_ipc
winjaserver
winpcs
winpharaoh
winpoplanmess
winport
winqedit
winrm
win_rpc
wins
winshadow
winshadow_hd
wip_port
wired
wizard
wkars
wkstn_mon
wlanauth
wlbs
wlcp
wmc_log_svc
wmedistribution
wmereceiving
wmereporting
wmlserver
wms
wms_messenger
wnn6
wnn6_Cn
wnn6_ds
wnn6_DS
wnn6_Kr
wnn6_Tw
workflow
workflowdir
work_sol
worldfusion1
worldfusion2
world_lm
worldscores
worldwire
wormux
wpages
wpgs
wpl_analytics
writesrv
wrspice
wrs_registry
wsdapi
wsdapi_s
ws_discovery
wsdl_event
wsicopy
wsman
wsmans
wsmlb
wsm_server
wsm_server_ssl
wso2esb_console
wspipe
wssauthsvc
wsscomfrmwk
wsynch
wta_wsp_s
wta_wsp_wtp_s
wusage
wv_csp_sms
wv_csp_sms_cir
wv_csp_udp_cir
wwiotalk
www_dev
www_ldap_gw
wxbrief
wysdma
wysdmc
x11
X11
X11_1
X11_2
X11_3
X11_4
X11_5
X11_59
X11_6
X11_7
X11_8
X11_9
x25_svc_port
x2_control
x2e_disc
x500ms
x9_icue
xaapi
xact_backup
xadmin
xandros_cms
xap_ha
xaudio
x_bone_api
x_bone_ctl
xbox
xdas
xdmcp
xds
xdsxdm
xdtp
xecp_node
xfer
xfr
xgrid
xic
xiip
xingcsm
xingmpeg
xinuexpansion1
xinuexpansion2
xinuexpansion3
xinuexpansion4
xinupageserver
xiostatus
xkotodrcp
xlog
xmail_ctrl
xmapi
xmcp
xmlblaster
xmlink_connect
XmlIpcRegSvc
xmlrpc_beep
xmltec_xmlmail
xmms2
xmp
xmpcr_interface
xmpp
xmpp_bosh
xmpp_client
xmpp_server
xmpv7
xmquery
xmsg
xn_control
xnds
xnm_clear_text
xnmp
xnm_ssl
xns_auth
xns_ch
xns_courier
xns_mail
xns_time
xoms
xoraya
xo_wave
xpanel
xpilot
xpl
xpra
xprint_server
xprtld
xqosd
xribs
xrl
xrpc_registry
xserveraid
xsmsvc
xs_openstorage
xss_port
xss_srv_port
xsync
xtell
xtgui
xtlserv
xtreamx
xtreelic
xtrm
xtrms
xvttp
xw_control
xxnetserver
xybrid_cloud
xybrid_rt
xyplex_mux
yak_chat
yawn
yo_main
z39_50
zabbix_agent
zabbix_trapper
zannet
zarkov
zebra
zebrasrv
zenginkyo_1
zenginkyo_2
zen_pawn
zented
zep
zephyr_clt
zephyr_hm
zephyr_srv
zeroconf
zeus_admin
zfirm_shiprush3
zicom
zieto_sock
zigbee_ip
zigbee_ips
zincite_a
zion_lm
zixi_transport
zmp
zoomcp
zsecure
zserv
z_wave
z_wave_s
z_wave_tunnel
zymed_zpp
