/*                                                                
**  Copyright (C) 1996,2007,2010,2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/* iraf.c
**
	Routines to manipulate iraf parameter files.
**/

#ifdef __linux__
#undef _XOPEN_SOURCE
#define _XOPEN_SOURCE 600
/* stdio.h: fileno, ftruncate */
#define _POSIX_C_SOURCE 2
#endif

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>

#include "pfile.h"
#include "ptemplat.h"
#include "parameter.h"
#include "pargs.h"
#include "putil.h"
#include "ipar.h"
#include "iraf.h"
#include "parament.h"
#include "tmatch.h"

static ParamList *IRAFParRead (FILE * file, char *mode);
static void IRAFParWrit (FILE * file, ParamList * plist);
static void IRAFParW (Parameter * p, FILE * f);



/*
	The mode argument:

		"l"	lock file for exclusive access.
*/

ParamList *
IRAFParOpen (const char *psetname, const char *filename, const char *mode)
{
  FILE *f;
  ParamList *plist;
  char *fmode = strpbrk (mode, "w") == NULL ? "r" : "r+";

  if (filename == NULL)
    return NULL;

  if ((f = fopen (filename, fmode)) == NULL)
    {
      fprintf (stderr, "Open error on param file: %s\n", filename);
      return NULL;
    }

  if (strpbrk (mode, "l"))
    Lock (f);

  if ((plist = IRAFParFile (f)) == NULL)
    {
      return (ParamList *) ParamFileCant;
    }

  plist->file = f;
  plist->filename = CXCNewString (filename);
  plist->psetname = CXCNewString (psetname);
  plist->mode = CXCNewString (mode);
  plist->read = IRAFParRead;
  plist->writ = IRAFParWrit;
  plist->clos = NULL;

  return plist;
}


ParamList *
IRAFStrOpen (const char *psetname, char *str)
{
  ParamList *plist;

  if ((plist = IRAFParseStr (str)) == NULL)
    {
      return (ParamList *) ParamFileCant;
    }



  plist->file = NULL;

  if (psetname)
    {
      char *here;

      if ((here = strstr (psetname, ".par")) && *here == '\0')
	{
	  plist->filename = CXCNewString (psetname);
	}
      else
	{
	  plist->filename = malloc (strlen (psetname) + 5);
	  strcpy (plist->filename, psetname);
	  strcat (plist->filename, ".par");
	}
      plist->psetname = CXCNewString (psetname);
    }
  else
    {
      plist->psetname = NULL;
      plist->filename = NULL;
    }

  plist->read = IRAFParRead;
  plist->writ = IRAFParWrit;
  plist->clos = NULL;

  return plist;
}


static ParamList *
IRAFParRead (FILE * file, char *mode)
{
  if (strpbrk (mode, "l"))
    Lock (file);
  fseek (file, 0, 0);

  return IRAFParFile (file);
}


static void
IRAFParWrit (FILE * file, ParamList * plist)
{
  int i;

  if (fseek (file, 0, 0))
    perror ("fseek");
  for (i = 0; i < plist->nparam; i++)
    IRAFParW (plist->param[i], file);

  fflush (file);
  ftruncate (fileno (file), ftello (file));
}


static void
IRAFParW (Parameter * p, FILE * f)
{
  ParamFile PFFFile;
  char namestr[SZ_PFLINE];
  char typestr[SZ_PFLINE];
  char modestr[SZ_PFLINE];
  char valustr[SZ_PFLINE];
  char minnstr[SZ_PFLINE];
  char maxxstr[SZ_PFLINE];
  char promstr[SZ_PFLINE];
  char strustr[SZ_PFLINE];
  char quote[4];
  int sent;
  int use_default = 1;		/* output default or updated value */
  Value tval;

  /* This fixes bug #14209 where the buffer passed in pvalue.value is larger
     than the receipient buffer which is SZ_PFLINE. at the VOnvert function
     the data is copied and caused a SegV
  
     SL-98: Added check for FileType */
  if(p)
  {
    if((p->pvalue.value != NULL)&&((TypeIndex (p->pvalue.type) == StringType)
                                || (TypeIndex (p->pvalue.type) == FileType)))
    {
      if(strlen(p->pvalue.value) > SZ_PFLINE-1){
        char *r = (char *)( p->pvalue.value);
        r[SZ_PFLINE-1] = '\0';
      }
    }  
  }

  ParamEntry (NULL, p, namestr, modestr, typestr, valustr, minnstr, maxxstr,
	      promstr);

  if (p->ptype == CommentType)
    {
      fprintf (f, "%s\n", valustr);
      return;
    }

  /* write the updated value if the parameter file was opened in
   ** LLMODE ('L' specified in paramopen call) or if the param is
   ** learned mode OR auto mode w/ mode parameter in learned mode. 
   ** Otherwise write the default value back to the parameter file.
   **/
  if ((PFFFile = ParamFileLookup ("\0")) != NULL)
    use_default = ((PFFFile->mode & LLMODE) != LLMODE);
  if (use_default)
    if ((use_default = ((p->pmode & LMODE) != LMODE)) &&
	((p->pmode & AMODE) == AMODE))
      {
	if (PFFFile && ((ParamFileMode (PFFFile) & LMODE) == LMODE))
	  use_default = 0;
      }

  if (use_default)
    strcpy (valustr, p->pdefault);

  if (p->ptype == StructType)
    {
      strcpy (strustr, valustr);
      strcpy (valustr, "160");
    }

  ESCQuote (valustr);
  ESCQuote (minnstr);
  ESCQuote (maxxstr);
  ESCQuote (promstr);

  /* enclose strings, indirects, and command evals in quotes */
  tval.type = StringType;
  tval.value = valustr;
  tval.indir = NULL;
  if ((p->ptype == StringType) || (p->ptype == FileType)
      || VIsIndirect (&tval))
    strcpy (quote, "\"");
  else
    quote[0] = '\0';

  sent =
    fprintf (f, "%s,%s,%s,%s%s%s,%s,%s,%s%s%s\n", namestr, typestr, modestr,
	     quote, valustr, quote, minnstr, maxxstr, promstr[0] ? "\"" : "",
	     promstr, promstr[0] ? "\"" : "");

  if (sent == -1)
    {
      char tbuf[SZ_PFLINE*2];
      sprintf (tbuf, "could not write param '%s'", namestr);
      perror (tbuf);
    }

  if (p->ptype == StructType)
    {
      ESCQuote (strustr);
      fprintf (f, "%s\n", strustr);
    }
}


Parameter *
ParamPositional (ParamList * plist, int position, char *value)
{
  Parameter **p = plist->param;
  Parameter *param;
  int i, j;
  char *name;

  p = plist->param;

  for (j = plist->nparam, i = position; j; j--, p++)
    {
      if ((*p)->ptype == CommentType)
	continue;
      if (((*p)->pmode & HMODE) == 0)
	i--;
      if (!i)
	break;
    }

  if (i && !j)
    {
      parerr = PARTOOMANYPOS;
      return NULL;
    }

  name = CXCNewString ((*p)->pname);

  param = ParamName (NULL, name);
  param = ParamType (param, (*p)->ptype);
  param = ParamValue (param, StringType, value);

  return param;
}


Parameter *
ParamCreate (char *name, VType type, int *mode, char *value,
	     Parameter * mmptp)
{
  Parameter *param;
  char *hack;

  if (name == NULL)
    {
      if (value)
	free (value);
      if (mmptp)
	ParamFree (mmptp);

      return NULL;
    }

  if (type == 0)
    {
      paramerr (0, "iraf parser", name);

      if (name)
	free (name);
      if (value)
	free (value);
      if (mmptp)
	ParamFree (mmptp);

      return NULL;
    }

  if (mode == NULL)
    {
      paramerr (0, "iraf parser", name);

      if (name)
	free (name);
      if (value)
	free (value);
      if (mmptp)
	ParamFree (mmptp);

      return NULL;
    }

  param = ParamName (mmptp, name);
  param = ParamType (param, type);
  param = ParamMode (param, mode);


  ESCUnQuote (value);

  /* This mess is to work-around the case where the value
     string is "".  In that case, ParamFree() in pfile.c
     winds up freeing a bad place in memory when parameter
     clean-up takes place (at the end of program execution).
     Here, we malloc a piece of memory for ParamFree() to free. 
   */
  if (value && (*value == 0))
    {
      hack = (char *) malloc (1);
      *hack = 0;
      free (value);
    }
  else
    hack = value;

  param = ParamValue (param, StringType, hack);
  param = ParamDefault (param);	/* store original value */
  return param;
}


VType
PLookupType (ParamList * reference, char *name)
{
  Parameter *p;

  if (reference)
    {
      p = ParamLookup (reference, name);

      if (p == NULL)
	{
	  fprintf (stderr, "parameter not found : %s \n", name);
	  return (VType) 0;
	}

      return p->ptype;
    }
  else
    return StringType;
}


void
ESCQuote (char *str)
{
  char *t, *k, *s = str;

  int len = strlen (str) + 1;

  if ((str == NULL) || (*str == '\0'))
    return;

  k = t = (char *) malloc (len + 100);
  *t = '\0';

  while (*str)
    {
      if (*str == '"')
	{
	  *t++ = '\'';
	  str++;
	}
      *t++ = *str++;
    }
  *t = '\0';

  strcpy (s, k);
  free (k);

  return;
}

char *
ESCUnQuote (char *str)
{
  char *t, *k, *s = str;

  int len;

  if ((str == NULL) || (*str == '\0'))
    return str;

  len = strlen (str) + 1;

  k = t = (char *) malloc (len + 100);
  *t = '\0';

  while (*str)
    {
      if (*str == '\\' && *(str + 1) == '"')
	{
	  str++;
	}
      *t++ = *str++;
    }
  *t = '\0';

  strcpy (s, k);
  free (k);

  return s;
}

/* Rip this stuff from /proj/asc/xiraf/irafhack.c
*/

#ifndef vax

#if NeXT
#include <sys/types.h>
typedef void *DIR;
#include <sys/dirent.h>
#else
#include <dirent.h>
#endif

#define SZ_LINE 	SZ_PFLINE
#define SZ_FNAME 	SZ_PFLINE
#define PARAMFILE_EXT ".par"
#define     LEN_PFILENAME   6
#define     LEN_PKPREFIX    3

static char templatebuf[SZ_LINE];


/*
 * locatepfile - look in likely directories for a param file
 *
 * We look in the following directories:
 *
 *	the current directory
 *	the directory pointed to by UPARM environment variable
 *
*/

char *
locatepfile (const char *fname)
{
  char *e;
  char namebuff[SZ_PFLINE];

  strcpy (namebuff, fname);
  e = strstr (namebuff, ".par");

  /* If the loser provides a .par extension we knock it off */
  if (e && (e[strlen (".par")] == '\0'))
    *e = '\0';

  return locateptemplate (namebuff, getenv ("UPARM"));
}


/*
 *  locateptemplate
 *   -- look at templates of IRAF task param files in a directory
 */
char *
locateptemplate (char *fname, char *dname)
{
  DIR *dirp;
  struct dirent *direntp;
  char template[132];
  char *dot;
  int got = 0;

  if (dname == NULL)
    return NULL;

  /* open the directory */
  if ((dirp = opendir (dname)) == NULL)
    return (NULL);

  /* look for the case where package.task was specified as a param */
  if ((dot = strchr (fname, '.')) != NULL)
    {
      *dot++ = '\0';
      mkpfilename (template, "", fname, dot, ".par");
    }
  /* the template uses matches against any package (the "???" part) */
  else
    mkpfilename (template, "", "???", fname, ".par");

  /* look for files that match the template */
  while ((direntp = readdir (dirp)) != NULL)
    {
      if (cxctmatch (direntp->d_name, template))
	{
	  got++;
	  if (got == 1)
	    /* return the first one */
	    sprintf (templatebuf, "%s%s", dname, direntp->d_name);
	  else
	    {
	      if (got == 2)
		{
		  fprintf (stderr,
			   "warning: multiple matches of %s in dir %s\n",
			   fname, dname);
		  fprintf (stderr, "using file %s\n", templatebuf);
		}
	      fprintf (stderr, "ignoring file %s\n", direntp->d_name);
	    }
	}
    }

  /* close the directory */
  closedir (dirp);

  /* didn't find parameter file */
  if (got)
    return (templatebuf);
  else
    return (NULL);
}

/* MAPNAME -- Apply the N+1 mapping convention (first N-1 plus last chars)
 * to generate a name no longer than N characters.  Returns the number of
 * characters generated.
 */
static void
mapname (char *in, char *out, int maxlen)
{
  register int ip, op;

  ip = 0;
  op = 0;
  while (op < maxlen - 1 && (out[op++] = in[ip++]) != '\0')
    ;
  if (out[op - 1] != '\0')
    {				/* append last char     */
      if (in[ip] != '\0')
	{
	  while (in[ip] != '\0')
	    ip++;
	  out[op++] = in[ip - 1];
	}
      out[op++] = '\0';
    }
}


/* MKPFILENAME -- Generate a parameter file name, given a directory prefix
 * the names of the package and ltask, and the filename extension.  The form
 * of the filename depends upon whether the pfile is to be stored in UPARM.
 * UPARM pfile names have the form "uparm$ // pakltask.par", where `pak' is
 * the package prefix, consisting of the first LEN_PKPREFIX-1 characters of
 * the package name plus the final character, and `ltask' is the ltask name
 * squeezed to LEN_PFILENAME characters.  If not writing to UPARM, we just
 * use the full filename.
 */
void
mkpfilename (char *buf,		/* receives output filename     */
	     char *dir,		/* dir name or prefix           */
	     char *pkname,	/* package name                 */
	     char *ltname,	/* ltask name                   */
	     char *extn		/* filename extension           */
  )
{
  char temp[SZ_FNAME + 1];
  int len;

  strcpy (buf, dir);		/* start with directory name    */

  /* add final "/", if necessary */
  len = strlen (buf);
  if (len)
    {
      if (buf[len - 1] != '/')
	strcat (buf, "/");
    }
  mapname (pkname, temp, LEN_PKPREFIX);
  strcat (buf, temp);
  mapname (ltname, temp, LEN_PFILENAME);
  strcat (buf, temp);
  strcat (buf, extn);		/* add extension for pfile      */
}

#endif /* vax */
