#!/usr/bin/perl

# $Id: clean_results.pl,v 1.2 2002/02/28 03:32:54 lachoy Exp $

use strict;
use lib qw( .. );
use Getopt::Long;
use OpenInteract::ResultsManage;
use OpenInteract::Startup;

use constant DEBUG                => 0;
use constant DEFAULT_REMOVAL_TIME => 60 * 30; # 30 minutes

{
    my ( $OPT_minutes );
    my %opts = ( 'minutes=s' => \$OPT_minutes );
    my $usage = "Parameter '--website_dir' or environment variable 'OIWEBSITE' must be set " .
                "to existing website directory. Cannot remove results.";
    my $R = OpenInteract::Startup->setup_static_environment_options(
                    $usage, \%opts, { temp_lib => 'lazy' } );
    my $removal_time = ( $OPT_minutes )
                         ? $OPT_minutes * 60 : DEFAULT_REMOVAL_TIME;

    require OpenInteract::ResultsManage;
    my $results = OpenInteract::ResultsManage->new();
    my $results_files = $results->get_all_result_filenames();

    my $now = time;
    foreach my $search_id ( @{ $results_files } ) {
        DEBUG() && warn "Try search ID [$search_id]\n";
        my $meta_info = $results->get_meta( $search_id );
        next unless ( ref $meta_info eq 'HASH' );
        if ( $now - $meta_info->{time} > $removal_time ) {
            $results->results_clear( $search_id );
            DEBUG() && warn "-- Removed result [$search_id] which was originally ",
                            "searched ", scalar localtime( $meta_info->{time} ), "\n";
        }
    }
    DEBUG() && warn "Cleanup of results complete\n";
}

__END__

=pod

=head1 NAME

clean_results.pl - Script to cleanup the results directory of stale results

=head1 SYNOPSIS

 # From the command line

 # Use default 30 minute threshold
 $ perl clean_results.pl --website_dir=/path/to/mysite

 # Use 45 minute threshold
 $ perl clean_results.pl --website_dir=/path/to/mysite --minutes=45

 # From a cron job - run every hour at 45 minutes past.
 45 * * * * perl /path/to/mysite/script/clean_results.pl --website_dir=/path/to/mysite

=head1 DESCRIPTION

Simple script -- just scan the entries in the results directory and
get rid of the ones older than x (default: 30) minutes.

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

L<OpenInteract::ResultsManage|OpenInteract::ResultsManage>

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
