=head1 Name

Math::Cartesian::Product - Generate the cartesian product of zero or more lists.

=head1 Synopsis

 use Math::Cartesian::Product;

 cartesian {say "@_"} [qw(a b c)], [1..2];

 #  a 1
 #  a 2
 #  b 1
 #  b 2
 #  c 1
 #  c 2

=cut

use strict;

package Math::Cartesian::Product;
use Carp;

sub cartesian(&@)
 {my $s = shift;       # Subroutine to call to process each element of the product

  my @C = @_;          # Lists to be multiplied
  my @c = ();          # Current element of cartesian product
  my @P = ();          # Cartesian product
  my $n = 0;           # Number of elements in product

  return 0 if @C == 0; # Empty product

  @C == grep {ref eq 'ARRAY'} @C or croak("Array of arrays requierd by cartesian");

  my $p; $p = sub      # Generate each cartesian product
   {if (@c < @C) 
     {for(@{$C[@c]})
       {push @c, $_;
        &$p();
        pop @c;
       }
     }
    else 
     {my $p = [map {ref eq __PACKAGE__ ? @$_ : $_} @c];
      push @P, bless $p if &$s(@$p);
     }
   };

  &$p;

  @P;
 }

# Export details
 
require 5;
require Exporter;

use vars qw(@ISA @EXPORT $VERSION);

@ISA     = qw(Exporter);
@EXPORT  = qw(cartesian);
$VERSION = '1.001';

=head1 Description

Generate the cartesian product of zero or more lists.

Given two lists, say: [a,b] and [1,2,3], the cartesian product is the
set of all possible ordered pairs:

 (a,1), (a,2), (a,3), (b,1), (b,2), (b,3)

which select their first element from all the possibilities listed in
the first list, and select their second element from all the
possibilities in the second list.

The idea can be generalized to any number of lists. In particular, the
cartesian product of zero lists is the empty set, as is the cartesian
product of any set of lists which contain a list with no elements.

C<cartesian()> takes the following parameters:

1. A block of code to process each selection of list elements.

2. zero or more lists.

C<cartesian()> returns an array of references to all the pairs (more
generally n-tuples) produced, each pair being held as an array.

C<cartesian()> croaks if you try to form the cartesian product of
something other than a list of things.

The cartesian product of lists A,B,C is associative, that is: 

  (A X B) X C = A X (B X C)

C<cartesian()> respects associativity by allowing you to include a
cartesian product produced by an earlier call to C<cartesian()> in the
set of lists whose cartesian product is to be formed.

  my $a = [qw(a b)];
  my $b = [cartesian {} $a, $a;
  ok 16 == cartesian {} $b, $b;

  # a a a a
  # a a a b       
  # a a b a 
  # ...
 

C<cartesian()> is easy to use and fast. It is written in 100% Pure Perl.


=head1 Export

The C<cartesian()> function is exported.

=head1 Installation

Standard Module::Build process for building and installing modules:

  perl Build.PL
  ./Build
  ./Build test
  ./Build install

Or, if you're on a platform (like DOS or Windows) that doesn't require
the "./" notation, you can do this:

  perl Build.PL
  Build
  Build test
  Build install

=head1 Author

PhilipRBrenan@handybackup.com

http://www.handybackup.com

=head1 See Also

=head1 Copyright

Copyright (c) 2009 Philip R Brenan.

This module is free software. It may be used, redistributed and/or
modified under the same terms as Perl itself.

=cut
