package Test::Unit::Assert;
use strict;
use constant DEBUG => 0;

sub assert {
    my $self = shift;
    print ref($self) . "::assert() called\n" if DEBUG;
    my ($condition, $message) = @_;
    $self->fail($message) unless $condition;
}

sub fail {
    my $self = shift;
    print ref($self) . "::fail() called\n" if DEBUG;
    my ($message) = @_;
    my $ex = Test::Unit::ExceptionFailure->new($message);
    $ex->hide_backtrace() unless $self->get_backtrace_on_fail();
    die $ex;
}

sub quell_backtrace {
    my $self = shift;
    $self->{_no_backtrace_on_fail} = 1;
}

sub get_backtrace_on_fail {
    my $self = shift;
    return $self->{_no_backtrace_on_fail} ? 0 : 1;
}



1;
__END__

=head1 NAME

    Test::Unit::Assert - unit testing framework assertion class

=head1 SYNOPSIS

    # this class is not intended to be used directly, 
    # normally you get the functionality by subclassing from 
    # Test::Unit::TestCase

    use Test::Unit::TestCase;
   
    # more code here ...
    
    $self->assert($your_condition_here, $your_optional_message_here);

    # NOTE: if you want to use regexes in comparisons, do it like this:

    $self->assert(scalar("foo" =~ /bar/), $your_optional_message_here);

=head1 DESCRIPTION

    This class is used by the framework to assert boolean conditions
    that determine the result of a given test. The optional message
    will be displayed if the condition fails. Normally, it is not
    used directly, but you get the functionality by subclassing from 
    Test::Unit::TestCase.

    There is one problem with assert(): the arguments to assert() are
    evaluated in list context, e.g. making a failing regex "pull" the
    message into the place of the first argument. Since this is ususally
    just plain wrong, please use scalar() to force the regex comparison
    to yield a useful boolean value. I currently do not see a way around
    this, since function prototypes don't work for object methods, and
    any other tricks (counting argument number, and complaining if there
    is only one argument and it looks like a string, etc.) don't appeal
    to me. Thanks to Matthew Astley for noting this effect. 

    The procedural interface to this framework, Test::Unit, does not have
    this problem, as it exports a "normal" assert() function, and that can
    and does use a function prototype to correct the problem.


=head1 AUTHOR

    Copyright (c) 2000 Christian Lemburg, <lemburg@acm.org>.

    All rights reserved. This program is free software; you can
    redistribute it and/or modify it under the same terms as
    Perl itself.

    Thanks go to the other PerlUnit framework people: 
    Brian Ewins, Cayte Lindner, J.E. Fritz, Zhon Johansen.

    Thanks for patches go to:
    Matthew Astley, David Esposito.

=head1 SEE ALSO

    - Test::Unit::TestCase
    - Test::Unit::Exception
    - The framework self-testing suite (Test::Unit::tests::AllTests)

=cut
