/* Copyright (C) 2001-2020 Artifex Software, Inc.
   All Rights Reserved.

   This software is provided AS-IS with no warranty, either express or
   implied.

   This software is distributed under license and may not be copied,
   modified or distributed except as expressly authorized under the terms
   of the license contained in the file LICENSE in this distribution.

   Refer to licensing information at http://www.artifex.com or contact
   Artifex Software, Inc.,  1305 Grant Avenue - Suite 200, Novato,
   CA 94945, U.S.A., +1(415)492-9861, for further information.
*/


/* plsymbol.c */
/* Built-in PCL symbol sets */

#include "stdpre.h"
#include "std.h"
#include "gdebug.h"
#include "plsymbol.h"
#include "plvocab.h"
#include "plvalue.h"

/* Define the MSL code for unmapped characters. */
#define ffff 0xffff

/* Define a macro for generating the header for built-in symbol sets. */
#define map_number(n, l) ((n) * 32 + (l) - '@')
#define map_header(n, l, vocab, t, cc0, cc1, cc2, cc3, cc4, cc5, cc6, cc7, mapping_type) \
    {0, 18}, {map_number(n, l) >> 8, map_number(n, l) & 0xff},          \
        (vocab) * 2 + 1, t, {0, 0}, {0, 255},                           \
  {cc0, cc1, cc2, cc3, cc4, cc5, cc6, ((cc7) & 0xf8) | (vocab)},        \
        (mapping_type)

/*
 * This file may exist in many different source formats over time.
 * To help convert between formats, if this file is compiled with the
 * -dMAIN switch, it will include a 'main' program that will write out
 * the data in directly usable format.
 */
#ifdef MAIN
#include <stdio.h>
#undef printf
int
main(int argc, char *argv[])
{
    const pl_symbol_map_t **ppmap = pl_built_in_symbol_maps;

    for (; *ppmap; ++ppmap) {
        const pl_symbol_map_t *pmap = *ppmap;
        uint id = (pmap->id[0] << 8) + pmap->id[1];
        int n = id >> 5;
        int l = (id & 31) + '@';
        uint i;

        printf("\n/**** %d%c ****/", n, l);
        for (i = 0; i < 256; ++i) {
            if (!(i & 7)) {
                if (!(i & 15))
                    printf("\n/* 0x%02x */\n   ", i);
                else
                    fputs("\n   ", stdout);
            }
            printf(" 0x%04x", (uint) pmap->codes[i]);
            if (i != 255)
                putchar(',');
        }
        puts("\n");
    }
    return 0;
}
#endif /* MAIN */

/* Define common symbol set elements. */

/***************************************
 * ASCII and national variant mappings *
 ***************************************/
/* ISO ASCII */
static const pl_symbol_map_t map_0U_unicode = {
    map_header(0, 'U', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /* 0x60 */
     0x2018, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};
/* United Kingdom */
static const pl_symbol_map_t map_1E_unicode = {
    map_header(1, 'E', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x00a3, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x00af, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};
/* French */
static const pl_symbol_map_t map_1F_unicode = {
    map_header(1, 'F', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x00a3, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x00e0, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x00b0, 0x00e7, 0x00a7, 0x005e, 0x005f,
     /* 0x60 */
     0x00b5, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x00e9, 0x00f9, 0x00e8, 0x00a8, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};
/* German */
static const pl_symbol_map_t map_1G_unicode = {
    map_header(1, 'G', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x00a7, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x00c4, 0x00d6, 0x00dc, 0x005e, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x00e4, 0x00f6, 0x00fc, 0x00df, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};

/* Italian */
static const pl_symbol_map_t map_0I_unicode = {
    map_header(0, 'I', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x00a3, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x00a7, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x00b0, 0x00e7, 0x00e9, 0x02c6, 0x005f,
     /* 0x60 */
     0x00f9, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x00e0, 0x00f2, 0x00e8, 0x00ec, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};

/* Swedish for names */
static const pl_symbol_map_t map_0S_unicode = {
    map_header(0, 'S', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x00a4, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x00c9, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x00c4, 0x00d6, 0x00c5, 0x00dc, 0x005f,
     /* 0x60 */
     0x00e9, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x00e4, 0x00f6, 0x00e5, 0x00fc, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};

/* PC-865 Norway */
static const pl_symbol_map_t map_25U_unicode = {
    map_header(20, 'U', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
        /* 0x00 */
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        /* 0x10 */
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        /* 0x20 */
        0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
        0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
        /* 0x30 */
        0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
        0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
        /* 0x40 */
        0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
        0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
        /* 0x50 */
        0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
        0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
        /* 0x60 */
        0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
        0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
        /* 0x70 */
        0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
        0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0xffff,
        /* 0x80 */
        0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
        0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
        /* 0x90 */
        0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
        0x00ff, 0x00d6, 0x00dc, 0x00f8, 0x00a3, 0x00d8, 0x20a7, 0x0192,
        /* 0xa0 */
        0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
        0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00a4,
        /* 0xb0 */
        0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
        0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
        /* 0xc0 */
        0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
        0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
        /* 0xd0 */
        0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
        0x256a, 0x2518, 0x250c, 0xffff, 0x2584, 0x258c, 0x2590, 0x2580,
        /* 0xe0 */
        0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4,
        0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x03c6, 0x03b5, 0xffff,
        /* 0xf0 */
        0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248,
        0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2, 0x25a0, 0x00a0
    }
};

/* PC-860 Portugal */
static const pl_symbol_map_t map_20U_unicode = {
    map_header(20, 'U', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
        /* 0x00 */
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        /* 0x10 */
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        /* 0x20 */
        0xffff, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
        0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
        /* 0x30 */
        0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
        0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
        /* 0x40 */
        0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
        0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
        /* 0x50 */
        0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
        0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
        /* 0x60 */
        0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
        0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
        /* 0x70 */
        0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
        0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0xffff,
        /* 0x80 */
        0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e3, 0x00e0, 0x00c1, 0x00e7,
        0x00ea, 0x00ca, 0x00e8, 0x00cd, 0x00d4, 0x00ec, 0x00c3, 0x00c2,
        /* 0x90 */
        0xffff, 0x00c0, 0x00c8, 0x00f4, 0x00f5, 0x00f2, 0x00da, 0x00f9,
        0x00cc, 0x00d5, 0x00dc, 0x00a2, 0x00a3, 0x00d9, 0x20a7, 0x00d3,
        /* 0xa0 */
        0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
        0x00bf, 0x00d2, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb,
        /* 0xb0 */
        0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
        0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
        /* 0xc0 */
        0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
        0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
        /* 0xd0 */
        0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
        0x256a, 0x2518, 0x250c, 0xffff, 0x2584, 0x258c, 0x2590, 0x2580,
        /* 0xe0 */
        0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4,
        0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x03c6, 0x03b5, 0xffff,
        /* 0xf0 */
        0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248,
        0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2, 0x25a0, 0x00a0
    }
};

/* Spanish */
static const pl_symbol_map_t map_2S_unicode = {
    map_header(2, 'S', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x00a3, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x00a7, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x00a1, 0x00d1, 0x00bf, 0x02c6, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x00b0, 0x00f1, 0x00e7, 0x02dc, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};
/* Norwegian version 1 */
static const pl_symbol_map_t map_0D_unicode = {
    map_header(0, 'D', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x00c6, 0x00d8, 0x00c5, 0x02c6, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x00e6, 0x00f8, 0x00e5, 0x00af, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};

/***************************
 * Roman-8 Unicode mapping *
 ***************************/
const pl_symbol_map_t map_8U_unicode = {
    map_header(8, 'U', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /* 0x60 */
     0x2018, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0x00a0, 0x00c0, 0x00c2, 0x00c8, 0x00ca, 0x00cb, 0x00ce, 0x00cf,
     0x00b4, 0x0060, 0x02c6, 0x00a8, 0x02dc, 0x00d9, 0x00db, 0x00a3,
     /* 0xb0 */
     0x00af, 0x00dd, 0x00fd, 0x00b0, 0x00c7, 0x00e7, 0x00d1, 0x00f1,
     0x00a1, 0x00bf, 0x00a4, 0x00a3, 0x00a5, 0x00a7, 0x0192, 0x00a2,
     /* 0xc0 */
     0x00e2, 0x00ea, 0x00f4, 0x00fb, 0x00e1, 0x00e9, 0x00f3, 0x00fa,
     0x00e0, 0x00e8, 0x00f2, 0x00f9, 0x00e4, 0x00eb, 0x00f6, 0x00fc,
     /* 0xd0 */
     0x00c5, 0x00ee, 0x00d8, 0x00c6, 0x00e5, 0x00ed, 0x00f8, 0x00e6,
     0x00c4, 0x00ec, 0x00d6, 0x00dc, 0x00c9, 0x00ef, 0x00df, 0x00d4,
     /* 0xe0 */
     0x00c1, 0x00c3, 0x00e3, 0x00d0, 0x00f0, 0x00cd, 0x00cc, 0x00d3,
     0x00d2, 0x00d5, 0x00f5, 0x0160, 0x0161, 0x00da, 0x0178, 0x00ff,
     /* 0xf0 */
     0x00de, 0x00fe, 0x2219, 0x00b5, 0x00b6, 0x00be, 0x2212, 0x00bc,
     0x00bd, 0x00aa, 0x00ba, 0x00ab, 0x25a0, 0x00bb, 0x00b1, 0xffff}
};

/***************************
 * Desktop Unicode mapping *
 ***************************/
static const pl_symbol_map_t map_7J_unicode = {
    map_header(7, 'J', plgv_Unicode, 1, 0, 0, 0, 0, 0x88, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /* 0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /* 0x60 */
     0x2018, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0x0001, 0x00b6, 0x00a7, 0x2020, 0x2021, 0x00a9, 0x00ae, 0x2122,
     0x2105, 0x00a2, 0x2013, 0x2014, 0x2026, 0xfb01, 0xfb02, 0xffff,
     /* 0xb0 */
     0x201c, 0x201d, 0x00b5, 0x2030, 0x2022, 0x25cf, 0x25e6, 0x25cb,
     0x25aa, 0x25a0, 0x25ab, 0x25a1, 0x0027, 0x00ac, 0x00a6, 0x2017,
     /* 0xc0 */
     0x2212, 0x00b1, 0x00d7, 0x00f7, 0x00b0, 0x2032, 0x2033, 0x00bc,
     0x00bd, 0x00be, 0x00b9, 0x00b2, 0x00b3, 0x2215, 0xffff, 0xffff,
     /* 0xd0 */
     0x2039, 0x203a, 0x00ab, 0x00bb, 0x201a, 0x201e, 0x2219, 0x00a1,
     0x00bf, 0x20A7, 0x2113, 0x00a3, 0x00a5, 0x00a4, 0x0192, 0x00df,
     /* 0xe0 */
     0x00aa, 0x00ba, 0x00e6, 0x00c6, 0x00f0, 0x00d0, 0x0133, 0x0132,
     0x0142, 0x0141, 0x0153, 0x0152, 0x00f8, 0x00d8, 0x00fe, 0x00de,
     /* 0xf0 */
     0x00b4, 0x0060, 0x02c6, 0x00a8, 0x02dc, 0x02c7, 0x02d8, 0x02dd,
     0x02da, 0x02d9, 0x02c9, 0x00b8, 0x02db, 0x00b7, 0x0131, 0xffff}
};

/***************************
 * PC-1004 OS/2         9J *
 ***************************/
static const pl_symbol_map_t map_9J_unicode = {
    map_header(9, 'J', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0x02c9, 0x02d8, 0x02d9, 0xffff,
     0x02da, 0xffff, 0x02dd, 0x02db, 0x02c7, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     /* NB check if 0x7f is 0x7f instead 0x2302 */
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2302,
     /*0x80 */
     0xffff, 0xffff, 0x201a, 0xffff, 0x201e, 0x2026, 0x2020, 0x2021,
     0xeff7, 0x2030, 0x0160, 0x2039, 0x0152, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014,
     0xeff5, 0x2122, 0x0161, 0x203a, 0x0153, 0xffff, 0xffff, 0x0178,
     /*0xa0 */
     0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
     0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x002d, 0x00ae, 0x00af,
     /*0xb0 */
     0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x2219,
     0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
     /*0xc0 */
     0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
     0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
     /*0xd0 */
     0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
     0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
     /*0xe0 */
     0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
     0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
     /*0xf0 */
     0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
     0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff}
};

/**************************************
 * ISO 8859/1 Latin 1 Unicode mapping *
 **************************************/
static const pl_symbol_map_t map_0N_unicode = {
    map_header(0, 'N', plgv_Unicode, 1, 0, 0, 0, 0, 0xc4, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2592,
     /* 0x80 */
     0x20ac, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
     0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x02c9,
     /* 0xb0 */
     0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x2219,
     0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
     /* 0xc0 */
     0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
     0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
     /* 0xd0 */
     0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
     0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
     /* 0xe0 */
     0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
     0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
     /* 0xf0 */
     0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
     0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff}
};

/***************************************
 * Windows 3.1 Latin 1 Unicode mapping *
 ***************************************/
/*
 * This is identical to the ISO 8859/1 Latin 1 mapping, except that
 * it has added characters in positions 0x80 - 0x9f.
 */
static const pl_symbol_map_t map_19U_unicode = {
    map_header(19, 'U', plgv_Unicode, 2, 0, 0, 0, 0, 0xc6, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2592,
     /* 0x80 */
     0x20ac, 0xffff, 0x201a, 0x0192, 0x201e, 0x2026, 0x2020, 0x2021,
     0x02c6, 0x2030, 0x0160, 0x2039, 0x0152, 0xffff, 0x017d, 0xffff,
     /* 0x90 */
     0xffff, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014,
     0x02dc, 0x2122, 0x0161, 0x203a, 0x0153, 0xffff, 0x017e, 0x0178,
     /* 0xa0 */
     0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
     0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x002d, 0x00ae, 0x00af,
     /* 0xb0 */
     0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7,
     0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
     /* 0xc0 */
     0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
     0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
     /* 0xd0 */
     0x0110, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
     0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
     /* 0xe0 */
     0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
     0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
     /* 0xf0 */
     0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
     0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff,
     }
};

/**************************************
 * PC-8 Code Page 437 Unicode mapping *
 **************************************/
/*
 * Codes 0x00-0x1f are shared with other PC mappings;
 * codes 0x80-0xaf are (almost) shared with the PC-850 Multilingual map.
 */
static const pl_symbol_map_t map_10U_unicode = {
    map_header(10, 'U', plgv_Unicode, 2, 0, 0, 0, 0, 0xc2, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0x263a, 0x263b, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
     0x25d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x266a, 0x266b, 0x263c,
     /* 0x10 */
     0x25ba, 0x25c4, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8,
     0x2191, 0x2193, 0x2192, 0x2190, 0x221f, 0x2194, 0x25b2, 0x25bC,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2302,
     /* 0x80 */
     0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
     0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
     /* 0x90 */
     0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
     0x00ff, 0x00d6, 0x00dc, 0x00a2, 0x00a3, 0x00a5, 0x20a7, 0x0192,
     /* 0xa0 */
     0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
     0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb,
     /*0xb0 */
     0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
     0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
     /*0xc0 */
     0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
     0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
     /*0xd0 */
     0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
     0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
     /*0xe0 */
     0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4,
     0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x03c6, 0x03b5, 0x2229,
     /*0xf0 */
     0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248,
     0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2, 0x25aa, 0x00a0}
};

/***************************************
 * PC-850 Multilingual Unicode mapping *
 ***************************************/
static const pl_symbol_map_t map_12U_unicode = {
    map_header(12, 'U', plgv_Unicode, 2, 0, 0, 0, 0, 0xe6, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0x263a, 0x263b, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
     0x25d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x266a, 0x266b, 0x263c,
     /* 0x10 */
     0x25ba, 0x25c4, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8,
     0x2191, 0x2193, 0x2192, 0x2190, 0x221f, 0x2194, 0x25b2, 0x25bc,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2302,
     /* 0x80 */
     0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
     0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
     /* 0x90 */
     0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
     0x00ff, 0x00d6, 0x00dc, 0x00f8, 0x00a3, 0x00d8, 0x00d7, 0x0192,
     /* 0xa0 */
     0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
     0x00bf, 0x00ae, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb,
     /*0xb0 */
     0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1, 0x00c2, 0x00c0,
     0x00a9, 0x2563, 0x2551, 0x2557, 0x255d, 0x00a2, 0x00a5, 0x2510,
     /*0xc0 */
     0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x00e3, 0x00c3,
     0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4,
     /*0xd0 */
     0x00f0, 0x00d0, 0x00ca, 0x00cb, 0x00c8, 0x0131, 0x00cd, 0x00ce,
     0x00cf, 0x2518, 0x250c, 0x2588, 0x2584, 0x00a6, 0x00cc, 0x2580,
     /*0xe0 */
     0x00d3, 0x00df, 0x00d4, 0x00d2, 0x00f5, 0x00d5, 0x00b5, 0x00fe,
     0x00de, 0x00da, 0x00db, 0x00d9, 0x00fd, 0x00dd, 0x02c9, 0x00b4,
     /*0xf0 */
     0x002d, 0x00b1, 0x2017, 0x00be, 0x00b6, 0x00a7, 0x00f7, 0x00b8,
     0x00b0, 0x00a8, 0x00b7, 0x00b9, 0x00b3, 0x00b2, 0x25aa, 0x00a0}
};

/********************************
 * PS Text (TS) Unicode mapping *
 ********************************/
static const pl_symbol_map_t map_10J_unicode = {
    map_header(10, 'J', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0x80, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x01 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x02 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /*0x60 */
     0x2018, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x0097,
     /*0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x2215, 0x00a5, 0x0192, 0x00a7,
     0x00a4, 0x0027, 0x201c, 0x00ab, 0x2039, 0x203a, 0xfb01, 0xfb02,
     /*0xb0 */
     0xffff, 0x2013, 0x2020, 0x2021, 0x2219, 0xffff, 0x00b6, 0x2022,
     0x201a, 0x201e, 0x201d, 0x00bb, 0x2026, 0x2030, 0xffff, 0x00bf,
     /*0xc0 */
     0xffff, 0x0060, 0x00b4, 0x02c6, 0x02dc, 0x02c9, 0x02d8, 0x02d9,
     0x00a8, 0xffff, 0x02da, 0x00b8, 0xffff, 0x02dd, 0x02db, 0x02c7,
     /*0xd0 */
     0x2014, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xe0 */
     0xffff, 0x00c6, 0xffff, 0x00aa, 0xffff, 0xffff, 0xffff, 0xffff,
     0x0141, 0x00d8, 0x0152, 0x00ba, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xf0 */
     0xffff, 0x00e6, 0xffff, 0xffff, 0xffff, 0x0131, 0xffff, 0xffff,
     0x0142, 0x00f8, 0x0153, 0x00df, 0xffff, 0xffff, 0xffff, 0xffff}
};

/********************************
 * MC Text (MC) Unicode mapping *
 ********************************/
static const pl_symbol_map_t map_12J_unicode = {
    map_header(12, 'J', plgv_Unicode, 2, 0, 0, 0, 0, 0xc1, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0013, /* urw 14 bad */ 0x0015, 0x0016,
     0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0xffff,
     /*0x80 */
     0x00c4, 0x00c5, 0x00c7, 0x00c9, 0x00d1, 0x00d6, 0x00dc, 0x00e1,
     0x00e0, 0x00e2, 0x00e4, 0x00e3, 0x00e5, 0x00e7, 0x00e9, 0x00e8,
     /*0x90 */
     0x00ea, 0x00eb, 0x00ed, 0x00ec, 0x00ee, 0x00ef, 0x00f1, 0x00f3,
     0x00f2, 0x00f4, 0x00f6, 0x00f5, 0x00fa, 0x00f9, 0x00fb, 0x00fc,
     /*0xa0 */
     0x2020, 0x00b0, 0x00a2, 0x00a3, 0x00a7, 0x2022, 0x00b6, 0x00df,
     0x00ae, 0x00a9, 0x2122, 0x00b4, 0x00a8, 0x2260, 0x00c6, 0x00d8,
     /*0xb0 */
     0x221e, 0x00b1, 0x2264, 0x2265, 0x00a5, 0x00b5, 0x2202, 0x2211,
     0x220f, 0x03c0, 0x222b, 0x00aa, 0x00ba, 0x03a9, 0x00e6, 0x00f8,
     /*0xc0 */
     0x00bf, 0x00a1, 0x00ac, 0x221a, 0x0192, 0x2248, 0x2206, 0x00ab,
     0x00bb, 0x2026, 0x00a0, 0x00c0, 0x00c3, 0x00d5, 0x0152, 0x0153,
     /*0xd0 */
     0x2013, 0x2014, 0x201c, 0x201d, 0x2018, 0x2019, 0x00f7, 0x25ca,
     0x00ff, 0x0178, 0x2215, 0x20ac, 0x2039, 0x203a, 0xfb01, 0xfb02,
     /*0xe0 */
     0x2021, 0x2219, 0x201a, 0x201e, 0x2030, 0x00c2, 0x00ca, 0x00c1,
     0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00d3, 0x00d4,
     /*0xf0 */
     0xffff, 0x00d2, 0x00da, 0x00db, 0x00d9, 0x0131, 0x02c6, 0x02dc,
     0x02c9, 0x02d8, 0x02d9, 0x02da, 0x00b8, 0x02dd, 0x02db, 0x02c7}
};

/***********************************************
 * PC-8 Danish/Norweigian (PD) Unicode mapping *
 ***********************************************/
static const pl_symbol_map_t map_11U_unicode = {
    map_header(11, 'U', plgv_Unicode, 2, 0, 0, 0, 0, 0xc0, 0x40, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0x263a, 0x263b, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
     0x25d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x266a, 0x266b, 0x263c,
     /* 0x10 */
     0x25ba, 0x25c4, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8,
     0x2191, 0x2193, 0x2192, 0x2190, 0x221f, 0x2194, 0x25b2, 0x25bC,
     /*0x20 */
     0xffff, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2302,
     /*0x80 */
     0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
     0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
     /*0x90 */
     0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
     0x00ff, 0x00d6, 0x00dc, 0x00f8, 0x00a3, 0x00d8, 0x013f, 0x0140,
     /*0xa0 */
     0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00f5, 0x00d5,
     0x00bf, 0x00e3, 0x00c3, 0x2113, 0x0149, 0x00a1, 0x00b3, 0x00a4,
     /*0xb0 */
     0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
     0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
     /*0xc0 */
     0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
     0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
     /*0xd0 */
     0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
     0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
     /*0xe0 */
     0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4,
     0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x03c6, 0x03b5, 0x2229,
     /*0xf0 */
     0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248,
     0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2, 0x25aa, 0x00a0}
};

/******************************
 * Legal (LG) Unicode mapping *
 ******************************/
static const pl_symbol_map_t map_1U_unicode = {
    map_header(1, 'U', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x001e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x2033, 0x0023, 0x0024, 0x0025, 0x0026, 0x2032,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x2017, 0x003d, 0x00a2, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x00ae, 0x005d, 0x00a9, 0x005f,
     /*0x60 */
     0x00b0, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x00a7, 0x00b6, 0x2020, 0x2122, 0x2592,
     /*0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};

/***************************************
 * Windows 3.0 Latin 1 Unicode mapping *
 ***************************************/
/*
 * This is identical to the ISO 8859/1 Latin 1 mapping, except that
 * it has added characters in positions 0x91 - 0x92.
 */
static const pl_symbol_map_t map_9U_unicode = {
    map_header(9, 'U', plgv_Unicode, 2, 0, 0, 0, 0, 0xcc, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2592,
     /* 0x80 */
     0x20ac, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0x2018, 0x2019, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
     0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x02c9,
     /* 0xb0 */
     0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7,
     0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
     /* 0xc0 */
     0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
     0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
     /* 0xd0 */
     0x0110, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
     0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
     /* 0xe0 */
     0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
     0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
     /* 0xf0 */
     0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
     0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff}
};

/*********************************************
 * Microsoft Publishing (PB) Unicode mapping *
 *********************************************/
static const pl_symbol_map_t map_6J_unicode = {
    map_header(6, 'J', plgv_Unicode, 2, 0, 0, 0, 0, 0xc8, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x20 */
     0x0020, 0x00b9, 0x201d, 0x00b3, 0x2074, 0x2075, 0x2077, 0x2019,
     0x2079, 0x2070, 0x2078, 0x2020, 0x201a, 0x2212, 0x2026, 0x2215,
     /*0x30 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0x201e, 0x2021, 0xffff, 0xffff,
     /*0x40 */
     0x00b2, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0x013d, 0x2014, 0x2013, 0x0152,
     /*0x50 */
     0xffff, 0xffff, 0x211e, 0x0160, 0x2122, 0xffff, 0xffff, 0xffff,
     0xffff, 0x0178, 0x017d, 0xffff, 0xffff, 0xffff, 0x2076, 0x2017,
     /*0x60 */
     0x2018, 0xffff, 0xffff, 0x2105, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0x2113, 0xffff, 0xffff, 0x0153,
     /*0x70 */
     0xffff, 0xffff, 0xffff, 0x0161, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0x017e, 0xffff, 0xffff, 0xffff, 0x201c, 0xffff,
     /*0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x20a7, 0x0192,
     /*0xa0 */
     0xffff, 0x2032, 0x2033, 0x005e, 0x007E, 0xffff, 0xffff, 0xffff,
     0xffff, 0xfb01, 0xfb02, 0xfb00, 0xfb03, 0xfb04, 0x2039, 0x203a,
     /*0xb0 */
     0x00b0, 0x2022, 0x25cf, 0xeffa, 0x25e6, 0x25cb, 0xeffd, 0x25aa,
     0x25a0, 0xeffb, 0x25ab, 0x25a1, 0xeffc, 0x2030, 0x25c6, 0x25c7,
     /*0xc0 */
     0xffff, 0xeff8, 0xeff9, 0xeff7, 0xeff5, 0xefef, 0xefee, 0xefed,
     0xeff6, 0xffff, 0xeff3, 0xeff2, 0xffff, 0xeff0, 0xeff1, 0xeff4,
     /*0xd0 */
     0xffff, 0x0060, 0x00b4, 0x02c6, 0x02dc, 0x02c9, 0x02d8, 0x02d9,
     0x00a8, 0xffff, 0x02da, 0x00b8, 0xffff, 0x02dd, 0x02db, 0x02c7,
     /*0xe0 */
     0x03a9, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0132, 0x013f,
     0x0141, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0149,
     /*0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0131, 0x0133, 0x0140,
     0x0142, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};

/**********************************
 * PC-852 Latin 2 Unicode mapping *
 **********************************/
static const pl_symbol_map_t map_17U_unicode = {
    map_header(17, 'U', plgv_Unicode, 2, 0, 0, 0, 0, 0xe4, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0x263a, 0x263b, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
     0x25d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x266a, 0x266b, 0x263c,
     /* 0x10 */
     0x25ba, 0x25c4, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8,
     0x2191, 0x2193, 0x2192, 0x2190, 0x221f, 0x2194, 0x25b2, 0x25bc,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2302,
     /* 0x80 */
     0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x016f, 0x0107, 0x00e7,
     0x0142, 0x00eb, 0x0150, 0x0151, 0x00ee, 0x0179, 0x00c4, 0x0106,
     /* 0x90 */
     0x00c9, 0x0139, 0x013a, 0x00f4, 0x00f6, 0x013d, 0x013e, 0x015a,
     0x015b, 0x00d6, 0x00dc, 0x0164, 0x0165, 0x0141, 0x00d7, 0x010d,
     /* 0xa0 */
     0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x0104, 0x0105, 0x017d, 0x017e,
     0x0118, 0x0119, 0xffff, 0x017a, 0x010c, 0x015f, 0x00ab, 0x00bb,
     /*0xb0 */
     0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1, 0x00c2, 0x011a,
     0x015e, 0x2563, 0x2551, 0x2557, 0x255d, 0x017b, 0x017c, 0x2510,
     /*0xc0 */
     0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x0102, 0x0103,
     0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4,
     /*0xd0 */
     0x0111, 0x0110, 0x010e, 0x00cb, 0x010f, 0x0147, 0x00cd, 0x00ce,
     0x011b, 0x2518, 0x250c, 0x2588, 0x2584, 0x0162, 0x016e, 0x2580,
     /*0xe0 */
     0x00d3, 0x00df, 0x00d4, 0x0143, 0x0144, 0x0148, 0x0160, 0x0161,
     0x0154, 0x00da, 0x0155, 0x0170, 0x00fd, 0x00dd, 0x0163, 0x00b4,
     /*0xf0 */
     0x00ad, 0x02dd, 0x02db, 0x02c7, 0x02d8, 0x00a7, 0x00f7, 0x00b8,
     0x00b0, 0x00a8, 0x02d9, 0x0171, 0x0158, 0x0159, 0x25aa, 0x00a0}
};

/************************************
 * PC-Turkish  (PT) Unicode mapping *
 ************************************/
static const pl_symbol_map_t map_9T_unicode = {
    map_header(9, 'T', plgv_Unicode, 2, 0, 0, 0, 0, 0xd0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0xffff, 0x263a, 0x263b, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
     0x25d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x266a, 0x266b, 0x263c,
     /*0x10 */
     0x25ba, 0x25c4, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8,
     0x2191, 0x2193, 0x2192, 0x2190, 0x221f, 0x2194, 0x25b2, 0x25bC,
     /*0x20 */
     0xffff, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2302,
     /*0x80 */
     0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
     0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x0131, 0x00c4, 0x00c5,
     /*0x90 */
     0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
     0x0130, 0x00d6, 0x00dc, 0x00a2, 0x00a3, 0x00a5, 0x015e, 0x15f,
     /*0xa0 */
     0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x011e, 0x011f,
     0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb,
     /*0xb0 */
     0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
     0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
     /*0xc0 */
     0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
     0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
     /*0xd0 */
     0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
     0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
     /*0xe0 */
     0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4,
     0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x03c6, 0x03b5, 0x2229,
     /*0xf0 */
     0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248,
     0x00b0, 0x2219, 0x2219, 0x221a, 0x207f, 0x00b2, 0x25aa, 0x00a0}
};

/*******************************************
 * ISO 8859/2 Latin 2 (E2) Unicode mapping *
 *******************************************/
static const pl_symbol_map_t map_2N_unicode = {
    map_header(2, 'N', plgv_Unicode, 1, 0, 0, 0, 0, 0xe0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2592,
     /*0x80 */
     0x20ac, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0x00a0, 0x0104, 0x02d8, 0x0141, 0x00a4, 0x013d, 0x015a, 0x00a7,
     0x00a8, 0x0160, 0x015e, 0x0164, 0x0179, 0x00ad, 0x017d, 0x017b,
     /*0xb0 */
     0x00b0, 0x0105, 0x02db, 0x0142, 0x00b4, 0x013e, 0x015b, 0x02c7,
     0x00b8, 0x0161, 0x015f, 0x0165, 0x017a, 0x02dd, 0x017e, 0x017c,
     /*0xc0 */
     0x0154, 0x00c1, 0x00c2, 0x0102, 0x00c4, 0x0139, 0x0106, 0x00c7,
     0x010c, 0x00c9, 0x0118, 0x00cb, 0x011A, 0x00cd, 0x00ce, 0x010e,
     /*0xd0 */
     0x00d0, 0x0143, 0x0147, 0x00d3, 0x00d4, 0x0150, 0x00d6, 0x00d7,
     0x0158, 0x016e, 0x00da, 0x0170, 0x00dc, 0x00dd, 0x0162, 0x00df,
     /*0xe0 */
     0x0155, 0x00e1, 0x00e2, 0x0103, 0x00e4, 0x013a, 0x0107, 0x00e7,
     0x010d, 0x00e9, 0x0119, 0x00eb, 0x011b, 0x00ed, 0x00ee, 0x010f,
     /*0xf0 */
     0x0111, 0x0144, 0x0148, 0x00f3, 0x00f4, 0x0151, 0x00f6, 0x00f7,
     0x0159, 0x016f, 0x00fa, 0x0171, 0x00fc, 0x00fd, 0x0163, 0x02d9}
};

/********************************************
 * Windows 3.1 Latin 2 (EE) Unicode mapping *
 * windows-1250                             *
 ********************************************/
static const pl_symbol_map_t map_9E_unicode = {
    map_header(9, 'E', plgv_Unicode, 2, 0, 0, 0, 0, 0xe0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0013, /*urw 14 bad */ 0x0015, 0x0016,
     0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x07e /*0x023c */ ,
     0x2592 /*0x0097 */ ,
     /*0x80 */
     0x20ac, 0xffff, 0x201a, 0xffff, 0x201e, 0x2026, 0x2020, 0x2021,
     0xffff, 0x2030, 0x0160, 0x2039, 0x015a, 0x0164, 0x017d, 0x0179,
     /*0x90 */
     0xffff, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014,
     0xffff, 0x2122, 0x0161, 0x203a, 0x015b, 0x0165, 0x017e, 0x017a,
     /*0xa0 */
     0x00a0, 0x02c7, 0x02d8, 0x0141, 0x00a4, 0x0104, 0x00a6, 0x00a7,
     0x00a8, 0x00a9, 0x015e, 0x00ab, 0x00ac, 0x002d, 0x00ae, 0x017b,
     /*0xb0 */
     0x00b0, 0x00b1, 0x02db, 0x0142, 0x00b4, 0x00b5, 0x00b6, 0x2219,
     0x00b8, 0x0105, 0x015f, 0x00bb, 0x013d, 0x02dd, 0x013e, 0x017c,
     /*0xc0 */
     0x0154, 0x00c1, 0x00c2, 0x0102, 0x00c4, 0x0139, 0x0106, 0x00c7,
     0x010c, 0x00c9, 0x0118, 0x00cb, 0x011a, 0x00cd, 0x00ce, 0x010e,
     /*0xd0 */
     0x00d0, 0x0143, 0x0147, 0x00d3, 0x00d4, 0x0150, 0x00d6, 0x00d7,
     0x0158, 0x016e, 0x00da, 0x0170, 0x00dc, 0x00dd, 0x0162, 0x00df,
     /*0xe0 */
     0x0155, 0x00e1, 0x00e2, 0x0103, 0x00e4, 0x013a, 0x0107, 0x00e7,
     0x010d, 0x00e9, 0x0119, 0x00eb, 0x011b, 0x00ed, 0x00ee, 0x010f,
     /*0xf0 */
     0x0111, 0x0144, 0x0148, 0x00f3, 0x00f4, 0x0151, 0x00f6, 0x00f7,
     0x0159, 0x016f, 0x00fa, 0x0171, 0x00fc, 0x00fd, 0x0163, 0x02d9,
     }
};

/* NB The following 3 symbols sets need to be formatted consistently
   with the other symbol sets. */

/********************************************
 * windows-1251 Cyrillic                    *
 ********************************************/
static const pl_symbol_map_t map_9R_unicode = {
    map_header(9, 'R', plgv_Unicode, 2, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     0x0000,
     0x0001,
     0x0002,
     0x0003,
     0x0004,
     0x0005,
     0x0006,
     0x0007,
     0x0008,
     0x0009,
     0x000a,
     0x000b,
     0x000c,
     0x000d,
     0x000e,
     0x000f,
     0x0010,
     0x0011,
     0x0012,
     0x0013,
     0x0014,
     0x0015,
     0x0016,
     0x0017,
     0x0018,
     0x0019,
     0x001a,
     0x001b,
     0x001c,
     0x001d,
     0x001e,
     0x001f,
     0x0020,
     0x0021,
     0x0022,
     0x0023,
     0x0024,
     0x0025,
     0x0026,
     0x0027,
     0x0028,
     0x0029,
     0x002a,
     0x002b,
     0x002c,
     0x002d,
     0x002e,
     0x002f,
     0x0030,
     0x0031,
     0x0032,
     0x0033,
     0x0034,
     0x0035,
     0x0036,
     0x0037,
     0x0038,
     0x0039,
     0x003a,
     0x003b,
     0x003c,
     0x003d,
     0x003e,
     0x003f,
     0x0040,
     0x0041,
     0x0042,
     0x0043,
     0x0044,
     0x0045,
     0x0046,
     0x0047,
     0x0048,
     0x0049,
     0x004a,
     0x004b,
     0x004c,
     0x004d,
     0x004e,
     0x004f,
     0x0050,
     0x0051,
     0x0052,
     0x0053,
     0x0054,
     0x0055,
     0x0056,
     0x0057,
     0x0058,
     0x0059,
     0x005a,
     0x005b,
     0x005c,
     0x005d,
     0x005e,
     0x005f,
     0x0060,
     0x0061,
     0x0062,
     0x0063,
     0x0064,
     0x0065,
     0x0066,
     0x0067,
     0x0068,
     0x0069,
     0x006a,
     0x006b,
     0x006c,
     0x006d,
     0x006e,
     0x006f,
     0x0070,
     0x0071,
     0x0072,
     0x0073,
     0x0074,
     0x0075,
     0x0076,
     0x0077,
     0x0078,
     0x0079,
     0x007a,
     0x007b,
     0x007c,
     0x007d,
     0x007e,
     0x007f,
     0x0402,
     0x0403,
     0x201a,
     0x0453,
     0x201e,
     0x2026,
     0x2020,
     0x2021,
     0x20ac,
     0x2030,
     0x0409,
     0x2039,
     0x040a,
     0x040c,
     0x040b,
     0x040f,
     0x0452,
     0x2018,
     0x2019,
     0x201c,
     0x201d,
     0x2022,
     0x2013,
     0x2014,
     0xffff,
     0x2122,
     0x0459,
     0x203a,
     0x045a,
     0x045c,
     0x045b,
     0x045f,
     0x00a0,
     0x040e,
     0x045e,
     0x0408,
     0x00a4,
     0x0490,
     0x00a6,
     0x00a7,
     0x0401,
     0x00a9,
     0x0404,
     0x00ab,
     0x00ac,
     0x00ad,
     0x00ae,
     0x0407,
     0x00b0,
     0x00b1,
     0x0406,
     0x0456,
     0x0491,
     0x00b5,
     0x00b6,
     0x00b7,
     0x0451,
     0x2116,
     0x0454,
     0x00bb,
     0x0458,
     0x0405,
     0x0455,
     0x0457,
     0x0410,
     0x0411,
     0x0412,
     0x0413,
     0x0414,
     0x0415,
     0x0416,
     0x0417,
     0x0418,
     0x0419,
     0x041a,
     0x041b,
     0x041c,
     0x041d,
     0x041e,
     0x041f,
     0x0420,
     0x0421,
     0x0422,
     0x0423,
     0x0424,
     0x0425,
     0x0426,
     0x0427,
     0x0428,
     0x0429,
     0x042a,
     0x042b,
     0x042c,
     0x042d,
     0x042e,
     0x042f,
     0x0430,
     0x0431,
     0x0432,
     0x0433,
     0x0434,
     0x0435,
     0x0436,
     0x0437,
     0x0438,
     0x0439,
     0x043a,
     0x043b,
     0x043c,
     0x043d,
     0x043e,
     0x043f,
     0x0440,
     0x0441,
     0x0442,
     0x0443,
     0x0444,
     0x0445,
     0x0446,
     0x0447,
     0x0448,
     0x0449,
     0x044a,
     0x044b,
     0x044c,
     0x044d,
     0x044e,
     0x044f}
};
/********************************************
 * windows-1253 Greek                       *
 ********************************************/
static const pl_symbol_map_t map_9G_unicode = {
    map_header(9, 'G', plgv_Unicode, 2, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     0x0000,
     0x0001,
     0x0002,
     0x0003,
     0x0004,
     0x0005,
     0x0006,
     0x0007,
     0x0008,
     0x0009,
     0x000a,
     0x000b,
     0x000c,
     0x000d,
     0x000e,
     0x000f,
     0x0010,
     0x0011,
     0x0012,
     0x0013,
     0x0014,
     0x0015,
     0x0016,
     0x0017,
     0x0018,
     0x0019,
     0x001a,
     0x001b,
     0x001c,
     0x001d,
     0x001e,
     0x001f,
     0x0020,
     0x0021,
     0x0022,
     0x0023,
     0x0024,
     0x0025,
     0x0026,
     0x0027,
     0x0028,
     0x0029,
     0x002a,
     0x002b,
     0x002c,
     0x002d,
     0x002e,
     0x002f,
     0x0030,
     0x0031,
     0x0032,
     0x0033,
     0x0034,
     0x0035,
     0x0036,
     0x0037,
     0x0038,
     0x0039,
     0x003a,
     0x003b,
     0x003c,
     0x003d,
     0x003e,
     0x003f,
     0x0040,
     0x0041,
     0x0042,
     0x0043,
     0x0044,
     0x0045,
     0x0046,
     0x0047,
     0x0048,
     0x0049,
     0x004a,
     0x004b,
     0x004c,
     0x004d,
     0x004e,
     0x004f,
     0x0050,
     0x0051,
     0x0052,
     0x0053,
     0x0054,
     0x0055,
     0x0056,
     0x0057,
     0x0058,
     0x0059,
     0x005a,
     0x005b,
     0x005c,
     0x005d,
     0x005e,
     0x005f,
     0x0060,
     0x0061,
     0x0062,
     0x0063,
     0x0064,
     0x0065,
     0x0066,
     0x0067,
     0x0068,
     0x0069,
     0x006a,
     0x006b,
     0x006c,
     0x006d,
     0x006e,
     0x006f,
     0x0070,
     0x0071,
     0x0072,
     0x0073,
     0x0074,
     0x0075,
     0x0076,
     0x0077,
     0x0078,
     0x0079,
     0x007a,
     0x007b,
     0x007c,
     0x007d,
     0x007e,
     0x007f,
     0x20ac,
     0xffff,
     0x201a,
     0x0192,
     0x201e,
     0x2026,
     0x2020,
     0x2021,
     0xffff,
     0x2030,
     0xffff,
     0x2039,
     0xffff,
     0xffff,
     0xffff,
     0xffff,
     0xffff,
     0x2018,
     0x2019,
     0x201c,
     0x201d,
     0x2022,
     0x2013,
     0x2014,
     0xffff,
     0x2122,
     0xffff,
     0x203a,
     0xffff,
     0xffff,
     0xffff,
     0xffff,
     0x00a0,
     0x0385,
     0x0386,
     0x00a3,
     0x00a4,
     0x00a5,
     0x00a6,
     0x00a7,
     0x00a8,
     0x00a9,
     0xffff,
     0x00ab,
     0x00ac,
     0x00ad,
     0x00ae,
     0x2015,
     0x00b0,
     0x00b1,
     0x00b2,
     0x00b3,
     0x0384,
     0x00b5,
     0x00b6,
     0x00b7,
     0x0388,
     0x0389,
     0x038a,
     0x00bb,
     0x038c,
     0x00bd,
     0x038e,
     0x038f,
     0x0390,
     0x0391,
     0x0392,
     0x0393,
     0x0394,
     0x0395,
     0x0396,
     0x0397,
     0x0398,
     0x0399,
     0x039a,
     0x039b,
     0x039c,
     0x039d,
     0x039e,
     0x039f,
     0x03a0,
     0x03a1,
     0xffff,
     0x03a3,
     0x03a4,
     0x03a5,
     0x03a6,
     0x03a7,
     0x03a8,
     0x03a9,
     0x03aa,
     0x03ab,
     0x03ac,
     0x03ad,
     0x03ae,
     0x03af,
     0x03b0,
     0x03b1,
     0x03b2,
     0x03b3,
     0x03b4,
     0x03b5,
     0x03b6,
     0x03b7,
     0x03b8,
     0x03b9,
     0x03ba,
     0x03bb,
     0x03bc,
     0x03bd,
     0x03be,
     0x03bf,
     0x03c0,
     0x03c1,
     0x03c2,
     0x03c3,
     0x03c4,
     0x03c5,
     0x03c6,
     0x03c7,
     0x03c8,
     0x03c9,
     0x03ca,
     0x03cb,
     0x03cc,
     0x03cd,
     0x03ce,
     0xffff}
};
/********************************************
 * windows-1256 Arabic                      *
 ********************************************/
static const pl_symbol_map_t map_9V_unicode = {
    map_header(9, 'V', plgv_Unicode, 2, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     0x0000,
     0x0001,
     0x0002,
     0x0003,
     0x0004,
     0x0005,
     0x0006,
     0x0007,
     0x0008,
     0x0009,
     0x000a,
     0x000b,
     0x000c,
     0x000d,
     0x000e,
     0x000f,
     0x0010,
     0x0011,
     0x0012,
     0x0013,
     0x0014,
     0x0015,
     0x0016,
     0x0017,
     0x0018,
     0x0019,
     0x001a,
     0x001b,
     0x001c,
     0x001d,
     0x001e,
     0x001f,
     0x0020,
     0x0021,
     0x0022,
     0x0023,
     0x0024,
     0x0025,
     0x0026,
     0x0027,
     0x0028,
     0x0029,
     0x002a,
     0x002b,
     0x002c,
     0x002d,
     0x002e,
     0x002f,
     0x0030,
     0x0031,
     0x0032,
     0x0033,
     0x0034,
     0x0035,
     0x0036,
     0x0037,
     0x0038,
     0x0039,
     0x003a,
     0x003b,
     0x003c,
     0x003d,
     0x003e,
     0x003f,
     0x0040,
     0x0041,
     0x0042,
     0x0043,
     0x0044,
     0x0045,
     0x0046,
     0x0047,
     0x0048,
     0x0049,
     0x004a,
     0x004b,
     0x004c,
     0x004d,
     0x004e,
     0x004f,
     0x0050,
     0x0051,
     0x0052,
     0x0053,
     0x0054,
     0x0055,
     0x0056,
     0x0057,
     0x0058,
     0x0059,
     0x005a,
     0x005b,
     0x005c,
     0x005d,
     0x005e,
     0x005f,
     0x0060,
     0x0061,
     0x0062,
     0x0063,
     0x0064,
     0x0065,
     0x0066,
     0x0067,
     0x0068,
     0x0069,
     0x006a,
     0x006b,
     0x006c,
     0x006d,
     0x006e,
     0x006f,
     0x0070,
     0x0071,
     0x0072,
     0x0073,
     0x0074,
     0x0075,
     0x0076,
     0x0077,
     0x0078,
     0x0079,
     0x007a,
     0x007b,
     0x007c,
     0x007d,
     0x007e,
     0x007f,
     0x20ac,
     0x067e,
     0x201a,
     0x0192,
     0x201e,
     0x2026,
     0x2020,
     0x2021,
     0x02c6,
     0x2030,
     0x0679,
     0x2039,
     0x0152,
     0x0686,
     0x0698,
     0x0688,
     0x06af,
     0x2018,
     0x2019,
     0x201c,
     0x201d,
     0x2022,
     0x2013,
     0x2014,
     0x06a9,
     0x2122,
     0x0691,
     0x203a,
     0x0153,
     0x200c,
     0x200d,
     0x06ba,
     0x00a0,
     0x060c,
     0x00a2,
     0x00a3,
     0x00a4,
     0x00a5,
     0x00a6,
     0x00a7,
     0x00a8,
     0x00a9,
     0x06be,
     0x00ab,
     0x00ac,
     0x00ad,
     0x00ae,
     0x00af,
     0x00b0,
     0x00b1,
     0x00b2,
     0x00b3,
     0x00b4,
     0x00b5,
     0x00b6,
     0x00b7,
     0x00b8,
     0x00b9,
     0x061b,
     0x00bb,
     0x00bc,
     0x00bd,
     0x00be,
     0x061f,
     0x06c1,
     0x0621,
     0x0622,
     0x0623,
     0x0624,
     0x0625,
     0x0626,
     0x0627,
     0x0628,
     0x0629,
     0x062a,
     0x062b,
     0x062c,
     0x062d,
     0x062e,
     0x062f,
     0x0630,
     0x0631,
     0x0632,
     0x0633,
     0x0634,
     0x0635,
     0x0636,
     0x00d7,
     0x0637,
     0x0638,
     0x0639,
     0x063a,
     0x0640,
     0x0641,
     0x0642,
     0x0643,
     0x00e0,
     0x0644,
     0x00e2,
     0x0645,
     0x0646,
     0x0647,
     0x0648,
     0x00e7,
     0x00e8,
     0x00e9,
     0x00ea,
     0x00eb,
     0x0649,
     0x064a,
     0x00ee,
     0x00ef,
     0x064b,
     0x064c,
     0x064d,
     0x064e,
     0x00f4,
     0x064f,
     0x0650,
     0x00f7,
     0x0651,
     0x00f9,
     0x0652,
     0x00fb,
     0x00fc,
     0x200e,
     0x200f,
     0x06d2}
};

/*******************************************
 * ISO 8859/9 Latin 5 (E5) Unicode mapping *
 *******************************************/
static const pl_symbol_map_t map_5N_unicode = {
    map_header(5, 'N', plgv_Unicode, 1, 0, 0, 0, 0, 0xd0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2592,
     /*0x80 */
     0x20ac, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
     0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x002d, 0x00ae, 0x02c9,
     /*0xb0 */
     0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x2219,
     0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
     /*0xc0 */
     0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
     0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
     /*0xd0 */
     0x011e, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
     0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x0130, 0x015e, 0x00df,
     /*0xe0 */
     0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
     0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
     /*0xf0 */
     0x011f, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
     0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x0131, 0x015f, 0x00ff}
};

/********************************************
 * Windows 3.1 Latin 5 (WT) Unicode mapping *
 ********************************************/
static const pl_symbol_map_t map_5T_unicode = {
    map_header(5, 'T', plgv_Unicode, 2, 0, 0, 0, 0, 0xd0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2592,
     /*0x80 */
     0x20ac, 0xffff, 0x002c, 0x0192, 0x201e, 0x2026, 0x2020, 0x2021,
     0x02c6, 0x2030, 0x0160, 0x2039, 0x0152, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014,
     0x02dc, 0x2122, 0x0161, 0x203a, 0x0153, 0xffff, 0xffff, 0x0178,
     /*0xa0 */
     0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
     0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af,
     /*0xb0 */
     0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x2219,
     0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
     /*0xc0 */
     0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
     0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
     /*0xd0 */
     0x011e, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
     0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x0130, 0x015e, 0x00df,
     /*0xe0 */
     0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
     0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
     /*0xf0 */
     0x20ac, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
     0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x0131, 0x015f, 0x00ff}
};

/*******************************
 * Math-8 (M8) Unicode mapping *
 *******************************/
static const pl_symbol_map_t map_8M_unicode = {
    map_header(8, 'M', plgv_Unicode, 1, 0, 0, 0, 4, 0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0xefbf, 0x2033, 0x00b0, 0x221e, 0x00f7, 0x221d, 0x2032,
     0x0028, 0x0029, 0x00d7, 0x002b, 0x002c, 0x2212, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x212f, 0xefec, 0x003c, 0x003d, 0x003e, 0x2248,
     /*0x40 */
     0x2234, 0x0041, 0x0042, 0x0393, 0x2206, 0x0045, 0x005a, 0x0048,
     0x0398, 0x0049, 0x004b, 0x039b, 0x004d, 0x004e, 0x039e, 0x004f,
     /*0x50 */
     0x03a0, 0x0050, 0x03a3, 0x0054, 0x03a5, 0x03a6, 0x0058, 0x03a8,
     0x03a9, 0x2207, 0x2202, 0x03c2, 0x2264, 0x2260, 0x2265, 0xefeb,
     /*0x60 */
     0x2235, 0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x03b7,
     0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf,
     /*0x70 */
     0x03c0, 0x03c1, 0x03c3, 0x03c4, 0x03c5, 0x03c6, 0x03c7, 0x03c8,
     0x03c9, 0x03d1, 0x03d5, 0x03d6, 0x2243, 0x2261, 0x2262, 0x2592,
     /*0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0xffff, 0x2191, 0x2192, 0x2193, 0x2190, 0x21d1, 0x21d2, 0x21d3,
     0x21d0, 0x2195, 0x2194, 0x21d5, 0x21d4, 0x21c4, 0x21c6, 0xefe9,
     /*0xb0 */
     0xefea, 0x2200, 0x2203, 0x22a4, 0x22a5, 0x222a, 0x2229, 0x2208,
     0x220b, 0x2209, 0x2282, 0x2283, 0x2284, 0x2285, 0x2286, 0x2287,
     /*0xc0 */
     0x2295, 0x2299, 0x2297, 0x2296, 0x2298, 0x2227, 0x2228, 0x22bb,
     0x00ac, 0x2218, 0x2219, 0x2022, 0x25cf, 0x20dd, 0x2020, 0x2021,
     /*0xd0 */
     0x00c5, 0x22a3, 0x22a2, 0x221f, 0x220d, 0x222b, 0x222e, 0x2220,
     0x2205, 0x2135, 0x2136, 0x2137, 0x212d, 0x2111, 0x211c, 0x2128,
     /*0xe0 */
     0xefe7, 0xefe6, 0xefe3, 0xefe2, 0xefe1, 0x2320, 0xefd4, 0x2321,
     0xefd3, 0x2225, 0xefd2, 0xefd1, 0xefe8, 0xefcb, 0x2217, 0x2245,
     /*0xf0 */
     0xefe5, 0xefe4, 0xefe0, 0xefdf, 0xefde, 0xefdd, 0x2223, 0xefdc,
     0xefd0, 0xefcf, 0xefce, 0xefcd, 0xefcc, 0x2213, 0x00b1, 0xffff}
};

/*******************************
 * PI Font(PI) Unicode mapping *
 *******************************/
static const pl_symbol_map_t map_15U_unicode = {
    map_header(15, 'U', plgv_Unicode, 1, 0, 0, 0, 4, 0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0xffff, 0x201e, 0x201a, 0x201c, 0x201d, 0x2018, 0x2019,
     0x2329, 0x232a, 0x2122, 0x2120, 0x00ae, 0x00a9, 0xefff, 0xffff,
     /*0x30 */
     0x02c9, 0x02d8, 0x02c7, 0x02d9, 0x2197, 0x2198, 0x2199, 0x2196,
     0x25b5, 0x25b9, 0x25bf, 0x25c3, 0x226a, 0x00a7, 0x226b, 0x00b6,
     /*0x40 */
     0x2237, 0xefca, 0xffff, 0xffff, 0xffff, 0xffff, 0xefd5, 0xffff,
     0x210f, 0xffff, 0xffff, 0xffff, 0x2112, 0x2113, 0xffff, 0xffff,
     /*0x50 */
     0xeffe, 0x2118, 0x211e, 0x2211, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0x301a, 0xefc9, 0x301b, 0x2039, 0x203a,
     /*0x60 */
     0x250c, 0x2514, 0x256d, 0x2570, 0x253c, 0x251c, 0x2500, 0xefc8,
     0xefc7, 0xefc6, 0xefc5, 0xefc4, 0xeffc, 0x25c7, 0xffff, 0xffff,
     /*0x70 */
     0x2510, 0x2518, 0x256e, 0x256f, 0x252c, 0x2524, 0x2534, 0x2502,
     0xefc3, 0xefc2, 0xefc1, 0xefc0, 0xeffb, 0x25c6, 0xffff, 0x2592,
     /*0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};

/********************************
 * PS Math (MS) Unicode mapping *
 ********************************/
static const pl_symbol_map_t map_5M_unicode = {
    map_header(5, 'M', plgv_Unicode, 1, 0, 0, 0, 4, 0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x2200, 0x0023, 0x2203, 0x0025, 0x0026, 0x220d,
     0x0028, 0x0029, 0x2217, 0x002b, 0x002c, 0x2212, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x2245, 0x0041, 0x0042, 0x0058, 0x2206, 0x0045, 0x03a6, 0x0393,
     0x0048, 0x0049, 0x03d1, 0x004b, 0x039b, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x03a0, 0x0398, 0x0050, 0x03a3, 0x0054, 0x0059, 0x03c2, 0x03a9,
     0x039e, 0x03a8, 0x005a, 0x005b, 0x2234, 0x005d, 0x22a5, 0xefeb,
     /*0x60 */
     0xefea, 0x03b1, 0x03b2, 0x03c7, 0x03b4, 0x03b5, 0x03c6, 0x03b3,
     0x03b7, 0x03b9, 0x03d5, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03bf,
     /*0x70 */
     0x03c0, 0x03b8, 0x03c1, 0x03c3, 0x03c4, 0x03c5, 0x03d6, 0x03c9,
     0x03be, 0x03c8, 0x03b6, 0x007b, 0x007c, 0x007d, 0x007e, 0xffff,
     /*0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0xffff, 0x03a5, 0x2032, 0x2264, 0x2215, 0x221e, 0x0192, 0x2663,
     0x2666, 0x2665, 0x2660, 0x2194, 0x2190, 0x2191, 0x2192, 0x2193,
     /*0xb0 */
     0x00b0, 0x00b1, 0x2033, 0x2265, 0x00d7, 0x221d, 0x2202, 0x2022,
     0x00f7, 0x2260, 0x2261, 0x2248, 0x2026, 0x2223, 0xefe8, 0x21b5,
     /*0xc0 */
     0x2135, 0x2111, 0x211c, 0x2118, 0x2297, 0x2295, 0x2205, 0x2229,
     0x222a, 0x2283, 0x2287, 0x2284, 0x2282, 0x2286, 0x2208, 0x2209,
     /*0xd0 */
     0x2220, 0x2207, 0xefdb, 0xefda, 0xefd9, 0x220f, 0xefbf, 0x2219,
     0x00ac, 0x2227, 0x2228, 0x21d4, 0x21d0, 0x21d1, 0x21d2, 0x21d3,
     /*0xe0 */
     0x25ca, 0x2329, 0xefd8, 0xefd7, 0xefd6, 0x2211, 0xefe3, 0xefdd,
     0xefe1, 0xefe7, 0x2223, 0xefe6, 0xefe3, 0xefe2, 0xefe1, 0xefdd,
     /*0xf0 */
     0x20ac, 0x232a, 0x222b, 0x2320, 0xefdd, 0x2321, 0xefe0, 0xefdd,
     0xefde, 0xefe5, 0x2223, 0xefe4, 0xefe0, 0xefdf, 0xefde, 0xffff}
};

/************************************
 * Symbol Font (AS) Unicode mapping *
 ************************************/
static const pl_symbol_map_t map_19M_unicode = {
    map_header(19, 'M', plgv_Unicode, 1, 0, 0, 0, 4, 0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*00 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*10 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*20 */
     0x0020, 0xf021, 0xf022, 0xf023, 0xf024, 0xf025, 0xf026, 0xf027,
     0xf028, 0xf029, 0xf02a, 0xf02b, 0xf02c, 0xf02d, 0xf02e, 0xf02f,
     /*30 */
     0xf030, 0xf031, 0xf032, 0xf033, 0xf034, 0xf035, 0xf036, 0xf037,
     0xf038, 0xf039, 0xf03a, 0xf03b, 0xf03c, 0xf03d, 0xf03e, 0xf03f,
     /*40 */
     0xf040, 0xf041, 0xf042, 0xf043, 0xf044, 0xf045, 0xf046, 0xf047,
     0xf048, 0xf049, 0xf04a, 0xf04b, 0xf04c, 0xf04d, 0xf04e, 0xf04f,
     /*50 */
     0xf050, 0xf051, 0xf052, 0xf053, 0xf054, 0xf055, 0xf056, 0xf057,
     0xf058, 0xf059, 0xf05a, 0xf05b, 0xf05c, 0xf05d, 0xf05e, 0xf05f,
     /*60 */
     0xf060, 0xf061, 0xf062, 0xf063, 0xf064, 0xf065, 0xf066, 0xf067,
     0xf068, 0xf069, 0xf06a, 0xf06b, 0xf06c, 0xf06d, 0xf06e, 0xf06f,
     /*70 */
     0xf070, 0xf071, 0xf072, 0xf073, 0xf074, 0xf075, 0xf076, 0xf077,
     0xf078, 0xf079, 0xf07a, 0xf07b, 0xf07c, 0xf07d, 0xf07e, 0xf07f,
     /*80 */
     0xf080, 0xf081, 0xf082, 0xf083, 0xf084, 0xf085, 0xf086, 0xf087,
     0xf088, 0xf089, 0xf08a, 0xf08b, 0xf08c, 0xf08d, 0xf08e, 0xf08f,
     /*90 */
     0xf090, 0xf091, 0xf092, 0xf093, 0xf094, 0xf095, 0xf096, 0xf097,
     0xf098, 0xf099, 0xf09a, 0xf09b, 0xf09c, 0xf09d, 0xf09e, 0xf09f,
     /*a0 */
     0xf0a0, 0xf0a1, 0xf0a2, 0xf0a3, 0xf0a4, 0xf0a5, 0xf0a6, 0xf0a7,
     0xf0a8, 0xf0a9, 0xf0aa, 0xf0ab, 0xf0ac, 0xf0ad, 0xf0ae, 0xf0af,
     /*b0 */
     0xf0b0, 0xf0b1, 0xf0b2, 0xf0b3, 0xf0b4, 0xf0b5, 0xf0b6, 0xf0b7,
     0xf0b8, 0xf0b9, 0xf0ba, 0xf0bb, 0xf0bc, 0xf0bd, 0xf0be, 0xf0bf,
     /*c0 */
     0xf0c0, 0xf0c1, 0xf0c2, 0xf0c3, 0xf0c4, 0xf0c5, 0xf0c6, 0xf0c7,
     0xf0c8, 0xf0c9, 0xf0ca, 0xf0cb, 0xf0cc, 0xf0cd, 0xf0ce, 0xf0cf,
     /*d0 */
     0xf0d0, 0xf0d1, 0xf0d2, 0xf0d3, 0xf0d4, 0xf0d5, 0xf0d6, 0xf0d7,
     0xf0d8, 0xf0d9, 0xf0da, 0xf0db, 0xf0dc, 0xf0dd, 0xf0de, 0xf0df,
     /*e0 */
     0xf0e0, 0xf0e1, 0xf0e2, 0xf0e3, 0xf0e4, 0xf0e5, 0xf0e6, 0xf0e7,
     0xf0e8, 0xf0e9, 0xf0ea, 0xf0eb, 0xf0ec, 0xf0ed, 0xf0ee, 0xf0ef,
     /*f0 */
     0xf0f0, 0xf0f1, 0xf0f2, 0xf0f3, 0xf0f4, 0xf0f5, 0xf0f6, 0xf0f7,
     0xf0f8, 0xf0f9, 0xf0fa, 0xf0fb, 0xf0fc, 0xf0fd, 0xf0fe, 0xf0ff,
     }
};

/***********************************
 * Wingdings Font (L$) MSL mapping *
 ***********************************/
static const pl_symbol_map_t map_579L_unicode = {
    map_header(579, 'L', plgv_Unicode, 2, 0, 0, 0, 1, 0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*00 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*10 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*20 */
     0x0020, 0xf021, 0xf022, 0xf023, 0xf024, 0xf025, 0xf026, 0xf027,
     0xf028, 0xf029, 0xf02a, 0xf02b, 0xf02c, 0xf02d, 0xf02e, 0xf02f,
     /*30 */
     0xf030, 0xf031, 0xf032, 0xf033, 0xf034, 0xf035, 0xf036, 0xf037,
     0xf038, 0xf039, 0xf03a, 0xf03b, 0xf03c, 0xf03d, 0xf03e, 0xf03f,
     /*40 */
     0xf040, 0xf041, 0xf042, 0xf043, 0xf044, 0xf045, 0xf046, 0xf047,
     0xf048, 0xf049, 0xf04a, 0xf04b, 0xf04c, 0xf04d, 0xf04e, 0xf04f,
     /*50 */
     0xf050, 0xf051, 0xf052, 0xf053, 0xf054, 0xf055, 0xf056, 0xf057,
     0xf058, 0xf059, 0xf05a, 0xf05b, 0xf05c, 0xf05d, 0xf05e, 0xf05f,
     /*60 */
     0xf060, 0xf061, 0xf062, 0xf063, 0xf064, 0xf065, 0xf066, 0xf067,
     0xf068, 0xf069, 0xf06a, 0xf06b, 0xf06c, 0xf06d, 0xf06e, 0xf06f,
     /*70 */
     0xf070, 0xf071, 0xf072, 0xf073, 0xf074, 0xf075, 0xf076, 0xf077,
     0xf078, 0xf079, 0xf07a, 0xf07b, 0xf07c, 0xf07d, 0xf07e, 0xf07f,
     /*80 */
     0xf080, 0xf081, 0xf082, 0xf083, 0xf084, 0xf085, 0xf086, 0xf087,
     0xf088, 0xf089, 0xf08a, 0xf08b, 0xf08c, 0xf08d, 0xf08e, 0xf08f,
     /*90 */
     0xf090, 0xf091, 0xf092, 0xf093, 0xf094, 0xf095, 0xf096, 0xf097,
     0xf098, 0xf099, 0xf09a, 0xf09b, 0xf09c, 0xf09d, 0xf09e, 0xf09f,
     /*a0 */
     0xf0a0, 0xf0a1, 0xf0a2, 0xf0a3, 0xf0a4, 0xf0a5, 0xf0a6, 0xf0a7,
     0xf0a8, 0xf0a9, 0xf0aa, 0xf0ab, 0xf0ac, 0xf0ad, 0xf0ae, 0xf0af,
     /*b0 */
     0xf0b0, 0xf0b1, 0xf0b2, 0xf0b3, 0xf0b4, 0xf0b5, 0xf0b6, 0xf0b7,
     0xf0b8, 0xf0b9, 0xf0ba, 0xf0bb, 0xf0bc, 0xf0bd, 0xf0be, 0xf0bf,
     /*c0 */
     0xf0c0, 0xf0c1, 0xf0c2, 0xf0c3, 0xf0c4, 0xf0c5, 0xf0c6, 0xf0c7,
     0xf0c8, 0xf0c9, 0xf0ca, 0xf0cb, 0xf0cc, 0xf0cd, 0xf0ce, 0xf0cf,
     /*d0 */
     0xf0d0, 0xf0d1, 0xf0d2, 0xf0d3, 0xf0d4, 0xf0d5, 0xf0d6, 0xf0d7,
     0xf0d8, 0xf0d9, 0xf0da, 0xf0db, 0xf0dc, 0xf0dd, 0xf0de, 0xf0df,
     /*e0 */
     0xf0e0, 0xf0e1, 0xf0e2, 0xf0e3, 0xf0e4, 0xf0e5, 0xf0e6, 0xf0e7,
     0xf0e8, 0xf0e9, 0xf0ea, 0xf0eb, 0xf0ec, 0xf0ed, 0xf0ee, 0xf0ef,
     /*f0 */
     0xf0f0, 0xf0f1, 0xf0f2, 0xf0f3, 0xf0f4, 0xf0f5, 0xf0f6, 0xf0f7,
     0xf0f8, 0xf0f9, 0xf0fa, 0xf0fb, 0xf0fc, 0xf0fd, 0xf0fe, 0xf0ff,
     }
};

/*****************************************
 * PS ITC Zapf Dingbats (DS) MSL mapping *
 *****************************************/
static const pl_symbol_map_t map_10L_msl = {
    map_header(10, 'L', plgv_MSL, 1, 0, 0, 0, 1, 0, 0, 0, 0, PLGV_NO_MAPPING),
    {
     /*00 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*10 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*20 */
     ffff, 2109, 2096, 2120, 2497, 2498, 2315, 2458,
     2013, 2108, 2095, 2295, 2313, 2515, 2499, 2307,
     /*30 */
     2710, 2324, 2121, 2010, 2210, 2286, 2254, 2027,
     2077, 2444, 2054, 2488, 2456, 2044, 2076, 2476,
     /*40 */
     2230, 2411, 2014, 2472, 2478, 2502, 2448, 2011,
     2051, 2729, 2052, 2047, 2032, 2060, 2078, 2092,
     /*50 */
     2058, 2256, 2252, 2484, 2447, 2479, 2046, 2090,
     2240, 2739, 2452, 2279, 2483, 2430, 2290, 2451,
     /*60 */
     2460, 2288, 2490, 2258, 2464, 2492, 2432, 2260,
     2292, 2072, 2264, 2232, 2091, 2116, 2291, 2282,
     /*70 */
     2325, 2250, 2308, 2048, 2248, 2491, 2089, 2281,
     2480, 2486, 2454, 2100, 2113, 2501, 2517, ffff,
     /*80 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*90 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*a0 */
     ffff, 2500, 2428, 2410, 2414, 2213, 2317, 2300,
     2445, 2433, 2453, 2434, 2216, 217, 2218, 2219,
     /*b0 */
     2220, 2221, 2222, 2223, 2224, 2215, 2200, 2231,
     2202, 2203, 2204, 2261, 2205, 2209, 2207, 2208,
     /*c0 */
     2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023,
     2024, 2015, 2000, 2031, 2002, 2003, 2004, 2061,
     /*d0 */
     2005, 2009, 2007, 2008, 2306, 2042, 2043, 2075,
     2233, 2305, 2310, 2053, 2037, 2070, 2071, 2520,
     /*e0 */
     2504, 2001, 2267, 2326, 2235, 2055, 2087, 2041,
     2262, 2441, 2523, 2442, 2507, 2443, 2508, 2475,
     /*f0 */
     ffff, 2524, 2412, 2234, 2265, 2287, 2327, 2245,
     2105, 2309, 2285, 2253, 2242, 2104, 2212, 2003}
};

/*************************************************
 * ITC Zapf Dingbats series 100 (D1) MSL mapping *
 *************************************************/
static const pl_symbol_map_t map_11L_msl = {
    map_header(11, 'L', plgv_MSL, 1, 0, 0, 0, 1, 0, 0, 0, 0, PLGV_NO_MAPPING),
    {
     /*00 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*10 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*20 */
     ffff, 2000, 2001, 2002, 2003, 2004, 2005, 2006,
     2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014,
     /*30 */
     2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022,
     2023, 2024, 2025, 2026, 2229, 2028, 2029, 2030,
     /*40 */
     2031, 2032, 2038, 2034, 2323, 2036, 2037, 2070,
     2039, 2040, 2041, 2042, 2043, 2044, 2045, 2046,
     /*50 */
     2047, 2048, 2049, 2050, 2051, 2052, 2053, 2054,
     2055, 2064, 2057, 2058, 2059, 2060, 2061, 2062,
     /*60 */
     2063, 2088, 2065, 2066, 2067, 2068, 2069, 2038,
     2071, 2072, 2073, 2074, 2075, 2076, 2077, 2078,
     /*70 */
     2079, 2080, 2081, 2082, 2083, 2046, 2085, 2086,
     2087, 2056, 2089, 2090, 2091, 2092, 2093, ffff,
     /*80 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*90 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*a0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*b0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*c0 */
     ffff, 2094, 2095, 2096, 2097, 2098, 2099, 2110,
     2101, 2425, 2426, 2104, 2105, 2106, ffff, ffff,
     /*d0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*e0 */
     ffff, 2495, 2108, 2109, 2110, 2111, 2112, 2113,
     2114, 2115, 2116, 2117, 2118, 2119, ffff, ffff,
     /*f0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff}
};

/*************************************************
 * ITC Zapf Dingbats series 200 (D2) MSL mapping *
 *************************************************/
static const pl_symbol_map_t map_12L_msl = {
    map_header(12, 'L', plgv_MSL, 1, 0, 0, 0, 1, 0, 0, 0, 0, PLGV_NO_MAPPING),
    {
     /*00 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*10 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*20 */
     ffff, 2200, 2201, 2202, 2203, 2204, 2205, 2206,
     2207, 2208, 2209, 2210, 2211, 2212, 2213, 2214,
     /*30 */
     2215, 2216, 2217, 2218, 2219, 2220, 2221, 2222,
     2223, 2224, 2225, 2226, 2239, 2228, 2229, 2230,
     /*40 */
     2231, 2232, 2233, 2234, 2235, 2236, 2237, 2238,
     2273, 2240, 2241, 2242, 2243, 2244, 2245, 2246,
     /*50 */
     2247, 2248, 2249, 2250, 2251, 2252, 2253, 2254,
     2255, 2256, 2257, 2258, 2259, 2260, 2261, 2262,
     /*60 */
     2263, 2264, 2265, 2266, 2267, 2268, 2269, 2270,
     2271, 2739, 2241, 2274, 2275, 2276, 2277, 2430,
     /*70 */
     2279, 2280, 2281, 2282, 2251, 2284, 2285, 2286,
     2287, 2288, 2289, 2290, 2291, 2292, 2293, ffff,
     /*80 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*90 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*a0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*b0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*c0 */
     ffff, 2294, 2295, 2296, 2297, 2298, 2299, 2300,
     2301, 2302, 2303, 2304, 2305, 2306, ffff, ffff,
     /*d0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*e0 */
     ffff, 2494, 2095, 2313, 2314, 2315, 2316, 2317,
     2318, 2319, 2320, 2321, 2322, 2271, ffff, ffff,
     /*f0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff}
};

/*************************************************
 * ITC Zapf Dingbats series 300 (D3) MSL mapping *
 *************************************************/
static const pl_symbol_map_t map_13L_msl = {
    map_header(13, 'L', plgv_MSL, 1, 0, 0, 0, 1, 0, 0, 0, 0, PLGV_NO_MAPPING),
    {
     /*00 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*10 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*20 */
     ffff, 2400, 2401, 2402, 2403, 2404, 2405, 2406,
     2407, 2408, 2409, 2410, 2411, 2412, 2413, 2414,
     /*30 */
     2415, 2416, 2417, 2418, 2419, 2420, 2421, 2422,
     2423, 2424, 2425, 2426, 2427, 2428, 2429, 2430,
     /*40 */
     2431, 2432, 2433, 2434, 2435, 2436, 2437, 2438,
     2439, 2440, 2441, 2442, 2443, 2444, 2445, 2446,
     /*50 */
     2477, 2488, 2499, 2450, 2451, 2452, 2453, 2454,
     2455, 2456, 2457, 2458, 2459, 2460, 2461, 2462,
     /*60 */
     2463, 2464, 2465, 2466, 2467, 2468, 2469, 2470,
     2471, 2427, 2473, 2474, 2475, 2476, 2477, 2478,
     /*70 */
     2479, 2480, 2481, 2482, 2483, 2484, 2485, 2486,
     2487, 2488, 2457, 2490, 2491, 2492, 2493, ffff,
     /*80 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*90 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*a0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*b0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*c0 */
     ffff, 2494, 2107, 2496, 2497, 2498, 2499, 2500,
     2501, 2502, 2252, 2504, 2505, 2506, ffff, ffff,
     /*d0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*e0 */
     ffff, 2510, 2511, 2512, 2513, 2514, 2515, 2516,
     2517, 2518, 2519, 2520, 2521, 2522, ffff, ffff,
     /*f0 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff}
};

/*************************************************
 * Truetype OCR-B_10 *
 *************************************************/
static const pl_symbol_map_t map_1O_unicode = {
    map_header(1, 'O', plgv_Unicode, 1, 0, 0, 0, 0, 0, 0, 0, 1,
               PLGV_NO_MAPPING),
    {
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     0xf020, 0xf021, 0xf022, 0xf023, 0xf024, 0xf025, 0xf026, 0xf027,
     0xf028, 0xf029, 0xf02a, 0xf02b, 0xf02c, 0xf02d, 0xf02e, 0xf02f,
     0xf030, 0xf031, 0xf032, 0xf033, 0xf034, 0xf035, 0xf036, 0xf037,
     0xf038, 0xf039, 0xf03a, 0xf03b, 0xf03c, 0xf03d, 0xf03e, 0xf03f,
     0xf040, 0xf041, 0xf042, 0xf043, 0xf044, 0xf045, 0xf046, 0xf047,
     0xf048, 0xf049, 0xf04a, 0xf04b, 0xf04c, 0xf04d, 0xf04e, 0xf04f,
     0xf050, 0xf051, 0xf052, 0xf053, 0xf054, 0xf055, 0xf056, 0xf057,
     0xf058, 0xf059, 0xf05a, 0xf05b, 0xf05c, 0xf05d, 0xf05e, 0xf05f,
     0xf060, 0xf061, 0xf062, 0xf063, 0xf064, 0xf065, 0xf066, 0xf067,
     0xf068, 0xf069, 0xf06a, 0xf06b, 0xf06c, 0xf06d, 0xf06e, 0xf06f,
     0xf070, 0xf071, 0xf072, 0xf073, 0xf074, 0xf075, 0xf076, 0xf077,
     0xf078, 0xf079, 0xf07a, 0xf07b, 0xf07c, 0xf07d, 0xf07e, 0xf07f,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, 0xf0a3, 0xf0a4, 0xf0a5, 0xf0a6, 0xf0a7,
     0xf0a8, 0xf0a9, 0xf0aa, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, 0xf0b2, 0xf0b3, 0xf0b4, 0xf0b5, 0xf0b6, 0xf0b7,
     0xf0b8, 0xf0b9, 0xf0ba, ffff, ffff, ffff, ffff, ffff,
     ffff, 0xf0c1, 0xf0c2, 0xf0c3, 0xf0c4, 0xf0c5, 0xf0c6, 0xf0c7,
     0xf0c8, 0xf0c9, 0xf0ca, 0xf0cb, 0xf0cc, 0xf0cd, 0xf0ce, 0xf0cf,
     0xf0d0, 0xf0d1, 0xf0d2, 0xf0d3, 0xf0d4, 0xf0d5, 0xf0d6, 0xf0d7,
     0xf0d8, 0xf0d9, 0xf0da, 0xf0db, 0xf0dc, 0xf0dd, 0xf0de, 0xf0df,
     ffff, ffff, ffff, ffff, ffff, 0xf0e5, 0xf0e6, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     0xf0f8, ffff, ffff, ffff, ffff, ffff, ffff, ffff}
};
/*************************************************
 * Windows Baltic
 *************************************************/
static const pl_symbol_map_t map_19L_unicode = {
    map_header(19, 'L', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0x00, 0, 0,
               PLGV_U2M_MAPPING),
    {
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005A, 0x005B, 0x005C, 0x005D, 0x005E, 0x005F,
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007A, 0x007B, 0x007C, 0x007D, 0x007E, 0x2592,
     0x20AC, 0xffff, 0x201A, 0xffff, 0x201E, 0x2026, 0x2020, 0x2021,
     0xffff, 0x2030, 0xffff, 0x2039, 0xffff, 0xEFF6, 0xEFF4, 0x00B8,
     0xffff, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
     0xffff, 0x2122, 0xffff, 0x203A, 0xffff, 0xEFEF, 0x02DB, 0xffff,
     0xffff, 0xffff, 0x00A2, 0x00A3, 0x00A4, 0xffff, 0x00A6, 0x00A7,
     0x00D8, 0x00A9, 0x0156, 0x00AB, 0x00AC, 0x002D, 0x00AE, 0x00C6,
     0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x2219,
     0x00F8, 0x00B9, 0x0157, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00E6,
     0x0104, 0x012E, 0x0100, 0x0106, 0x00C4, 0x00C5, 0x0118, 0x0112,
     0x010C, 0x00C9, 0x0179, 0x0116, 0x0122, 0x0136, 0x012A, 0x013B,
     0x0160, 0x0143, 0x0145, 0x00D3, 0x014C, 0x00D5, 0x00D6, 0x00D7,
     0x0172, 0x0141, 0x015A, 0x016A, 0x00DC, 0x017B, 0x017D, 0x00DF,
     0x0105, 0x012F, 0x0101, 0x0107, 0x00E4, 0x00E5, 0x0119, 0x0113,
     0x010D, 0x00E9, 0x017A, 0x0117, 0x0123, 0x0137, 0x012B, 0x013C,
     0x0161, 0x0144, 0x0146, 0x00F3, 0x014D, 0x00F5, 0x00F6, 0x00F7,
     0x0173, 0x0142, 0x015B, 0x016B, 0x00FC, 0x017C, 0x017E, 0x02D9}
};
/*************************************************
 * Ventura International
 *************************************************/
static const pl_symbol_map_t map_13J_unicode = {
    map_header(13, 'J', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0x80, 0, 0,
               PLGV_U2M_MAPPING),
    {
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005A, 0x005B, 0x005C, 0x005D, 0x005E, 0x005F,
     0x2018, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007A, 0x007B, 0x007C, 0x007D, 0x007E, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0x201E, 0x00C0, 0x00C2, 0x00C8, 0x00CA, 0x00CB, 0x00CE, 0x00CF,
     0x00A9, 0x00AE, 0x2122, 0x2039, 0x203A, 0x00D9, 0x00DB, 0xffff,
     0x2030, 0x201C, 0x201D, 0x00B0, 0x00C7, 0x00E7, 0x00D1, 0x00F1,
     0x00A1, 0x00BF, 0x00A4, 0x00A3, 0x00A5, 0x00A7, 0x0192, 0x00A2,
     0x00E2, 0x00EA, 0x00F4, 0x00FB, 0x00E1, 0x00E9, 0x00F3, 0x00FA,
     0x00E0, 0x00E8, 0x00F2, 0x00F9, 0x00E4, 0x00EB, 0x00F6, 0x00FC,
     0x00C5, 0x00EE, 0x00D8, 0x00C6, 0x00E5, 0x00ED, 0x00F8, 0x00E6,
     0x00C4, 0x00EC, 0x00D6, 0x00DC, 0x00C9, 0x00EF, 0x00DF, 0x00D4,
     0x00C1, 0x00C3, 0x00E3, 0xffff, 0xffff, 0x00CD, 0x00CC, 0x00D3,
     0x00D2, 0x00D5, 0x00F5, 0x0160, 0x0161, 0x00DA, 0x0178, 0x00FF,
     0x0152, 0x0153, 0x00B6, 0x2020, 0x2021, 0x2014, 0x2013, 0xffff,
     0xffff, 0x00AA, 0x00BA, 0x00AB, 0x2022, 0x00BB, 0xffff, 0x2026}
};
/*************************************************
 * Ventura US
 *************************************************/
static const pl_symbol_map_t map_14J_unicode = {
    map_header(14, 'J', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0x80, 0, 0,
               PLGV_U2M_MAPPING),
    {
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005A, 0x005B, 0x005C, 0x005D, 0x005E, 0x005F,
     0x2018, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007A, 0x007B, 0x007C, 0x007D, 0x007E, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0x201E, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0x00A9, 0x00AE, 0x2122, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0x2030, 0x201C, 0x201D, 0x00B0, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x00A7, 0xffff, 0x00A2,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0x00B6, 0x2020, 0x2021, 0x2014, 0x2013, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0x2022, 0xffff, 0xffff, 0x2026}
};
/*************************************************
 * Ventura US
 *************************************************/
static const pl_symbol_map_t map_6M_unicode = {
    map_header(6, 'M', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0x80, 0, 0,
               PLGV_U2M_MAPPING),
    {
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0x0021, 0x2200, 0x0023, 0x2203, 0x0025, 0x0026, 0x220D,
     0x0028, 0x0029, 0x2217, 0x002B, 0x002C, 0x2212, 0x002E, 0x002F,
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
     0x2245, 0x0041, 0x0042, 0x0058, 0x2206, 0x0045, 0x03A6, 0x0393,
     0x0048, 0x0049, 0x03D1, 0x004B, 0x039B, 0x004D, 0x004E, 0x004F,
     0x03A0, 0x0398, 0x0050, 0x03A3, 0x0054, 0x0059, 0x03C2, 0x03A9,
     0x039E, 0x03A8, 0x005A, 0x005B, 0x2234, 0x005D, 0x22A5, 0xEFEB,
     0xEFEA, 0x03B1, 0x03B2, 0x03C7, 0x03B4, 0xEFEC, 0x03C6, 0x03B3,
     0x03B7, 0x03B9, 0x03D5, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BF,
     0x03C0, 0x03B8, 0x03C1, 0x03C3, 0x03C4, 0x03C5, 0x03D6, 0x03C9,
     0x03BE, 0x03C8, 0x03B6, 0x007B, 0x007C, 0x007D, 0x007E, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0x25C7, 0xEFBF, 0xEFE1, 0xEFE7, 0x2223, 0xEFE6, 0xEFE1, 0xEFDD,
     0x21D1, 0x21D2, 0x21D3, 0x2284, 0x2282, 0x2321, 0xEFDD, 0xEFDF,
     0xEFD8, 0x2283, 0x2287, 0xEFE3, 0xffff, 0x2663, 0x2295, 0x2297,
     0x2286, 0x222A, 0xEFE8, 0x2026, 0x2223, 0x2227, 0x21B5, 0x2248,
     0x2264, 0x2666, 0x2265, 0x2202, 0x2135, 0x2032, 0x211C, 0x2118,
     0x221E, 0x2660, 0x221D, 0x2022, 0x2215, 0x2665, 0x00D7, 0x03A5,
     0x2193, 0x2190, 0xEFDB, 0x2033, 0x0192, 0x2111, 0xEFDA, 0x00B1,
     0x2192, 0x2191, 0x2260, 0x2261, 0x00B0, 0x2194, 0xEFE5, 0x222B,
     0xEFDD, 0x2219, 0x2220, 0xEFDE, 0xffff, 0xEFE2, 0xEFE3, 0x232A,
     0xffff, 0x00AC, 0x2207, 0x2320, 0xEFDD, 0xEFE0, 0xEFDE, 0x00F7,
     0xEFC2, 0xEFD9, 0x21D0, 0x21D4, 0x2228, 0x2211, 0xEFD6, 0x2223,
     0xEFE4, 0x2205, 0x2229, 0x2208, 0xEFD7, 0x2209, 0xEFE0, 0x2329}
};
/***************************************
 * PC-858 Multilingual Unicode mapping *
 ***************************************/
static const pl_symbol_map_t map_13U_unicode = {
    map_header(13, 'U', plgv_Unicode, 2, 0, 0, 0, 0, 0xe6, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0x263a, 0x263b, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
     0x25d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x266a, 0x266b, 0x263c,
     /* 0x10 */
     0x25ba, 0x25c4, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8,
     0x2191, 0x2193, 0x2192, 0x2190, 0x221f, 0x2194, 0x25b2, 0x25bc,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /* 0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2302,
     /* 0x80 */
     0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
     0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
     /* 0x90 */
     0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
     0x00ff, 0x00d6, 0x00dc, 0x00f8, 0x00a3, 0x00d8, 0x00d7, 0x0192,
     /* 0xa0 */
     0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
     0x00bf, 0x00ae, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb,
     /*0xb0 */
     0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1, 0x00c2, 0x00c0,
     0x00a9, 0x2563, 0x2551, 0x2557, 0x255d, 0x00a2, 0x00a5, 0x2510,
     /*0xc0 */
     0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x00e3, 0x00c3,
     0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4,
     /*0xd0 */
     0x00f0, 0x00d0, 0x00ca, 0x00cb, 0x00c8, 0x20ac, 0x00cd, 0x00ce,
     0x00cf, 0x2518, 0x250c, 0x2588, 0x2584, 0x00a6, 0x00cc, 0x2580,
     /*0xe0 */
     0x00d3, 0x00df, 0x00d4, 0x00d2, 0x00f5, 0x00d5, 0x00b5, 0x00fe,
     0x00de, 0x00da, 0x00db, 0x00d9, 0x00fd, 0x00dd, 0x02c9, 0x00b4,
     /*0xf0 */
     0x002d, 0x00b1, 0x2017, 0x00be, 0x00b6, 0x00a7, 0x00f7, 0x00b8,
     0x00b0, 0x00a8, 0x00b7, 0x00b9, 0x00b3, 0x00b2, 0x25aa, 0x00a0}
};
/*******************************************
 * ISO 8859/10 Latin 6 (6N) Unicode mapping *
 *******************************************/
static const pl_symbol_map_t map_6N_unicode = {
    map_header(6, 'N', plgv_Unicode, 1, 0, 0, 0, 0, 0xd0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2592,
     /*0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0xffff, 0x0104, 0x0112, 0x0122, 0x012A, 0x0128, 0x0136, 0x00A7,
     0x013B, 0x00D0, 0x0160, 0x0166, 0x017D, 0x002D, 0x016A, 0x014A,
     /*0xb0 */
     0x00B0, 0x0105, 0x0113, 0x0123, 0x012B, 0x0129, 0x0137, 0x00B7,
     0x013C, 0x0111, 0x0161, 0x0167, 0x017E, 0x2014, 0x016B, 0x014B,
     /*0xc0 */
     0x0100, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x012E,
     0x010C, 0x00C9, 0x0118, 0x00CB, 0x0116, 0x00CD, 0x00CE, 0x00CF,
     /*0xd0 */
     0x00D0, 0x0145, 0x014C, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x0168,
     0x00D8, 0x0172, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF,
     /*0xe0 */
     0x0101, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x012F,
     0x010D, 0x00E9, 0x0119, 0x00EB, 0x0117, 0x00ED, 0x00EE, 0x00EF,
     /*0xf0 */
     0x00F0, 0x0146, 0x014D, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x0169,
     0x00F8, 0x0173, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x0138}
};
/*******************************************
 * ISO 8859/15 Latin 9 Unicode mapping *
 *******************************************/
static const pl_symbol_map_t map_9N_unicode = {
    map_header(9, 'N', plgv_Unicode, 1, 0, 0, 0, 0, 0xd0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*0x00 */
     0x0000, 0x0002, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
     0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
     /*0x10 */
     0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
     0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
     /*0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /*0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /*0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /*0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x02c6, 0x005f,
     /*0x60 */
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /*0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x02dc, 0x2592,
     /*0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /*0xa0 */
     0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x20ac, 0x00a5, 0x0160, 0x00a7,
     0x0161, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x002d, 0x00ae, 0x02c9,
     /*0xb0 */
     0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x017d, 0x00b5, 0x00b6, 0x2219,
     0x017e, 0x00b9, 0x00ba, 0x00bb, 0x0152, 0x0153, 0x0178, 0x00bf,
     /*0xc0 */
     0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
     0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
     /*0xd0 */
     0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
     0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
     /*0xe0 */
     0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
     0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
     /*0xf0 */
     0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
     0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff}
};
/***************************
 * Roman-9 Unicode mapping *
 ***************************/
const pl_symbol_map_t map_4U_unicode = {
    map_header(4, 'U', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
     /* 0x30 */
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
     /* 0x40 */
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
     /* 0x50 */
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
     /* 0x60 */
     0x2018, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
     /* 0x70 */
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2592,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0x00a0, 0x00c0, 0x00c2, 0x00c8, 0x00ca, 0x00cb, 0x00ce, 0x00cf,
     0x00b4, 0x0060, 0x02c6, 0x00a8, 0x02dc, 0x00d9, 0x00db, 0x00a3,
     /* 0xb0 */
     0x00af, 0x00dd, 0x00fd, 0x00b0, 0x00c7, 0x00e7, 0x00d1, 0x00f1,
     0x00a1, 0x00bf, 0x20ac, 0x00a3, 0x00a5, 0x00a7, 0x0192, 0x00a2,
     /* 0xc0 */
     0x00e2, 0x00ea, 0x00f4, 0x00fb, 0x00e1, 0x00e9, 0x00f3, 0x00fa,
     0x00e0, 0x00e8, 0x00f2, 0x00f9, 0x00e4, 0x00eb, 0x00f6, 0x00fc,
     /* 0xd0 */
     0x00c5, 0x00ee, 0x00d8, 0x00c6, 0x00e5, 0x00ed, 0x00f8, 0x00e6,
     0x00c4, 0x00ec, 0x00d6, 0x00dc, 0x00c9, 0x00ef, 0x00df, 0x00d4,
     /* 0xe0 */
     0x00c1, 0x00c3, 0x00e3, 0x00d0, 0x00f0, 0x00cd, 0x00cc, 0x00d3,
     0x00d2, 0x00d5, 0x00f5, 0x0160, 0x0161, 0x00da, 0x0178, 0x00ff,
     /* 0xf0 */
     0x00de, 0x00fe, 0x2219, 0x00b5, 0x00b6, 0x00be, 0x2212, 0x00bc,
     0x00bd, 0x00aa, 0x00ba, 0x00ab, 0x25a0, 0x00bb, 0x00b1, 0xffff}
};
/***************************
 * Roman-9 Unicode mapping *
 ***************************/
const pl_symbol_map_t map_26U_unicode = {
    map_header(26, 'U', plgv_Unicode, 1, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     0xffff, 0x263A, 0x263B, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
     0x25D8, 0x25CB, 0x25D9, 0x2642, 0x2640, 0x266A, 0x266B, 0x263C,
     0x25BA, 0x25C4, 0x2195, 0x203C, 0x00B6, 0x00A7, 0x25AC, 0x21A8,
     0x2191, 0x2193, 0x2192, 0x2190, 0x221F, 0x2194, 0x25B2, 0x25BC,
     0xffff, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x2019,
     0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
     0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
     0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
     0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
     0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
     0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
     0x0058, 0x0059, 0x005A, 0x005B, 0x005C, 0x005D, 0x02C6, 0x005F,
     0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
     0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
     0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
     0x0078, 0x0079, 0x007A, 0x007B, 0x007C, 0x007D, 0x02DC, 0x2302,
     0x0106, 0x00FC, 0x00E9, 0x0101, 0x00E4, 0x0123, 0x00E5, 0x0107,
     0x0142, 0x0113, 0x0156, 0x0157, 0x012B, 0x0179, 0x00C4, 0x00C5,
     0x00C9, 0x00E6, 0x00C6, 0x014D, 0x00F6, 0x0122, 0x00A2, 0x015A,
     0x015B, 0x00D6, 0x00DC, 0x00F8, 0x00A3, 0x00D8, 0x00D7, 0x00A4,
     0x0100, 0x012A, 0x00F3, 0x017B, 0x017C, 0x017A, 0x201D, 0x00A6,
     0x00A9, 0x00AE, 0x00AC, 0x00BD, 0x00BC, 0x0141, 0x00AB, 0x00BB,
     0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x0104, 0x010C, 0x0118,
     0x0116, 0x2563, 0x2551, 0x2557, 0x255D, 0x012E, 0x0160, 0x2510,
     0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x0172, 0x016A,
     0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x017D,
     0x0105, 0x010D, 0x0119, 0x0117, 0x012F, 0x0161, 0x0173, 0x016B,
     0x017E, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580,
     0x00D3, 0x00DF, 0x014C, 0x0143, 0x00F5, 0x00D5, 0x00B5, 0x0144,
     0x0136, 0x0137, 0x013B, 0x013C, 0x0146, 0x0112, 0x0145, 0x2019,
     0x00AD, 0x00B1, 0x201C, 0x00BE, 0x00B6, 0x00A7, 0x00F7, 0x201E,
     0x00B0, 0x2219, 0x00B7, 0x00B9, 0x00B3, 0x00B2, 0x25AA, 0xffff}
};

/*****************************************
 * ITC Zapf Dingbats MS (DZ) Unicode mapping *
 *****************************************/
static const pl_symbol_map_t map_14L_unicode = {
    map_header(14, 'L', plgv_Unicode, 2, 0, 0, 0, 1, 0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /*00 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*10 */
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     ffff, ffff, ffff, ffff, ffff, ffff, ffff, ffff,
     /*20 */
     0x0020, 0xf021, 0xf022, 0xf023, 0xf024, 0xf025, 0xf026, 0xf027,
     0xf028, 0xf029, 0xf02a, 0xf02b, 0xf02c, 0xf02d, 0xf02e, 0xf02f,
     /*30 */
     0xf030, 0xf031, 0xf032, 0xf033, 0xf034, 0xf035, 0xf036, 0xf037,
     0xf038, 0xf039, 0xf03a, 0xf03b, 0xf03c, 0xf03d, 0xf03e, 0xf03f,
     /*40 */
     0xf040, 0xf041, 0xf042, 0xf043, 0xf044, 0xf045, 0xf046, 0xf047,
     0xf048, 0xf049, 0xf04a, 0xf04b, 0xf04c, 0xf04d, 0xf04e, 0xf04f,
     /*50 */
     0xf050, 0xf051, 0xf052, 0xf053, 0xf054, 0xf055, 0xf056, 0xf057,
     0xf058, 0xf059, 0xf05a, 0xf05b, 0xf05c, 0xf05d, 0xf05e, 0xf05f,
     /*60 */
     0xf060, 0xf061, 0xf062, 0xf063, 0xf064, 0xf065, 0xf066, 0xf067,
     0xf068, 0xf069, 0xf06a, 0xf06b, 0xf06c, 0xf06d, 0xf06e, 0xf06f,
     /*70 */
     0xf070, 0xf071, 0xf072, 0xf073, 0xf074, 0xf075, 0xf076, 0xf077,
     0xf078, 0xf079, 0xf07a, 0xf07b, 0xf07c, 0xf07d, 0xf07e, 0xf07f,
     /*80 */
     0xf080, 0xf081, 0xf082, 0xf083, 0xf084, 0xf085, 0xf086, 0xf087,
     0xf088, 0xf089, 0xf08a, 0xf08b, 0xf08c, 0xf08d, 0xf08e, 0xf08f,
     /*90 */
     0xf090, 0xf091, 0xf092, 0xf093, 0xf094, 0xf095, 0xf096, 0xf097,
     0xf098, 0xf099, 0xf09a, 0xf09b, 0xf09c, 0xf09d, 0xf09e, 0xf09f,
     /*a0 */
     0xf0a0, 0xf0a1, 0xf0a2, 0xf0a3, 0xf0a4, 0xf0a5, 0xf0a6, 0xf0a7,
     0xf0a8, 0xf0a9, 0xf0aa, 0xf0ab, 0xf0ac, 0xf0ad, 0xf0ae, 0xf0af,
     /*b0 */
     0xf0b0, 0xf0b1, 0xf0b2, 0xf0b3, 0xf0b4, 0xf0b5, 0xf0b6, 0xf0b7,
     0xf0b8, 0xf0b9, 0xf0ba, 0xf0bb, 0xf0bc, 0xf0bd, 0xf0be, 0xf0bf,
     /*c0 */
     0xf0c0, 0xf0c1, 0xf0c2, 0xf0c3, 0xf0c4, 0xf0c5, 0xf0c6, 0xf0c7,
     0xf0c8, 0xf0c9, 0xf0ca, 0xf0cb, 0xf0cc, 0xf0cd, 0xf0ce, 0xf0cf,
     /*d0 */
     0xf0d0, 0xf0d1, 0xf0d2, 0xf0d3, 0xf0d4, 0xf0d5, 0xf0d6, 0xf0d7,
     0xf0d8, 0xf0d9, 0xf0da, 0xf0db, 0xf0dc, 0xf0dd, 0xf0de, 0xf0df,
     /*e0 */
     0xf0e0, 0xf0e1, 0xf0e2, 0xf0e3, 0xf0e4, 0xf0e5, 0xf0e6, 0xf0e7,
     0xf0e8, 0xf0e9, 0xf0ea, 0xf0eb, 0xf0ec, 0xf0ed, 0xf0ee, 0xf0ef,
     /*f0 */
     0xf0f0, 0xf0f1, 0xf0f2, 0xf0f3, 0xf0f4, 0xf0f5, 0xf0f6, 0xf0f7,
     0xf0f8, 0xf0f9, 0xf0fa, 0xf0fb, 0xf0fc, 0xf0fd, 0xf0fe, 0xf0ff,

     }
};

/* ISO/IEC 8859-7 Latin/Greek alphabet */
static const pl_symbol_map_t map_12N_unicode = {
    map_header(12, 'N', plgv_Unicode, 2, 0, 0, 0, 0, 0xc0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
        /* 0x00 */
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        /* 0x10 */
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        /* 0x20 */
        0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
        0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
        /* 0x30 */
        0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
        0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
        /* 0x40 */
        0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
        0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
        /* 0x50 */
        0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
        0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
        /* 0x60 */
        0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
        0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
        /* 0x70 */
        0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
        0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x007f,
        /* 0x80 */
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        /* 0x90 */
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
        /* 0xa0 */
        0x00a0, 0x02bd, 0x02bc, 0x00a3, 0xffff, 0xffff, 0x00a6, 0x00a7,
        0x00a8, 0x00a9, 0xffff, 0x00ab, 0x00ac, 0x00ad, 0xffff, 0x2015,
        /* 0xb0 */
        0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x0384, 0x0385, 0x0386, 0x00b7,
        0x0388, 0x0389, 0x038a, 0x00bb, 0x038c, 0x00bd, 0x038e, 0x038f,
        /* 0xc0 */
        0x0390, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397,
        0x0398, 0x0399, 0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f,
        /* 0xd0 */
        0x03a0, 0x03a1, 0xffff, 0x03a3, 0x03a4, 0x03a5, 0x03a6, 0x03a7,
        0x03a8, 0x03a9, 0x03aa, 0x03ab, 0x03ac, 0x03ad, 0x03ae, 0x03af,
        /* 0xe0 */
        0x03b0, 0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x03b7,
        0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf,
        /* 0xf0 */
        0x03c0, 0x03c1, 0x03c2, 0x03c3, 0x03c4, 0x03c5, 0x03c6, 0x03c7,
        0x03c8, 0x03c9, 0x03ca, 0x03cb, 0x03cc, 0x03cd, 0x03ce, 0xffff
    }
};
    
/*************************************************************
 * HP-GL/2 stick font extended character set Unicode mapping *
 * (Note: this symbol set is not documented anywhere.)       *
 * Strange mapping is effectively: roman8(charcode - 0x80)   *
 *************************************************************/
static const pl_symbol_map_t map_0E_unicode = {
    map_header(0, 'E', plgv_Unicode, 0, 0, 0, 0, 0, 0x40, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {
     /* 0x00 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x10 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x20 */
     0x00a0, 0x00c0, 0x00c2, 0x00c8, 0x00ca, 0x00cb, 0x00ce, 0x00cf,
     0x00b4, 0x0060, 0x02c6, 0x00a8, 0x02dc, 0x00d9, 0x00db, 0x00a3,
     /* 0x30 */
     0x00af, 0x00dd, 0x00fd, 0x00b0, 0x00c7, 0x00e7, 0x00d1, 0x00f1,
     0x00a1, 0x00bf, 0x00a4, 0x00a3, 0x00a5, 0x00a7, 0x0192, 0x00a2,
     /* 0x40 */
     0x00e2, 0x00ea, 0x00f4, 0x00fb, 0x00e1, 0x00e9, 0x00f3, 0x00fa,
     0x00e0, 0x00e8, 0x00f2, 0x00f9, 0x00e4, 0x00eb, 0x00f6, 0x00fc,
     /* 0x50 */
     0x00c5, 0x00ee, 0x00d8, 0x00c6, 0x00e5, 0x00ed, 0x00f8, 0x00e6,
     0x00c4, 0x00ec, 0x00d6, 0x00dc, 0x00c9, 0x00ef, 0x00df, 0x00d4,
     /* 0x60 */
     0x00c1, 0x00c3, 0x00e3, 0x0110, 0x00f0, 0x00cd, 0x00cc, 0x00d3,
     0x00d2, 0x00d5, 0x00f5, 0x0160, 0x0161, 0x00da, 0x0178, 0x00ff,
     /* 0x70 */
     0x00de, 0x00fe, 0x00b7, 0x00b5, 0x00b6, 0x00be, 0x2212, 0x00bc,
     0x00bd, 0x00aa, 0x00ba, 0x00ab, 0x25a0, 0x00bb, 0x00b1, 0xffff,
     /* 0x80 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0x90 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xa0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xb0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xc0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xd0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xe0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     /* 0xf0 */
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
     0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff}
};

static const pl_symbol_map_t map_18N_unicode = {
    map_header(18, 'N', plgv_Unicode, 2, 0, 0, 0, 0, 0, 0, 0, 0,
               PLGV_U2M_MAPPING),
    {0}
};

/*
 * Define the list of built-in symbol maps.
 */
const pl_symbol_map_t *pl_built_in_symbol_maps[] = {
    &map_0U_unicode,
    &map_1E_unicode,
    &map_1F_unicode,
    &map_1G_unicode,
    &map_0I_unicode,
    &map_0S_unicode,
    &map_2S_unicode,
    &map_0D_unicode,
    &map_8U_unicode,
    &map_7J_unicode,
    &map_19U_unicode,
    &map_0N_unicode,
    &map_10U_unicode,
    &map_12U_unicode,
    &map_10J_unicode,
    &map_12J_unicode,
    &map_11U_unicode,
    &map_1U_unicode,
    &map_9U_unicode,
    &map_6J_unicode,
    &map_17U_unicode,
    &map_9T_unicode,
    &map_2N_unicode,
    &map_9E_unicode,
    &map_5N_unicode,
    &map_5T_unicode,
    &map_8M_unicode,
    &map_15U_unicode,
    &map_5M_unicode,
    &map_19M_unicode,
    &map_579L_unicode,
    &map_10L_msl,
    &map_11L_msl,
    &map_12L_msl,
    &map_13L_msl,
    &map_0E_unicode,
    &map_9J_unicode,
    &map_1O_unicode,
    &map_13J_unicode,
    &map_14J_unicode,
    &map_6M_unicode,
    &map_13U_unicode,
    &map_19L_unicode,
    &map_6N_unicode,
    &map_9N_unicode,
    &map_4U_unicode,
    &map_26U_unicode,
    &map_14L_unicode,
    &map_9R_unicode,
    &map_9G_unicode,
    &map_9V_unicode,
    &map_18N_unicode,
    &map_20U_unicode,
    &map_25U_unicode,
    &map_12N_unicode,
    0                           /* end marker */
};

const int pl_built_in_symbol_map_count = countof(pl_built_in_symbol_maps) - 1;

ulong
pl_map_symbol(const pl_symbol_map_t * psm,
              uint chr, bool is_resident_font, bool is_MSL,
              const gs_memory_t * mem)
{
    uint first_code, last_code, code;

    /*
     * If there is no symbol map we assume the the character
     * implicitly indexes the font.  The symbol set 590 is not
     * documented it appears to do implicit mapping as well.
     *
     * symbol set 590/18N is for UTF-8 according to
     * http://docs.hp.com/en/5991-7956/5991-7956.pdf
     *
     */

    if ((psm == 0) || (pl_get_uint16(psm->id) == 590))
        return chr;


    first_code = pl_get_uint16(psm->first_code);
    last_code = pl_get_uint16(psm->last_code);
    if ((chr < first_code) || (chr > last_code))
        code = ((last_code <= 0xff) && (chr > 0xff) ? chr : 0xffff);
    else
        code = psm->codes[chr - first_code];
    /* we do not provide many of the symbol sets in MSL format.
       Instead we may use a corresponding unicode symbol set and
       convert the resulting unicode value to msl.  This will not
       necessarily match HP. */

    /* NB refactor me. */
    if (is_MSL && (code != 0xffff) &&
        (pl_symbol_map_vocabulary(psm) == plgv_Unicode)) {
#ifdef DEBUG
        if (gs_debug_c('=')) {
            dmprintf3(mem,
                      "[=] unicode to msl conversion: chr=%d, unicode=0x%x, msl code=%d\n",
                      chr, code, pl_map_Unicode_to_MSL(code,
                                                       (psm->id[0] << 8) +
                                                       psm->id[1]));
        }
#endif
        code = pl_map_Unicode_to_MSL(code, (psm->id[0] << 8) + psm->id[1]);
    } else if (!is_MSL && (code != 0xffff) &&
               (pl_symbol_map_vocabulary(psm) == plgv_MSL)) {
#ifdef DEBUG
        if (gs_debug_c('=')) {
            dmprintf3(mem,
                      "[=] msl to unicode conversion: chr=%d, msl=0x%x, unicode=%d\n",
                      chr, code, pl_map_MSL_to_Unicode(code,
                                                       (psm->id[0] << 8) +
                                                       psm->id[1]));
        }
#endif
        code = pl_map_MSL_to_Unicode(code, (psm->id[0] << 8) + psm->id[1]);
    }
    return code;
}
