// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/TriggerCDFRun0Run1.hh"

namespace Rivet {


  /// @brief CDF pseudorapidity analysis at 630 and 1800 GeV
  ///
  /// @author Andy Buckley
  class CDF_1990_I283352 : public Analysis {
  public:

    RIVET_DEFAULT_ANALYSIS_CTOR(CDF_1990_I283352);


    /// @name Analysis methods
    /// @{

    void init() {
      // Setup projections
      declare(TriggerCDFRun0Run1(), "Trigger");
      declare(ChargedFinalState(Cuts::abseta < 3.5), "CFS");

      // Book histo
      for (double eVal : allowedEnergies()) {
        size_t ih = eVal < 1000;
        if (isCompatibleWithSqrtS(eVal))  _sqs = ih;
        book(_h[ih], 3+ih, 1, 1);
        book(_c[ih], "TMP/sumWTrig_"+toString(ih));
      }
      raiseBeamErrorIf(_sqs > 1);
    }


    /// Do the analysis
    void analyze(const Event& event) {
      // Trigger
      const bool trigger = apply<TriggerCDFRun0Run1>(event, "Trigger").minBiasDecision();
      if (!trigger) vetoEvent;
      _c[_sqs]->fill();

      // Loop over final state charged particles to fill eta histos
      const FinalState& fs = apply<FinalState>(event, "CFS");
      for (const Particle& p : fs.particles()) {
        _h[_sqs]->fill(p.abseta());
      }
    }


    /// Finalize
    void finalize() {
      // Divide through by num events to get d<N>/d(eta) in bins
      // Factor of 1/2 for |eta| -> eta
      scale(_c, crossSectionPerEvent());
      scale(_h, crossSectionPerEvent());
      for (size_t ih=0; ih<2; ++ih) {
        if (_c[ih]->sumW()) scale(_h[ih], 0.5/ *_c[ih]);
      }
    }

    /// @}


  private:

    /// Counter
    CounterPtr _c[2];

    /// Histogram
    Histo1DPtr _h[2];

    size_t _sqs = 2;

  };



  RIVET_DECLARE_ALIASED_PLUGIN(CDF_1990_I283352, CDF_1990_S2089246);

}
