// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief R for energies between 3.1 and 4.8 GeV
  class PLUTO_1977_I110272 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(PLUTO_1977_I110272);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");

      // Book histograms
      const auto& ref = refData<YODA::BinnedEstimate<string>>(2,1,1);
      book(_c_hadrons, "/TMP/sigma_hadrons", ref);
      book(_c_muons,   "/TMP/sigma_muons"  , ref);
      for (const string& en : _c_hadrons.binning().edges<0>()) {
        double eval = stod(en);
        if (isCompatibleWithSqrtS(eval)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      if (nCount[-13]==1 and nCount[13]==1 && ntotal==2+nCount[22]) { // mu+mu- + photons
        _c_muons->fill(_sqs);
      }
      else { // everything else
        _c_hadrons->fill(_sqs);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      BinnedEstimatePtr<string> mult;
      book(mult, 2, 1, 1);
      divide(_c_hadrons,_c_muons,mult);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _c_hadrons, _c_muons;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(PLUTO_1977_I110272);

}
