// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/Thrust.hh"
#include "Rivet/Projections/Hemispheres.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// @brief jet masses for 14, 22, 34.8 and 43,5 GeV
  class TASSO_1989_I279165 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1989_I279165);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      const ChargedFinalState cfs;
      declare(cfs, "CFS");
      declare(Hemispheres(Thrust(cfs)), "Hemispheres");

      size_t ih = 0;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"diff"],  1, 1, ++ih);
        book(_h[en+"heavy"], 2, 1,   ih);
        book(_h[en+"light"], 3, 1,   ih);
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const ChargedFinalState& cfs = apply<ChargedFinalState>(event, "CFS");

      if (cfs.particles().size() < 3 ) vetoEvent;

      const Hemispheres& hemi = apply<Hemispheres>(event, "Hemispheres");
      _h[_sqs+"heavy"]->fill(hemi.scaledM2high());
      _h[_sqs+"light"]->fill(hemi.scaledM2low() );
      _h[_sqs+"diff"]->fill(hemi.scaledM2diff());
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      normalize(_h);
    }
    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1989_I279165);
}
