/*
 * PropertiesHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 2000-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): suhler.
 *
 * Version:  2.2
 * Created by suhler on 00/08/28
 * Last modified by suhler on 04/08/30 09:02:24
 */

package sunlabs.brazil.handler;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Properties;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.Glob;

/**
 * Handler for returning selected request properties as a text/plain document
 * in java properties format. 
 * A server using this handler may be called
 * by a server using the {@link ProxyPropertiesHandler}
 * to communicate per-request properties between the two servers.
 * <p>
 * Properties:
 * <dl class=props>
 * <dt>prefix, suffix, glob, match
 * <dd>Specify the URL that triggers this handler.
 * (See {@link MatchString}).
 * <dt>select	<dd>Glob pattern to match properties selected
 *		(Defaults to *).  This is re-examined at every request.
 * <dt>type	<dd>Type of output to generate (defaults to text/plain).
 * <dt>comment	<dd>Comment to put on output (defaults to <i>select</i>).
 * </dl>
 *
 * @author      Stephen Uhler
 * @version	@(#)PropertiesHandler.java	2.2
 */

public class PropertiesHandler implements Handler {
    String prefix;		// our handler name
    String select;		// glob pattern to select props
    String type;
    MatchString isMine;            // check for matching url

    public boolean
    init(Server server, String prefix) {
	this.prefix = prefix;
	isMine = new MatchString(prefix, server.props);
	type = server.props.getProperty(prefix + "type", "text/plain");
	return true;
    }

    /**
     * If this is one of our URL's, look through each
     * request property, and selct those that match the <i>Select</i>
     * property.  Then emit them all as text/plain.
     */

    public boolean
    respond(Request request) throws IOException {
	if (!isMine.match(request.url)) {
	    return false;
	}
	String select = request.props.getProperty(prefix + "select", "*");

	Properties p = new Properties();
	Enumeration enumer = request.props.propertyNames();
	while(enumer.hasMoreElements()) {
	    String key = (String) enumer.nextElement();
	    if (Glob.match(select, key)) {
		p.put(key, request.props.getProperty(key));
	    }
	}
	String comment = request.props.getProperty(prefix + "comment");
	if (comment == null) {
	    comment = prefix + this.getClass() + " selecting: " + select;
	}
	ByteArrayOutputStream out = new ByteArrayOutputStream();
	p.save(out,comment);
	request.sendResponse(out.toByteArray(), type);
	return true;
    }
}
