/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free
  Software Foundation; either version 2.1, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
  PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License along with
  TALER; see the file COPYING.LGPL.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant_api_patch_unit.c
 * @brief Implementation of PATCH /private/units/$ID
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include <curl/curl.h>
#include <jansson.h>
#include <microhttpd.h>
#include <gnunet/gnunet_util_lib.h>
#include "taler_merchant_service.h"
#include "merchant_api_curl_defaults.h"
#include "merchant_api_common.h"
#include <taler/taler_json_lib.h>
#include <taler/taler_curl_lib.h>


/**
 * Handle for a PATCH /private/units/$ID operation.
 */
struct TALER_MERCHANT_UnitPatchHandle
{
  /**
   * Fully qualified request URL.
   */
  char *url;

  /**
   * In-flight CURL job.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Completion callback.
   */
  TALER_MERCHANT_UnitPatchCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Execution context.
   */
  struct GNUNET_CURL_Context *ctx;

  /**
   * Keeps POST body and headers alive.
   */
  struct TALER_CURL_PostContext post_ctx;
};


/**
 * Called when the HTTP transfer finishes.
 *
 * @param cls operation handle
 * @param response_code HTTP status (0 on failure)
 * @param response parsed JSON reply (NULL if unavailable)
 */
static void
handle_patch_unit_finished (void *cls,
                            long response_code,
                            const void *response)
{
  struct TALER_MERCHANT_UnitPatchHandle *uph = cls;
  const json_t *json = response;
  struct TALER_MERCHANT_HttpResponse hr = {
    .http_status = (unsigned int) response_code,
    .reply = json
  };

  uph->job = NULL;
  GNUNET_log (GNUNET_ERROR_TYPE_INFO,
              "PATCH /private/units completed with status %u\n",
              (unsigned int) response_code);
  switch (response_code)
  {
  case MHD_HTTP_NO_CONTENT:
    break;
  case MHD_HTTP_BAD_REQUEST:
  case MHD_HTTP_UNAUTHORIZED:
  case MHD_HTTP_FORBIDDEN:
  case MHD_HTTP_NOT_FOUND:
  case MHD_HTTP_CONFLICT:
  case MHD_HTTP_INTERNAL_SERVER_ERROR:
    hr.ec = TALER_JSON_get_error_code (json);
    hr.hint = TALER_JSON_get_error_hint (json);
    break;
  case 0:
    hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
    break;
  default:
    TALER_MERCHANT_parse_error_details_ (json,
                                         response_code,
                                         &hr);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response %u/%d for PATCH /private/units\n",
                (unsigned int) response_code,
                (int) hr.ec);
    GNUNET_break_op (0);
    break;
  }
  uph->cb (uph->cb_cls,
           &hr);
  TALER_MERCHANT_unit_patch_cancel (uph);
}


struct TALER_MERCHANT_UnitPatchHandle *
TALER_MERCHANT_unit_patch (struct GNUNET_CURL_Context *ctx,
                           const char *backend_url,
                           const char *unit_id,
                           const char *unit_name_long,
                           const char *unit_name_short,
                           const json_t *unit_name_long_i18n,
                           const json_t *unit_name_short_i18n,
                           const bool *unit_allow_fraction,
                           const uint32_t *unit_precision_level,
                           const bool *unit_active,
                           TALER_MERCHANT_UnitPatchCallback cb,
                           void *cb_cls)
{
  struct TALER_MERCHANT_UnitPatchHandle *uph;
  json_t *req_obj;
  char *path;

  req_obj = json_object ();
  if (NULL == req_obj)
    return NULL;
  if (NULL != unit_name_long)
  {
    if (0 != json_object_set_new (req_obj,
                                  "unit_name_long",
                                  json_string (unit_name_long)))
    {
      json_decref (req_obj);
      return NULL;
    }
  }
  if (NULL != unit_name_short)
  {
    if (0 != json_object_set_new (req_obj,
                                  "unit_name_short",
                                  json_string (unit_name_short)))
    {
      json_decref (req_obj);
      return NULL;
    }
  }
  if (NULL != unit_name_long_i18n)
  {
    if (0 != json_object_set_new (req_obj,
                                  "unit_name_long_i18n",
                                  json_incref ((json_t *) unit_name_long_i18n)))
    {
      json_decref (req_obj);
      return NULL;
    }
  }
  if (NULL != unit_name_short_i18n)
  {
    if (0 != json_object_set_new (req_obj,
                                  "unit_name_short_i18n",
                                  json_incref (
                                    (json_t *) unit_name_short_i18n)))
    {
      json_decref (req_obj);
      return NULL;
    }
  }
  if (NULL != unit_allow_fraction)
  {
    if (0 != json_object_set_new (req_obj,
                                  "unit_allow_fraction",
                                  json_boolean (*unit_allow_fraction)))
    {
      json_decref (req_obj);
      return NULL;
    }
  }
  if (NULL != unit_precision_level)
  {
    if (0 != json_object_set_new (req_obj,
                                  "unit_precision_level",
                                  json_integer (
                                    (json_int_t) *unit_precision_level)))
    {
      json_decref (req_obj);
      return NULL;
    }
  }
  if (NULL != unit_active)
  {
    if (0 != json_object_set_new (req_obj,
                                  "unit_active",
                                  json_boolean (*unit_active)))
    {
      json_decref (req_obj);
      return NULL;
    }
  }
  if (0 == json_object_size (req_obj))
  {
    json_decref (req_obj);
    GNUNET_break (0);
    return NULL;
  }

  GNUNET_asprintf (&path,
                   "private/units/%s",
                   unit_id);
  uph = GNUNET_new (struct TALER_MERCHANT_UnitPatchHandle);
  uph->ctx = ctx;
  uph->cb = cb;
  uph->cb_cls = cb_cls;
  uph->url = TALER_url_join (backend_url,
                             path,
                             NULL);
  GNUNET_free (path);
  if (NULL == uph->url)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Failed to build /private/units/%s URL\n",
                unit_id);
    json_decref (req_obj);
    GNUNET_free (uph);
    return NULL;
  }
  {
    CURL *eh;

    eh = TALER_MERCHANT_curl_easy_get_ (uph->url);
    if (GNUNET_OK !=
        TALER_curl_easy_post (&uph->post_ctx,
                              eh,
                              req_obj))
    {
      GNUNET_break (0);
      curl_easy_cleanup (eh);
      json_decref (req_obj);
      GNUNET_free (uph->url);
      GNUNET_free (uph);
      return NULL;
    }
    json_decref (req_obj);
    GNUNET_assert (CURLE_OK ==
                   curl_easy_setopt (eh,
                                     CURLOPT_CUSTOMREQUEST,
                                     MHD_HTTP_METHOD_PATCH));
    uph->job = GNUNET_CURL_job_add2 (ctx,
                                     eh,
                                     uph->post_ctx.headers,
                                     &handle_patch_unit_finished,
                                     uph);
  }
  return uph;
}


void
TALER_MERCHANT_unit_patch_cancel (struct TALER_MERCHANT_UnitPatchHandle *uph)
{
  if (NULL != uph->job)
  {
    GNUNET_CURL_job_cancel (uph->job);
    uph->job = NULL;
  }
  TALER_curl_easy_post_finished (&uph->post_ctx);
  GNUNET_free (uph->url);
  GNUNET_free (uph);
}


/* end of merchant_api_patch_unit.c */
