/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free Software
  Foundation; either version 2.1, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License along with
  TALER; see the file COPYING.LGPL.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant_api_post_instance_token.c
 * @brief Implementation of the POST /instance/$ID/private/token request
 * @author Martin Schanzenbach
 */
#include "platform.h"
#include <curl/curl.h>
#include <jansson.h>
#include <microhttpd.h> /* just for HTTP status codes */
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include "taler_merchant_service.h"
#include "merchant_api_curl_defaults.h"
#include <taler/taler_json_lib.h>
#include <taler/taler_signatures.h>
#include <taler/taler_curl_lib.h>


/**
 * Handle for a POST /instances/$ID/private/token operation.
 */
struct TALER_MERCHANT_InstanceTokenPostHandle
{
  /**
   * The url for this request.
   */
  char *url;

  /**
   * Handle for the request.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Function to call with the result.
   */
  TALER_MERCHANT_InstanceTokenPostCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Reference to the execution context.
   */
  struct GNUNET_CURL_Context *ctx;

  /**
   * Minor context that holds body and headers.
   */
  struct TALER_CURL_PostContext post_ctx;

};


/**
 * Function called when we're done processing the
 * HTTP GET /instances/$ID/private/token request.
 *
 * @param cls the `struct TALER_MERCHANT_InstanceTokenPostHandle`
 * @param response_code HTTP response code, 0 on error
 * @param response response body, NULL if not in JSON
 */
static void
handle_post_instance_token_finished (void *cls,
                                     long response_code,
                                     const void *response)
{
  struct TALER_MERCHANT_InstanceTokenPostHandle *itph = cls;
  const json_t *json = response;
  struct TALER_MERCHANT_HttpResponse hr = {
    .http_status = (unsigned int) response_code,
    .reply = json
  };

  itph->job = NULL;
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Got /instances/$ID response with status code %u\n",
              (unsigned int) response_code);
  switch (response_code)
  {
  case MHD_HTTP_OK:
    break;
  case MHD_HTTP_BAD_REQUEST:
    /* happens if the auth token is malformed */
    hr.ec = TALER_JSON_get_error_code (json);
    hr.hint = TALER_JSON_get_error_hint (json);
    /* Nothing really to verify, merchant says we need to authenticate. */
    break;
  case MHD_HTTP_UNAUTHORIZED:
    hr.ec = TALER_JSON_get_error_code (json);
    hr.hint = TALER_JSON_get_error_hint (json);
    /* Nothing really to verify, merchant says we need to authenticate. */
    break;
  default:
    /* unexpected response code */
    hr.ec = TALER_JSON_get_error_code (json);
    hr.hint = TALER_JSON_get_error_hint (json);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u/%d\n",
                (unsigned int) response_code,
                (int) hr.ec);
    break;
  }
  itph->cb (itph->cb_cls,
            &hr);
  TALER_MERCHANT_instance_token_post_cancel (itph);
}


struct TALER_MERCHANT_InstanceTokenPostHandle *
TALER_MERCHANT_instance_token_post (
  struct GNUNET_CURL_Context *ctx,
  const char *backend_url,
  const char *instance_id,
  const char *scope,
  struct GNUNET_TIME_Relative duration,
  bool refreshable,
  TALER_MERCHANT_InstanceTokenPostCallback cb,
  void *cb_cls)
{
  struct TALER_MERCHANT_InstanceTokenPostHandle *itph;
  json_t *req_obj;

  itph = GNUNET_new (struct TALER_MERCHANT_InstanceTokenPostHandle);
  itph->ctx = ctx;
  itph->cb = cb;
  itph->cb_cls = cb_cls;
  if (NULL != instance_id)
  {
    char *path;

    GNUNET_asprintf (&path,
                     "instances/%s/private/token",
                     instance_id);
    itph->url = TALER_url_join (backend_url,
                                path,
                                NULL);
    GNUNET_free (path);
  }
  else
  {
    /* backend_url is already identifying the instance */
    itph->url = TALER_url_join (backend_url,
                                "private/token",
                                NULL);
  }
  if (NULL == itph->url)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Could not construct request URL.\n");
    GNUNET_free (itph);
    return NULL;
  }
  if (NULL == scope)
  {
    req_obj = GNUNET_JSON_PACK (
      GNUNET_JSON_pack_time_rel ("duration",
                                 duration),
      GNUNET_JSON_pack_bool ("refreshable",
                             refreshable),
      GNUNET_JSON_pack_string ("scope",
                               "readonly"));
  }
  else
  {
    req_obj = GNUNET_JSON_PACK (
      GNUNET_JSON_pack_time_rel ("duration",
                                 duration),
      GNUNET_JSON_pack_bool ("refreshable",
                             refreshable),
      GNUNET_JSON_pack_string ("scope",
                               scope));
  }
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Requesting URL '%s'\n",
              itph->url);
  {
    CURL *eh;

    eh = TALER_MERCHANT_curl_easy_get_ (itph->url);
    if (GNUNET_OK !=
        TALER_curl_easy_post (&itph->post_ctx,
                              eh,
                              req_obj))
    {
      GNUNET_break (0);
      curl_easy_cleanup (eh);
      json_decref (req_obj);
      GNUNET_free (itph->url);
      GNUNET_free (itph);
      return NULL;
    }
    json_decref (req_obj);
    GNUNET_assert (CURLE_OK ==
                   curl_easy_setopt (eh,
                                     CURLOPT_CUSTOMREQUEST,
                                     MHD_HTTP_METHOD_POST));
    itph->job = GNUNET_CURL_job_add2 (ctx,
                                      eh,
                                      itph->post_ctx.headers,
                                      &handle_post_instance_token_finished,
                                      itph);
  }
  return itph;
}


void
TALER_MERCHANT_instance_token_post_cancel (
  struct TALER_MERCHANT_InstanceTokenPostHandle *itph)
{
  if (NULL != itph->job)
    GNUNET_CURL_job_cancel (itph->job);
  TALER_curl_easy_post_finished (&itph->post_ctx);
  GNUNET_free (itph->url);
  GNUNET_free (itph);
}
