/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
  details.

  You should have received a copy of the GNU Affero General Public License
  along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant/backend/taler-merchant-httpd_private-post-pots.c
 * @brief implementation of POST /private/pots
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-post-pots.h"
#include <taler/taler_json_lib.h>


MHD_RESULT
TMH_private_post_pots (const struct TMH_RequestHandler *rh,
                       struct MHD_Connection *connection,
                       struct TMH_HandlerContext *hc)
{
  const char *pot_name;
  const char *description;
  enum GNUNET_DB_QueryStatus qs;
  uint64_t pot_id;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_string ("pot_name",
                             &pot_name),
    GNUNET_JSON_spec_string ("description",
                             &description),
    GNUNET_JSON_spec_end ()
  };

  (void) rh;
  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (connection,
                                     hc->request_body,
                                     spec);
    if (GNUNET_OK != res)
    {
      GNUNET_break_op (0);
      return (GNUNET_NO == res)
             ? MHD_YES
             : MHD_NO;
    }
  }

  qs = TMH_db->insert_money_pot (TMH_db->cls,
                                 hc->instance->settings.id,
                                 pot_name,
                                 description,
                                 &pot_id);

  if (qs < 0)
  {
    /* NOTE: Like product groups, we cannot distinguish between a
     * generic DB error and a unique constraint violation on pot_name.
     */
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_STORE_FAILED,
                                       "insert_money_pot");
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
  {
    /* Zero will be returned on conflict */
    return TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_CONFLICT,
      TALER_EC_MERCHANT_PRIVATE_MONEY_POT_CONFLICTING_NAME,
      pot_name);
  }
  return TALER_MHD_REPLY_JSON_PACK (
    connection,
    MHD_HTTP_OK,
    GNUNET_JSON_pack_uint64 ("pot_serial_id",
                             pot_id));
}
