/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  assertUnreachable,
  TranslatedString
} from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { h } from "preact";
import { useEffect, useState } from "preact/hooks";
import {
  datetimeFormatForSettings,
  usePreference,
} from "../../../../hooks/preference.js";

interface Props {
  events: Event[];
}

export function Timeline({ events: e }: Props) {
  const { i18n } = useTranslationContext();
  const events = [...e];
  events.push({
    when: new Date(),
    description: i18n.str`now`,
    type: "now",
  });

  events.sort((a, b) => a.when.getTime() - b.when.getTime());
  const [settings] = usePreference();
  const [state, setState] = useState(events);
  useEffect(() => {
    const handle = setTimeout(() => {
      const eventsWithoutNow = state.filter((e) => e.type !== "now");
      eventsWithoutNow.push({
        when: new Date(),
        description: i18n.str`now`,
        type: "now",
      });
      setState(eventsWithoutNow);
    }, 1000);
    return () => {
      clearTimeout(handle);
    };
  });
  return (
    <div class="timeline">
      {events.map((e, i) => {
        return (
          <div key={i} class="timeline-item">
            {(() => {
              switch (e.type) {
                case "refund-deadline":
                  return (
                    <div class="timeline-marker is-icon " data-tooltip={i18n.str`This is when the time for making refund has been expired.`}>
                      <i class="mdi mdi-flag" />
                    </div>
                  );
                case "pay-deadline":
                  return (
                    <div class="timeline-marker is-icon " data-tooltip={i18n.str`This is when the time for making the payment has been expired.`}>
                      <i class="mdi mdi-flag" />
                    </div>
                  );
                case "wire-deadline":
                  return (
                    <div class="timeline-marker is-icon " data-tooltip={i18n.str`This is when the wire transfer is going to be executed by the payment service provider.`}>
                      <i class="mdi mdi-flag" />
                    </div>
                  );
                case "delivery":
                  return (
                    <div class="timeline-marker is-icon ">
                      <i class="mdi mdi-delivery" />
                    </div>
                  );
                case "start":
                  return (
                    <div class="timeline-marker is-icon">
                      <i class="mdi mdi-flag " />
                    </div>
                  );
                case "wired-pending":
                  return (
                    <div class="timeline-marker is-icon is-warning" data-tooltip={i18n.str`This is when the wire transfer is going to be executed by the payment service provider.`}>
                      <i class="mdi mdi-cash" />
                    </div>
                  );
                case "wired-confirmed":
                  return (
                    <div class="timeline-marker is-icon is-success" data-tooltip={i18n.str`This wire transfer has been notified by the payment service provider.`}>
                      <i class="mdi mdi-cash" />
                    </div>
                  );
                case "refund-created":
                  return (
                    <div class="timeline-marker is-icon is-danger" data-tooltip={i18n.str`This refund is waiting to be claimed by the customer.`}>
                      <i class="mdi mdi-cash" />
                    </div>
                  );
                case "refund-taken":
                  return (
                    <div class="timeline-marker is-icon is-success" data-tooltip={i18n.str`This refund has been claimed by the customer.`}>
                      <i class="mdi mdi-cash" />
                    </div>
                  );
                case "now":
                  return (
                    <div class="timeline-marker is-icon is-info" data-tooltip={i18n.str`The current moment in time.`}>
                      <i class="mdi mdi-clock" />
                    </div>
                  );
                case "refund-missed": {
                  return (
                    <div class="timeline-marker is-icon is-danger" data-tooltip={i18n.str`This refund can't be claimed because the wire transfer has already be made.`}>
                      <i class="mdi mdi-cash" />
                    </div>
                  );
                }
                default: {
                  assertUnreachable(e.type);
                }
              }
            })()}
            <div class="timeline-content">
              {e.type !== "now" && (
                <p class="heading">
                  {format(e.when, datetimeFormatForSettings(settings))}
                </p>
              )}
              <p>{e.description}</p>
            </div>
          </div>
        );
      })}
    </div>
  );
}
export interface Event {
  when: Date;
  description: TranslatedString;
  type:
    | "start"
    | "refund-created"
    | "refund-taken"
    | "refund-missed"
    | "refund-deadline"
    | "pay-deadline"
    | "wire-deadline"
    | "wired-pending"
    | "wired-confirmed"
    | "delivery"
    | "now";
}
