/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  ButtonBetterBulma,
  SafeHandlerTemplate,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { FormProvider } from "../../../components/form/FormProvider.js";
import { Input } from "../../../components/form/Input.js";
import { undefinedIfEmpty } from "../../../utils/table.js";

const TALER_SCREEN_ID = 53;

interface Props {
  instanceId: string;
  onBack?: () => void;
  withoutCurrentPassword?: boolean;
  changePassword: SafeHandlerTemplate<[current: string, new: string], any>;
}

export function DetailPage({
  instanceId,
  onBack,
  changePassword,
  withoutCurrentPassword,
}: Props): VNode {
  type State = {
    current: string;
    next: string;
    repeat: string;
  };
  const [form, setValue] = useState<Partial<State>>({
    current: "",
    next: "",
    repeat: "",
  });

  const { i18n } = useTranslationContext();

  const errors = undefinedIfEmpty({
    current: withoutCurrentPassword
      ? undefined
      : !form.current
        ? i18n.str`Required`
        : undefined,
    next: !form.next ? i18n.str`Required` : undefined,
    repeat: !form.repeat
      ? i18n.str`Required`
      : form.next !== form.repeat
        ? i18n.str`Is not the same`
        : undefined,
  });

  const hasErrors = errors !== undefined;

  const text = i18n.str`You are updating the password for the instance with ID "${instanceId}"`;

  return (
    <div>
      <section class="section">
        <section class="hero is-hero-bar">
          <div class="hero-body">
            <div class="level">
              <div class="level-left">
                <div class="level-item">
                  <span class="is-size-4">{text}</span>
                </div>
              </div>
            </div>
          </div>
        </section>
        <hr />

        <div class="columns">
          <div class="column" />
          <div class="column is-four-fifths">
            <FormProvider errors={errors} object={form} valueHandler={setValue}>
              <Fragment>
                {withoutCurrentPassword ? undefined : (
                  <Input<State>
                    name="current"
                    label={i18n.str`Current password`}
                    tooltip={i18n.str`In order to verify that you have access.`}
                    inputType="password"
                  />
                )}
                <Input<State>
                  name="next"
                  label={i18n.str`New password`}
                  tooltip={i18n.str`Next password to be used`}
                  inputType="password"
                />
                <Input<State>
                  name="repeat"
                  label={i18n.str`Repeat password`}
                  tooltip={i18n.str`Confirm the same password`}
                  inputType="password"
                />
              </Fragment>
              <div class="buttons is-right mt-5">
                {onBack && (
                  <a class="button" onClick={onBack}>
                    <i18n.Translate>Cancel</i18n.Translate>
                  </a>
                )}
                <ButtonBetterBulma
                  type="submit"
                  data-tooltip={
                    hasErrors
                      ? i18n.str`Please complete the marked fields`
                      : i18n.str`Confirm operation`
                  }
                  onClick={
                    hasErrors
                      ? changePassword
                      : changePassword.withArgs(form.current ?? "", form.next!)
                  }
                >
                  <i18n.Translate>Confirm change</i18n.Translate>
                </ButtonBetterBulma>
              </div>
            </FormProvider>
          </div>
          <div class="column" />
        </div>
      </section>
    </div>
  );
}
