/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  HttpStatusCode,
  IncomingTransferDetails,
  TalerError,
  TransferDetails,
  TransferInformation,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import {
  PaginatedResult,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useEffect, useState } from "preact/hooks";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../../components/exception/loading.js";
import { useInstanceBankAccounts } from "../../../../hooks/bank.js";
import {
  useInstanceConfirmedTransfers,
  useInstanceIncomingTransfers,
} from "../../../../hooks/transfer.js";
import { LoginPage } from "../../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";
import { ListPage } from "./ListPage.js";
import { ConfirmModal, Row } from "../../../../components/modal/index.js";
import { Amount } from "../../../../components/Amount.js";
import { format } from "date-fns";
import {
  datetimeFormatForSettings,
  usePreference,
} from "../../../../hooks/preference.js";
import { useSessionContext } from "../../../../context/session.js";

interface Props {
  // onCreate: () => void;
}
interface Form {
  expected?: boolean;
  confirmed?: boolean;
  verified?: boolean;
  payto_uri?: string;
}

export default function ListTransfer({}: Props): VNode {
  const setFilter = (s?: boolean) => setForm({ ...form, verified: s });

  const { state: session, lib } = useSessionContext();

  const [position, setPosition] = useState<string | undefined>(undefined);
  const [settings] = usePreference();

  const instance = useInstanceBankAccounts();
  const accounts =
    !instance || instance instanceof TalerError || instance.type === "fail"
      ? []
      : instance.body.accounts.map((a) => a.payto_uri);
  const [form, setForm] = useState<Form>({ payto_uri: "", verified: false });
  const [selected, setSelected] = useState<IncomingTransferDetails>();
  const { i18n } = useTranslationContext();
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();

  const shoulUseDefaultAccount = accounts.length === 1;
  useEffect(() => {
    if (shoulUseDefaultAccount) {
      setForm({ ...form, payto_uri: accounts[0] });
    }
  }, [shoulUseDefaultAccount]);

  const confirm = safeFunctionHandler(
    lib.instance.informWireTransfer.bind(lib.instance),
    !session.token || !selected
      ? undefined
      : [
          session.token,
          {
            credit_amount: selected.expected_credit_amount!,
            exchange_url: selected.exchange_url,
            payto_uri: selected.payto_uri,
            wtid: selected.wtid,
          },
        ],
  );
  confirm.onSuccess = () => {
    setSelected(undefined)
  };
  confirm.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
      case HttpStatusCode.Conflict:
        return i18n.str`Wire transfer already confirmed.`;
    }
  };

  // const isVerifiedTransfers = form.verified === true;
  // const isNonVerifiedTransfers = form.verified === false;
  // const isAllTransfers = form.verified === undefined;

  let incoming: PaginatedResult<IncomingTransferDetails[]>;
  {
    const result = useInstanceIncomingTransfers(
      {
        position,
        payto_uri: form.payto_uri === "" ? undefined : form.payto_uri,
        // verified: form.verified,
        // confirmed: form.confirmed,
      },
      (id) => setPosition(id),
    );
    if (!result) return <Loading />;
    if (result instanceof TalerError) {
      return <ErrorLoadingMerchant error={result} />;
    }
    if (result.type === "fail") {
      switch (result.case) {
        case HttpStatusCode.Unauthorized: {
          return <LoginPage />;
        }
        case HttpStatusCode.NotFound: {
          return <NotFoundPageOrAdminCreate />;
        }
        default: {
          assertUnreachable(result);
        }
      }
    }
    incoming = result;
  }
  let confirmed: PaginatedResult<TransferDetails[]>;
  {
    const result = useInstanceConfirmedTransfers(
      {
        position,
        payto_uri: form.payto_uri === "" ? undefined : form.payto_uri,
        // expected: form.expected,
      },
      (id) => setPosition(id),
    );
    if (!result) return <Loading />;
    if (result instanceof TalerError) {
      return <ErrorLoadingMerchant error={result} />;
    }
    if (result.type === "fail") {
      switch (result.case) {
        case HttpStatusCode.Unauthorized: {
          return <LoginPage />;
        }
        case HttpStatusCode.NotFound: {
          return <NotFoundPageOrAdminCreate />;
        }
        default: {
          assertUnreachable(result);
        }
      }
    }
    confirmed = result;
  }

  const show = form.verified ? confirmed : incoming;
  return (
    <Fragment>
      {/* <LocalNotificationBannerBulma notification={notification} /> */}
      {!selected ? undefined : (
        <ConfirmModal
          label={i18n.str`I have received the wire transfer`}
          description={i18n.str`Confirm the wire transfer`}
          active
          onCancel={() => {
            setSelected(undefined);
          }}
          confirm={confirm}
        >
          <p style={{ paddingTop: 0 }}>
            <i18n.Translate>
              The wire transfer has been sent and should be in your bank account
              in any time. You can manually confirm the reception using the
              information below.
            </i18n.Translate>
          </p>
          <div class="table-container">
            <table>
              <tbody>
                <Row
                  name={i18n.str`Amount`}
                  value={<Amount value={selected.expected_credit_amount!} />}
                />
                {selected.execution_time &&
                selected.execution_time.t_s !== "never" ? (
                  <Row
                    name={i18n.str`Time`}
                    value={format(
                      selected.execution_time.t_s * 1000,
                      datetimeFormatForSettings(settings),
                    )}
                  />
                ) : undefined}
                <Row
                  name={i18n.str`Transfer ID`}
                  value={selected.wtid}
                  literal
                />
              </tbody>
            </table>
          </div>
        </ConfirmModal>
      )}
      <ListPage
        accounts={accounts}
        transfers={confirmed.body}
        incomings={incoming.body}
        onSelectedToConfirm={(d) => {
          setSelected(d);
        }}
        onLoadMoreBefore={show.loadFirst}
        onLoadMoreAfter={show.loadNext}
        // onShowAll={() => setFilter(undefined)}
        onShowUnverified={() => setFilter(false)}
        onShowVerified={() => setFilter(true)}
        // isAllTransfers={isAllTransfers}
        isVerifiedTransfers={form.verified}
        isNonVerifiedTransfers={!form.verified}
        payTo={form.payto_uri}
        onChangePayTo={(p) => setForm((v) => ({ ...v, payto_uri: p }))}
      />
    </Fragment>
  );
}
