/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Configuration,
  Logger,
  parsePaytoUri,
  succeedOrThrow,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  configureCommonKyc,
  createKycTestkudosEnvironment,
  postAmlDecisionNoRules,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  getTestHarnessPaytoForLabel,
  GlobalTestState,
} from "../harness/harness.js";

const logger = new Logger("test-kyc-deposit-deposit.ts");

function adjustExchangeConfig(config: Configuration) {
  configureCommonKyc(config);

  config.setString("KYC-RULE-R1", "operation_type", "deposit");
  config.setString("KYC-RULE-R1", "enabled", "yes");
  config.setString("KYC-RULE-R1", "exposed", "yes");
  config.setString("KYC-RULE-R1", "is_and_combinator", "yes");
  config.setString("KYC-RULE-R1", "threshold", "TESTKUDOS:5");
  config.setString("KYC-RULE-R1", "timeframe", "1d");
  config.setString("KYC-RULE-R1", "next_measures", "M1");

  config.setString("KYC-MEASURE-M1", "check_name", "C1");
  config.setString("KYC-MEASURE-M1", "context", "{}");
  config.setString("KYC-MEASURE-M1", "program", "NONE");

  config.setString("KYC-CHECK-C1", "type", "INFO");
  config.setString("KYC-CHECK-C1", "description", "my check!");
  config.setString("KYC-CHECK-C1", "fallback", "FREEZE");
}

export async function runKycDepositDepositKyctransferTest(t: GlobalTestState) {
  // Set up test environment

  const {
    walletClient,
    bankClient,
    exchange,
    amlKeypair,
    bank,
    wireGatewayApi,
  } = await createKycTestkudosEnvironment(t, {
    adjustExchangeConfig,
  });

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    bankClient,
    amount: "TESTKUDOS:50",
    exchange: exchange,
    walletClient: walletClient,
  });

  await wres.withdrawalFinishedCond;

  const depositPaytoUri = getTestHarnessPaytoForLabel("deposit-test");

  await bankClient.registerAccountExtended({
    name: "deposit-test",
    password: "deposit-password",
    username: "deposit-test",
    is_taler_exchange: false,
    payto_uri: depositPaytoUri,
  });

  const depositResp = await walletClient.call(
    WalletApiOperation.CreateDepositGroup,
    {
      amount: "TESTKUDOS:10",
      depositPaytoUri,
    },
  );

  console.log("waiting for kyc-required");

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositResp.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.KycAuthRequired,
    },
  });

  t.logStep("kyc-auth-requested");

  {
    const txDetails = await walletClient.call(
      WalletApiOperation.GetTransactionById,
      {
        transactionId: depositResp.transactionId,
      },
    );

    t.assertDeepEqual(txDetails.type, TransactionType.Deposit);
    const kycTx = txDetails.kycAuthTransferInfo;
    t.assertTrue(!!kycTx);

    {
      const kycAuthCreditPayto = kycTx.creditPaytoUris[0];
      t.assertTrue(!!kycAuthCreditPayto);
      const p = parsePaytoUri(kycAuthCreditPayto);
      t.assertTrue(!!p);
      t.assertAmountEquals(p.params["amount"], "TESTKUDOS:0.01");
    }

    logger.info(`account pub: ${kycTx.accountPub}`);

    succeedOrThrow(
      await wireGatewayApi.addKycAuth({
        auth: bank.getAdminAuth(),
        body: {
          amount: "TESTKUDOS:0.1",
          debit_account: depositPaytoUri,
          account_pub: kycTx.accountPub,
        },
      }),
    );
  }

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositResp.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.KycRequired,
    },
  });

  const txDetails = await walletClient.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: depositResp.transactionId,
    },
  );

  const kycPaytoHash = txDetails.kycPaytoHash;

  t.assertTrue(!!kycPaytoHash);

  await postAmlDecisionNoRules(t, {
    amlPriv: amlKeypair.priv,
    amlPub: amlKeypair.pub,
    exchangeBaseUrl: exchange.baseUrl,
    paytoHash: kycPaytoHash,
  });

  logger.info(`made decision to have no rules on ${kycPaytoHash}`);

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositResp.transactionId,
    txState: [
      {
        major: TransactionMajorState.Done,
      },
      {
        major: TransactionMajorState.Finalizing,
        minor: TransactionMinorState.Track,
      },
    ],
  });

  await exchange.runAggregatorOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60,
  });

  await exchange.runTransferOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositResp.transactionId,
    txState: {
      major: TransactionMajorState.Done,
    },
  });
}

runKycDepositDepositKyctransferTest.suites = ["wallet"];
