/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AmountString,
  TransactionMajorState,
  TransactionMinorState,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  GlobalTestState,
  getTestHarnessPaytoForLabel,
} from "../harness/harness.js";

const coinCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  feeDeposit: "TESTKUDOS:0",
  feeRefresh: "TESTKUDOS:0",
  feeRefund: "TESTKUDOS:0",
  feeWithdraw: "TESTKUDOS:0",
  rsaKeySize: 1024,
};

/**
 * Run test for refreshe after a payment.
 */
export async function runWalletBlockedDepositTest(t: GlobalTestState) {
  // Set up test environment

  const coinConfigList: CoinConfig[] = [
    {
      ...coinCommon,
      name: "n1",
      value: "TESTKUDOS:1",
    },
    {
      ...coinCommon,
      name: "n5",
      value: "TESTKUDOS:5",
    },
  ];

  const { bankClient, exchange, merchant, merchantAdminAccessToken } =
    await createSimpleTestkudosEnvironmentV3(t, coinConfigList);

  // Withdraw digital cash into the wallet.

  const { walletClient: w1 } = await createWalletDaemonWithClient(t, {
    name: "w1",
    persistent: true,
    config: {
      testing: {
        devModeActive: true,
      },
    },
  });

  await withdrawViaBankV3(t, {
    walletClient: w1,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await w1.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Prevent the wallet from doing refreshes by injecting a 5xx
  // status for all refresh requests.
  await w1.call(WalletApiOperation.ApplyDevExperiment, {
    devExperimentUri: "taler://dev-experiment/start-block-refresh",
  });

  await makeTestPaymentV2(t, {
    merchant,
    merchantAdminAccessToken,
    walletClient: w1,
    order: {
      summary: "test",
      amount: "TESTKUDOS:2",
    },
  });

  const userPayto = getTestHarnessPaytoForLabel("foo");

  const bal = await w1.call(WalletApiOperation.GetBalances, {});
  console.log(`balance: ${j2s(bal)}`);

  const balDet = await w1.call(WalletApiOperation.GetBalanceDetail, {
    currency: "TESTKUDOS",
  });
  console.log(`balance details: ${j2s(balDet)}`);

  const depositCheckResp = await w1.call(WalletApiOperation.CheckDeposit, {
    amount: "TESTKUDOS:18" as AmountString,
    depositPaytoUri: userPayto,
  });

  console.log(`check resp: ${j2s(depositCheckResp)}`);

  const depositCreateResp = await w1.call(
    WalletApiOperation.CreateDepositGroup,
    {
      amount: "TESTKUDOS:18" as AmountString,
      depositPaytoUri: userPayto,
    },
  );

  console.log(`create resp: ${j2s(depositCreateResp)}`);

  await w1.call(WalletApiOperation.ApplyDevExperiment, {
    devExperimentUri: "taler://dev-experiment/stop-block-refresh",
  });

  await w1.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositCreateResp.transactionId,
    txState: {
      major: TransactionMajorState.Finalizing,
      minor: TransactionMinorState.Track,
    },
  });
}

runWalletBlockedDepositTest.suites = ["wallet"];
