import {
  AbsoluteTime,
  assertUnreachable,
  Duration,
  OperationAlternative,
  OperationFail,
  OperationOk,
  OperationResult,
  TalerError,
  TalerErrorCode,
  TranslatedString,
} from "@gnu-taler/taler-util";
import { useEffect, useState } from "preact/hooks";
import {
  InternationalizationAPI,
  memoryMap,
  useTranslationContext,
} from "../index.browser.js";

export type NotificationMessage = ErrorNotification | InfoNotification;

export interface ErrorNotification {
  type: "error";
  title: TranslatedString;
  ack?: boolean;
  timeout?: boolean;
  description?: TranslatedString;
  debug?: any;
  when: AbsoluteTime;
}
export interface InfoNotification {
  type: "info";
  title: TranslatedString;
  ack?: boolean;
  timeout?: boolean;
  when: AbsoluteTime;
}

const storage = memoryMap<Map<string, NotificationMessage>>();
const NOTIFICATION_KEY = "notification";

export const GLOBAL_NOTIFICATION_TIMEOUT = Duration.fromSpec({
  seconds: 5,
});

function updateInStorage(n: NotificationMessage) {
  const h = hash(n);
  const mem = storage.get(NOTIFICATION_KEY) ?? new Map();
  const newState = new Map(mem);
  newState.set(h, n);
  storage.set(NOTIFICATION_KEY, newState);
}

export function notify(notif: NotificationMessage): void {
  const currentState: Map<string, NotificationMessage> =
    storage.get(NOTIFICATION_KEY) ?? new Map();
  const newState = currentState.set(hash(notif), notif);

  if (GLOBAL_NOTIFICATION_TIMEOUT.d_ms !== "forever") {
    setTimeout(() => {
      notif.timeout = true;
      updateInStorage(notif);
    }, GLOBAL_NOTIFICATION_TIMEOUT.d_ms);
  }

  storage.set(NOTIFICATION_KEY, newState);
}
export function notifyError(
  title: TranslatedString,
  description: TranslatedString | undefined,
  debug?: any,
) {
  notify({
    type: "error" as const,
    title,
    description,
    debug,
    when: AbsoluteTime.now(),
  });
}
export function notifyException(title: TranslatedString, ex: Error) {
  notify({
    type: "error" as const,
    title,
    description: ex.message as TranslatedString,
    debug: ex.stack,
    when: AbsoluteTime.now(),
  });
}
export function notifyInfo(title: TranslatedString) {
  notify({
    type: "info" as const,
    title,
    when: AbsoluteTime.now(),
  });
}

export type Notification = {
  message: NotificationMessage;
  acknowledge: () => void;
};

export function useNotifications(): Notification[] {
  const [, setLastUpdate] = useState<number>();
  const value = storage.get(NOTIFICATION_KEY) ?? new Map();

  useEffect(() => {
    return storage.onUpdate(NOTIFICATION_KEY, () => {
      setLastUpdate(Date.now());
      // const mem = storage.get(NOTIFICATION_KEY) ?? new Map();
      // setter(structuredClone(mem));
    });
  });

  return Array.from(value.values()).map((message, idx) => {
    return {
      message,
      acknowledge: () => {
        message.ack = true;
        updateInStorage(message);
      },
    };
  });
}

function hashCode(str: string): string {
  if (str.length === 0) return "0";
  let hash = 0;
  let chr;
  for (let i = 0; i < str.length; i++) {
    chr = str.charCodeAt(i);
    hash = (hash << 5) - hash + chr;
    hash |= 0; // Convert to 32bit integer
  }
  return hash.toString(16);
}

function hash(msg: NotificationMessage): string {
  let str = (msg.type + ":" + msg.title) as string;
  if (msg.type === "error") {
    if (msg.description) {
      str += ":" + msg.description;
    }
    if (msg.debug) {
      str += ":" + msg.debug;
    }
  }
  return hashCode(str);
}

function translateTalerError(
  cause: TalerError,
  i18n: InternationalizationAPI,
): TranslatedString {
  switch (cause.errorDetail.code) {
    case TalerErrorCode.GENERIC_TIMEOUT: {
      return i18n.str`Request timeout`;
    }
    case TalerErrorCode.GENERIC_CLIENT_INTERNAL_ERROR: {
      return i18n.str`Request cancelled`;
    }
    case TalerErrorCode.WALLET_HTTP_REQUEST_GENERIC_TIMEOUT: {
      return i18n.str`Request timeout`;
    }
    case TalerErrorCode.WALLET_HTTP_REQUEST_THROTTLED: {
      return i18n.str`Request throttled`;
    }
    case TalerErrorCode.WALLET_RECEIVED_MALFORMED_RESPONSE: {
      return i18n.str`Malformed response`;
    }
    case TalerErrorCode.WALLET_NETWORK_ERROR: {
      return i18n.str`Network error`;
    }
    case TalerErrorCode.WALLET_UNEXPECTED_REQUEST_ERROR: {
      return i18n.str`Unexpected request error`;
    }
    default: {
      return i18n.str`Unexpected error`;
    }
  }
}

/**
 * A function that may fail and return a message to be shown
 * as a notification
 */
export type FunctionThatMayFail<T extends any[]> = (
  ...args: T
) => Promise<NotificationMessage | undefined>;

type FunctionWrapperForButton = <T extends any[], R>(
  fn: FunctionThatMayFail<T>,
) => FunctionThatMayFail<T>;

type ReplaceReturnType<T extends (...a: any) => any, TNewReturn> = (
  ...a: Parameters<T>
) => TNewReturn;

/**
 * Initialize a notification handler.
 * @returns a tuple of notification and setter 
1) notification that may be set by a function when it fails.
 * 2) a error handling function that converts a function that returns a message
 * into a function that will set the notification.
 * 
 */
export function useLocalNotificationBetter(): [
  Notification | undefined,
  <Args extends any[], R extends OperationResult<any, any>>(
    doAction: (...args: Args) => Promise<R>,
    args?: Args,
  ) => SafeHandlerTemplate<Args, R>,
] {
  const [value, save] = useState<NotificationMessage>();
  const notif = !value
    ? undefined
    : {
        message: value,
        acknowledge: () => {
          save(undefined);
        },
      };

  // FIXME: we should move this outside of logic
  const { i18n } = useTranslationContext();

  function safeFunctionHandler<
    Args extends any[],
    R extends OperationResult<any, any>,
  >(
    doAction: (...args: Args) => Promise<R>,
    args?: Args,
  ): SafeHandlerTemplate<Args, R> {
    function buildSafeHandler(
      a: Args | undefined,
      doAction: (...args: Args) => Promise<R>,
    ): SafeHandlerTemplate<Args, R> {
      const thiz: SafeHandlerTemplate<Args, R> = {
        args: a,
        withArgs: (...newArgs) => {
          const r = buildSafeHandler(newArgs, doAction);
          r.onSuccess = thiz.onSuccess;
          r.onFail = thiz.onFail;
          return r;
        },
        lambda: (converter, init) => {
          type D = Parameters<typeof converter>;
          type SH = SafeHandlerTemplate<D, R>;

          const r = buildSafeHandler(
            init ? converter(...init) : undefined,
            doAction,
          );
          // @ts-expect-error
          r.withArgs = (...args: D) => {
            const d = converter(...args);
            if (!d) return thiz
            const e = thiz.withArgs(...d);
            return e;
          };
          r.onSuccess = thiz.onSuccess;
          r.onFail = thiz.onFail;
          return r as any as SH;
        },
        call: async (): Promise<void> => {
          if (!thiz.args) return;
          try {
            const resp = await doAction(...thiz.args);
            switch (resp.type) {
              case "ok": {
                const msg = thiz.onSuccess(resp.body, ...thiz.args);
                if (msg) {
                  save(successWithTitle(msg));
                }
                return;
              }
              case "fail": {
                const error = thiz.onFail(resp as any, ...thiz.args);
                if (error) {
                  save(failWithTitle(i18n, resp, error, thiz.args));
                }
                return;
              }
              default: {
                assertUnreachable(resp);
              }
            }
          } catch (error: unknown) {
            // This functions should not throw, this is a problem.
            logBugForDevelopers(error);
            onUnexpected(i18n, save)(error, thiz.args);
            return;
          }
        },
        onFail: (fail, ...rest) =>
          i18n.str`Unhandled failure, please report. Code ${fail.case}`,
        onSuccess: () => undefined,
      };
      return thiz;
    }
    return buildSafeHandler(args, doAction);
  }

  return [notif, safeFunctionHandler];
}

export function logBugForDevelopers(error: unknown) {
  console.error(
    `Internal error, this is mostly a bug in the application. Please report: `,
    error,
  );
}

function onUnexpected(
  i18n: InternationalizationAPI,
  save: (m: NotificationMessage) => void,
): (cause: unknown, args: any[]) => void {
  return (error, args) => {
    if (error instanceof TalerError) {
      save({
        title: translateTalerError(error, i18n),
        type: "error",
        description:
          error && error.errorDetail.hint
            ? (error.errorDetail.hint as TranslatedString)
            : undefined,
        debug: {
          error,
          stack: error instanceof Error ? error.stack : undefined,
          // args: sanitizeFunctionArguments(args),
          when: AbsoluteTime.now(),
        },
        when: AbsoluteTime.now(),
      });
    } else {
      const description = (
        error instanceof Error ? error.message : String(error)
      ) as TranslatedString;

      save({
        title: i18n.str`Operation failed.`,
        type: "error",
        description: i18n.str`Unexpected error, this is likely a bug. Please report `,
        debug: {
          error: String(error),
          stack: error instanceof Error ? error.stack : undefined,
          // args: sanitizeFunctionArguments(args),
          when: AbsoluteTime.now(),
        },
        when: AbsoluteTime.now(),
      });
    }
  };
}

function sanitizeFunctionArguments(args: any[]): string {
  return args
    .map((d) =>
      typeof d === "string" && d.startsWith("secret-token:")
        ? "<session>"
        : typeof d === "object"
          ? JSON.stringify(d, undefined, 2)
          : d,
    )
    .join(", ");
}

/**
 * A function converted into a safe handler.
 *
 *
 */
export interface SafeHandlerTemplate<Args extends any[], Errors> {
  readonly args: Args | undefined;
  /**
   * call the action with the arguments
   */
  call(): Promise<void>;
  /**
   * creates another handler for the same actions but different arguments
   * @param e
   */
  lambda<OtherArgs extends any[]>(
    e: (...d: OtherArgs) => Args | undefined,
    init?: OtherArgs,
  ): SafeHandlerTemplate<OtherArgs, Errors>;
  /**
   * creates another handler with new arguments
   * @param args
   */
  withArgs(...args: Args): SafeHandlerTemplate<Args, Error>;

  onSuccess: OnOperationSuccesReturnType<Errors, Args>;
  onFail: OnOperationFailReturnType<Errors, Args>;
}

function successWithTitle(title: TranslatedString): NotificationMessage {
  return {
    title,
    type: "info",
    when: AbsoluteTime.now(),
  };
}

function failWithTitle(
  i18n: InternationalizationAPI,
  fail: OperationFail<any>,
  description: TranslatedString,
  args: any[],
): NotificationMessage {
  return {
    title: i18n.str`The operation failed.`,
    type: "error",
    description,
    debug: {
      detail: fail.detail,
      case: fail.case,
      when: AbsoluteTime.now(),
      // args: sanitizeFunctionArguments(args),
    },
    when: AbsoluteTime.now(),
  };
}

export type OnOperationSuccesReturnType<T, K extends any[]> = (
  result: T extends OperationOk<infer B> ? B : never,
  ...args: K
) => TranslatedString | undefined | void;

export type OnOperationFailReturnType<T, K extends any[]> = (
  d:
    | (T extends OperationFail<any> ? T : never)
    | (T extends OperationAlternative<any, any> ? T : never),
  ...args: K
) => TranslatedString | undefined;

export type OnOperationUnexpectedFailReturnType = (e: unknown) => void;
