/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  HttpStatusCode,
  TalerError,
  TalerMerchantApi,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import {
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { useState } from "preact/hooks";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../../components/exception/loading.js";
import { JumpToElementById } from "../../../../components/form/JumpToElementById.js";
import { ConfirmModal } from "../../../../components/modal/index.js";
import { useSessionContext } from "../../../../context/session.js";
import { WithId } from "../../../../declaration.js";
import { useInstanceProducts } from "../../../../hooks/product.js";
import { LoginPage } from "../../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";
import { CardTable } from "./Table.js";

const TALER_SCREEN_ID = 55;

interface Props {
  onCreate: undefined | (() => void);
  onSelect: (id: string) => void;
}
export default function ProductList({ onCreate, onSelect }: Props): VNode {
  const result = useInstanceProducts();
  const { state: session, lib } = useSessionContext();
  const [deleting, setDeleting] = useState<
    (TalerMerchantApi.ProductDetail & WithId) | null
  >(null);

  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const { i18n } = useTranslationContext();

  const remove = safeFunctionHandler(
    lib.instance.deleteProduct.bind(lib.instance),
    !session.token || !deleting ? undefined : [session.token, deleting.id],
  );
  remove.onSuccess = (suc, t, id) => {
    setDeleting(null)
    return i18n.str`Product (ID: ${id}) has been deleted`
  }
  remove.onFail = (fail) =>{
    switch(fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
      case HttpStatusCode.Conflict:
        return i18n.str`Conflict.`;
    }
  }
  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />;
      }
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  return (
    <section class="section is-main-section">
      <LocalNotificationBannerBulma notification={notification} />

      <JumpToElementById
        onSelect={onSelect}
        description={i18n.str`Jump to product with the given product ID`}
        placeholder={i18n.str`Product id`}
      />

      <CardTable
        instances={result.body}
        onLoadMoreBefore={result.loadFirst}
        onLoadMoreAfter={result.loadNext}
        onCreate={onCreate}
        onSelect={(product) => onSelect(product.id)}
        onDelete={(prod: TalerMerchantApi.ProductDetail & WithId) =>
          setDeleting(prod)
        }
      />

      {deleting && (
        <ConfirmModal
          label={`Delete product`}
          description={`Delete the product "${deleting.description}"`}
          danger
          active
          onCancel={() => setDeleting(null)}
          confirm={remove}
        >
          <p>
            <i18n.Translate>
              If you delete the product named{" "}
              <b>&quot;{deleting.description}&quot;</b> (ID:{" "}
              <b>{deleting.id}</b>
              ), the stock and related information will be lost
            </i18n.Translate>
          </p>
          <p class="warning">
            <i18n.Translate>
              Deleting a product cannot be undone.
            </i18n.Translate>
          </p>
        </ConfirmModal>
      )}
    </section>
  );
}
