/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { ErrorMessage } from "../../components/ErrorMessage.js";
import {
    Centered,
  Input,
  LargeText,
  LightText,
  SmallLightText,
  SmallText,
  SubTitle,
  Title,
  WarningBox,
} from "../../components/styled/index.js";
import { Button } from "../../mui/Button.js";
import { State } from "./index.js";
import { assertUnreachable } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { Loading } from "../../components/Loading.js";
import { useBackendContext } from "../../context/backend.js";
import { useAsyncAsHook } from "../../hooks/useAsyncAsHook.js";

export function VerifyContactView({
  alias,
  aliasType,
  mailboxUri,
  onCancel,
  onAccept,
  result,
}: State.Verify): VNode {
  const { i18n } = useTranslationContext();

  return (
    <Fragment>
      <section>
        <Title>
          <i18n.Translate>Add new contact</i18n.Translate>
        </Title>
        {!result && (
          <LightText>
            <i18n.Translate>
              Enter the details of a contact you trust.
            </i18n.Translate>
          </LightText>
        )}
        {(() => {
          if (!result) return;
          if (result.type == "ok") {
            return (
              <LightText>
                <i18n.Translate>
                  Contact info valid.
                </i18n.Translate>
              </LightText>
            );
          }
          switch (result.case) {
            case "already-added": {
              return (
                <WarningBox>
                  <i18n.Translate>
                    This contact is already in your list.
                  </i18n.Translate>
                </WarningBox>
              );
            }
            default: {
              assertUnreachable(result);
            }
          }
        })()}
        <p>
          <Input invalid={result && result.type !== "ok"}>
            <label>Alias</label>
            <input
              type="text"
              placeholder="john.doe@id.example"
              value={alias.value}
              onInput={(e) => {
                if (alias.onInput) {
                  alias.onInput(e.currentTarget.value);
                }
              }}
            />
          </Input>
          <Input invalid={result && result.type !== "ok"}>
            <label>Alias type</label>
            <input
              type="text"
              placeholder="email"
              value={aliasType.value}
              onInput={(e) => {
                if (aliasType.onInput) {
                  aliasType.onInput(e.currentTarget.value);
                }
              }}
            />
          </Input>
          <Input invalid={result && result.type !== "ok"}>
            <label>Mailbox</label>
            <input
              type="text"
              placeholder="http://"
              value={mailboxUri.value}
              onInput={(e) => {
                if (mailboxUri.onInput) {
                  mailboxUri.onInput(e.currentTarget.value);
                }
              }}
            />
          </Input>
        </p>
      </section>
      <footer>
        <Button variant="contained" color="secondary" onClick={onCancel}>
          <i18n.Translate>Cancel</i18n.Translate>
        </Button>
        <Button
          variant="contained"
          disabled={!result || result.type !== "ok"}
          onClick={onAccept}
        >
          <i18n.Translate>Next</i18n.Translate>
        </Button>
      </footer>
    </Fragment>
  );
}

export function ConfirmAddContactView({
  contact,
  onCancel,
  onConfirm,
}: State.Confirm): VNode {
  const { i18n } = useTranslationContext();

  const api = useBackendContext();

  const state = useAsyncAsHook(async () => {
    const b = await api.wallet.call(WalletApiOperation.GetContacts, {});
    const contacts = b.contacts;
    return { contacts: contacts };
  }, []);

  if (!state) {
    return <Loading />;
  }
  async function onAddContact(): Promise<void> {
    await api.wallet.call(WalletApiOperation.AddContact, { contact: contact }).then();
    onConfirm();
  }
 
  return (
    <Fragment>
      <section style="">
        <LargeText>
          <i18n.Translate>Review contact</i18n.Translate>
        </LargeText>
        <SmallText style={{ marginTop: 5 }}>
          <b><i18n.Translate>Contact alias</i18n.Translate></b>: {contact.alias}
        </SmallText>
         <SmallText style={{ marginTop: 5 }}>
          <b><i18n.Translate>Alias type</i18n.Translate></b>: {contact.aliasType}
        </SmallText>
        <SmallText style={{ marginTop: 5 }}>
          <b><i18n.Translate>Mailbox service</i18n.Translate></b>: {contact.mailboxBaseUri}
        </SmallText>
        <SmallText style={{ marginTop: 5, marginBotton: 5 }}>
          <b><i18n.Translate>Mailbox address</i18n.Translate></b>: {contact.mailboxAddress.toUpperCase().slice(0,8)}[...]
        </SmallText>
        {(contact.source) && (
         <SmallText style={{ marginTop: 5 }}>
           <b><i18n.Translate>Contact source</i18n.Translate></b>: {contact.source}
         </SmallText>
        )}
     </section>

      <footer style={{padding: 8}}>
        <Button style={{ margin: 4 }}
          key="cancel"
          variant="contained"
          color="secondary"
          onClick={onCancel}
        >
          <i18n.Translate>Cancel</i18n.Translate>
        </Button>
        <Button
          style={{ margin: 4 }}
          key="add"
          variant="contained"
          color="success"
          onClick={onAddContact}
        >
          <i18n.Translate>Add contact</i18n.Translate>
        </Button>
      </footer>
    </Fragment>
  );
}
