# BEGIN BPS TAGGED BLOCK {{{
#
# COPYRIGHT:
#
# This software is Copyright (c) 1996-2022 Best Practical Solutions, LLC
#                                          <sales@bestpractical.com>
#
# (Except where explicitly superseded by other copyright notices)
#
#
# LICENSE:
#
# This work is made available to you under the terms of Version 2 of
# the GNU General Public License. A copy of that license should have
# been provided with this software, but in any event can be snarfed
# from www.gnu.org.
#
# This work is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301 or visit their web page on the internet at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
#
#
# CONTRIBUTION SUBMISSION POLICY:
#
# (The following paragraph is not intended to limit the rights granted
# to you to modify and distribute this software under the terms of
# the GNU General Public License and is only of importance to you if
# you choose to contribute your changes and enhancements to the
# community by submitting them to Best Practical Solutions, LLC.)
#
# By intentionally submitting any modifications, corrections or
# derivatives to this work, or any other work intended for use with
# Request Tracker, to Best Practical Solutions, LLC, you confirm that
# you are the copyright holder for those contributions and you grant
# Best Practical Solutions,  LLC a nonexclusive, worldwide, irrevocable,
# royalty-free, perpetual, license to use, copy, create derivative
# works based on those contributions, and sublicense and distribute
# those contributions and any derivatives thereof.
#
# END BPS TAGGED BLOCK }}}

package RT::REST2::Resource::Collection::QueryByJSON;
use strict;
use warnings;

use Moose::Role;
use namespace::autoclean;

use JSON ();

with (
    'RT::REST2::Resource::Collection::ProcessPOSTasGET',
    'RT::REST2::Resource::Role::RequestBodyIsJSON'
         => { type => 'ARRAY' },
);

requires 'collection';

has 'query_json' => (
    is          => 'ro',
    isa         => 'ArrayRef[HashRef]',
    required    => 1,
    lazy_build  => 1,
);

sub _build_query_json {
    my $self = shift;
    my $content = $self->request->method eq 'GET'
                ? $self->request->param('query')
                : $self->request->content;
    return [] unless $content && $content =~ /\s*\[/;
    return JSON::decode_json($content);
}

sub allowed_methods {
    [ 'GET', 'POST' ]
}

sub searchable_fields {
    $_[0]->collection->RecordClass->ReadableAttributes
}

sub limit_collection_from_json {
    my $self        = shift;
    my $collection  = $self->collection;
    my $query       = $self->query_json;

    return 1 unless ref $query eq 'ARRAY' && scalar @{$query};

    my @fields      = $self->searchable_fields;
    my %searchable  = map {; $_ => 1 } @fields;

    my $custom_field_object = RT::CustomField->new( $self->request->env->{"rt.current_user"} );

    for my $limit (@$query) {
        next unless $limit->{field} && defined $limit->{value};

        if ( $limit->{field} =~ /(?:CF|CustomField)\.\{(.*)\}/i ) {
            my $cf_name = $1;
            next unless $cf_name;

            my ($ret, $msg) = $custom_field_object->LoadByName(
                Name          => $cf_name,
                LookupType    => $collection->RecordClass->CustomFieldLookupType,
                IncludeGlobal => 1
            );

            unless ( $ret && $custom_field_object->Id ) {
                RT::Logger->error( "Could not load custom field: $limit->{'field'}: $msg" );
                next;
            }

            $collection->LimitCustomField(
              VALUE       => $limit->{'value'},
              CUSTOMFIELD => $custom_field_object->Id,
              ( $limit->{operator}
                ? (OPERATOR => $limit->{operator})
                : () ),
            );
        }
        else {
            next unless $searchable{$limit->{field}};

            $collection->Limit(
                FIELD       => $limit->{field},
                VALUE       => $limit->{value},
                ( $limit->{operator}
                    ? (OPERATOR => $limit->{operator})
                    : () ),
                CASESENSITIVE => ($limit->{case_sensitive} || 0),
                ( $limit->{entry_aggregator}
                    ? (ENTRYAGGREGATOR => $limit->{entry_aggregator})
                    : () ),
            );
        }
    }

    return 1;
}

1;
