/* playback.c - high-level audio playback implementation
 *
 * Copyright 2010, 2016 Petteri Hintsanen <petterih@iki.fi>
 *
 * This file is part of abx.
 *
 * abx is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * abx is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with abx.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "playback.h"
#include <glib.h>
#include <assert.h>
#include <stdio.h>

/* Sample file A. */
static Sound_file *sample_a;
/* Sample file B. */
static Sound_file *sample_b;
 /* The sample that is being played. */
static Sound_file *current_sample;

/*
 * Initialize the audio subsystem.  This function must be called
 * before any other playback functions.
 *
 * Return 0 on success, nonzero otherwise.
 */
int
init_audio(PaDeviceIndex outdev, unsigned int latency)
{
    g_debug("initializing audio subsystem, output device %d, latency %d",
            outdev, latency);
    if (init_player(outdev, latency) != 0) {
        g_error("can't initialize playback");
        return 1;
    } else {
        return 0;
    }
}

/*
 * Close the audio subsystem.  This function must be called before the
 * program exits, otherwise resource leaks may occur.
 */
void
close_audio(void)
{
    g_debug("closing audio subsystem");
    close_player();
}

/*
 * Initialize playback for sample files a and b.  The samples must
 * have equal duration.
 *
 * Return 
 *   0 on success
 *   1 if sample a could not be opened
 *   2 if sample b could not be opened
 *   3 if the samples have different durations
 */
int 
init_playback(const char *a, const char *b)
{
    Metadata ma;
    Metadata mb;
    if (sample_a) {
        assert(sample_b);
        return 1;
    }
    if (!(sample_a = open_sound_file(a))) {
        g_warning("can't open file '%s'", a);
        return 1;
    }
    if (!(sample_b = open_sound_file(b))) {
        g_warning("can't open file '%s'", b);
        close_playback();
        return 2;
    }
    ma = get_metadata(sample_a);
    mb = get_metadata(sample_b);
    if (ma.duration != mb.duration) {
        g_warning("samples '%s' and '%s' have different duration", a, b);
        close_playback();
        return 3;
    }
    current_sample = NULL;
    return 0;
}

/*
 * Close the current playback and release resources.
 */
void
close_playback(void)
{
    if ((sample_a && close_sound_file(sample_a) != 0)
        || (sample_b && close_sound_file(sample_b) != 0)) {
        g_warning("failed to close sound file");
    }
    sample_a = sample_b = current_sample = NULL;
}

/*
 * Fill the given metadata structures.
 */
void
get_sample_metadatas(Metadata *a, Metadata *b)
{
    assert(a && b);
    *a = get_metadata(sample_a);
    *b = get_metadata(sample_b);
}

/*
 * Get the current playback state and store it to *state.
 *
 * Return 0 on success, nonzero if playback is not in progress.
 */
int
get_playback_state(Player_state *state)
{
    assert(state);
    if (current_sample) {
        *state = get_player_state();
        return 0;
    } else return 1;
}

/*
 * Start playing sample from the given location (in seconds).
 */
void
start_playback(int sample, double location)
{
    Sound_file *s;
    if (sample == 0) s = sample_a;
    else s = sample_b;
    start_player(s, location);
    current_sample = s;
}

/*
 * Stop playback, if it is in progress.  Otherwise do nothing.
 */
void
stop_playback(void)
{
    if (current_sample == NULL) return;
    stop_player();
    current_sample = NULL;
}

/*
 * Pause or resume playback.
 *
 * Return the playback state after change.
 */
Player_state
pause_or_resume_playback(void)
{
    if (current_sample) {
        Player_state state;
        pause_or_resume_player();
        get_playback_state(&state);
        return state;
    } else {
        Player_state state = { STOPPED, 0 };
        return state;
    }
}

/*
 * Reposition playback to location (in seconds, measured from the
 * beginning).  If there is no playback in progress, do nothing.
 */
void
seek_playback(double offset)
{
    if (current_sample == NULL) return;
    seek_player(offset, SEEK_SET);
}
