# ES6 module support for JerryScript

The module system allows users to write import and export statements in scripts, which can be used to separate the logic of the application into custom modules.
The standard's relevant part can be found [here](https://www.ecma-international.org/ecma-262/6.0/#sec-modules).
Embedders wishing to use native builtin modules with ES6 imports can use the [Port API](05.PORT-API.md#jerry-module-system) to do so.

## General

If a script contains import statements, then JerryScript will open and evaluate the the referenced modules before the main script runs, resolving and creating bindings for the referenced identifiers in the process.
It is not necessary to use any specific filename extensions for modules, JerryScript will try to open the given file paths as they are, but will try to normalize them before doing so. The exact normalization process is dependant on the port implementation provided. It is the user's responsibility to verify that the given files are valid ECMAScript modules.

main.js

```js
import { exported_value } from "./module.js"

print (exported_value);
```

module.js

```js
var exported_value = 42;

export exported_value;
```

## Supported features

* exporting identifiers from the module's lexical environment
  * specifying export names for the exported values
* importing exported identifiers from a module
  * specifying local binding names for the imported values
* module namespace imports
  * `import * as module from 'module.js`
* indirect export statements
  * `export {variable} from 'module.js'`
* star export statements
  * `export * from 'module.js'`
* importing a module for side-effects
  * `import 'module.js'`
* default import and export statements
  * `export default local_identifier`
  * `import def from 'module.js'`
* anonymous default exports
  * `export default function () {}`

### Example

```js
import {
  engine,
  version as v
} from "./module.js"

import { getFeatureDetails } from "./module_2.js"

var version = "v3.1415";

print("> main.js");

print(">> Engine: " + engine);
print(">> Version: " + v);

print (">> " + getFeatureDetails());
print (">> Script version: " + version);
```

```js
// module.js
var _engine = "JerryScript";
export _engine as engine;

export var version = "1.0 (e92ae0fb)";
```

```js
// module_2.js
var featureName = "ECMAScript modules";
var year = 2018;

export function getFeatureDetails() {
  return "Feature name: " + featureName + " | developed in " + year;
}
```

### Module namespace import statements

A module namespace object can be imported. In this case the local binding will contain an object holding the exported values of the module, including local exports and all indirect exports. Ambiguous exported names are excluded from the namespace object.

```js
import * as module from './module.js';

print(">> Engine: " + module.engine);
print(">> Version: " + module.version);
```

### Indirect export statements

An export statement can transitively export variables from another module, either via named indirect exports or a star export statement. In this case the resolving process will follow the chain until it reaches a module containing a local binding for that export name. If there are multiple modules which satisfy the export, that means the export is ambiguous, and will result in a SyntaxError.

```js
import { a, b } from 'module.js'

print (a + b);
```

```js
// module.js
export var a = 2;
export { b } from 'module2.js'
```

```js
// module2.js
export var b = 40;
```

### Default imports and exports

Each module can optionally provide a single default export by using the `export default` statement. Default exports can either reference identifiers in the module's lexical environment, or be an anonymous default export, in which case they will only be accessible by an importing script.

```js
import defaultExport, { b as c } from 'module.js'

print (defaultExport); // 2
print (c ()); // 42
```

```js
// module.js
export default 2;
export function b () {
  return 42;
}
```

### Importing modules for side-effects

Evaluate a module without importing anything. Any errors encountered in the module will be propagated.

```js
import 'module.js' // > module.js
// "> module.js" is printed
b (); // (ReferenceError) b is not defined
```

```js
// module.js
export function b () {
  print ("> module.js");
  return 42;
}
b ();
```

## Unsupported features

* **snapshot**
