/*
 *  Author: pbrufal (and tican)
 *
 *  Declarations for Syscriptor.
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <fcntl.h>
#include <scsi/sg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>

#define INQ_REPLY_LEN 96
#define INQ_CMD_CODE 0x12
#define INQ_CMD_LEN 6

/* from $KERNEL/drivers/scsi/scsi.c */
char *scsi_types[] = {
	"Direct-Access",
	"Sequential-Access",
	"Printer",
	"Processor",
	"WORM",
	"CD-ROM",
	"Scanner",
	"Optical Device",
	"Medium Changer",
	"Communications",
	"Unknown",
	"Unknown",
	"Unknown",
	"Enclosure"
};

char *devices[] = {
	"/dev/sda", "/dev/sdb", "/dev/sdc", "/dev/sdd",
	"/dev/sde", "/dev/sdf", "/dev/sdg", "/dev/sdh",
	"/dev/scd0", "/dev/scd1", "/dev/sdc2", "/dev/scd3",
	"/dev/nst0", "/dev/nst1", "/dev/nst2", "/dev/nst3",
	"/dev/sg0", "/dev/sg1", "/dev/sg2", "/dev/sg3",
	"/dev/st0", "/dev/st1", "/dev/st2", "/dev/st3",
	"/dev/sr0", "/dev/sr1", "/dev/sr2", "/dev/srt"
};

/* based on scsiinfo.c, (c) Eric Youngdale */
int show_info(char *device)
{
	unsigned char inqCmdBlk[INQ_CMD_LEN] = {INQ_CMD_CODE, 0, 0, 0, INQ_REPLY_LEN, 0};
	unsigned char inqBuff[INQ_REPLY_LEN];
	unsigned char sense_buffer[32];
	sg_io_hdr_t io_hdr;
	int k, sg_fd;

	if ((sg_fd = open(device, O_RDONLY)) < 0) {
		return 0;
	}

	/* It is prudent to check we have a sg device by trying an ioctl */
	if ((ioctl(sg_fd, SG_GET_VERSION_NUM, &k) < 0) || (k < 30000)) {
		return 0;
	}

	/* Prepare INQUIRY command */
	memset(&io_hdr, 0, sizeof(sg_io_hdr_t));
	io_hdr.interface_id = 'S';
	io_hdr.cmd_len = sizeof(inqCmdBlk);
	io_hdr.mx_sb_len = sizeof(sense_buffer);
	io_hdr.dxfer_direction = SG_DXFER_FROM_DEV;
	io_hdr.dxfer_len = INQ_REPLY_LEN;
	io_hdr.dxferp = inqBuff;
	io_hdr.cmdp = inqCmdBlk;
	io_hdr.sbp = sense_buffer;
	io_hdr.timeout = 20000;

	if (ioctl(sg_fd, SG_IO, &io_hdr) < 0) {
		perror("sg_simple0: Inquiry SG_IO ioctl error");
		return 0;
	}

	if ((io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK) {
		if (io_hdr.sb_len_wr > 0) {
			printf("INQUIRY sense data: ");
			for (k = 0; k < io_hdr.sb_len_wr; ++k) {
				if ((k > 0) && (0 == (k % 10))) {
					printf("\n  ");
				}
				printf("0x%02x ", sense_buffer[k]);
			}
			printf("\n");
		}
		if (io_hdr.masked_status) {
			printf("INQUIRY SCSI status=0x%x\n", io_hdr.status);
		}
		if (io_hdr.host_status) {
			printf("INQUIRY host_status=0x%x\n", io_hdr.host_status);
		}
		if (io_hdr.driver_status) {
			printf("INQUIRY driver_status=0x%x\n", io_hdr.driver_status);
		}
	} else {
		unsigned char * p = (char *)inqBuff;
		int type = (*p) & 0x1f;
		printf("  %s:\n", device);
		printf("\tType:\t\t%s\n", scsi_types[type]);
		printf("\tVendor:\t\t%.8s\n", p + 8);
		printf("\tModel:\t\t%.16s\n", p + 16);
		printf("\tRevision:\t%.4s\n", p + 32);
	}
	close(sg_fd);
	return 1;
}

void scsi(const int BUF_SIZE)
{
	FILE *file;
	int found;
	size_t i;
	char buf[BUF_SIZE];
	char *card, *card2, *card3, *fname;

	fname = "/proc/pci";
	file = fopen(fname, "r");
	if ( !file ) {
		fprintf(stderr, "Could not open %s. Bailing out.\n", fname);
		return;
	}

	puts("");
        printf("  SCSI\n\n");

	printf("  SCSI Card:\n");
	found = 0;
	while ( fgets(buf, BUF_SIZE, file) ) {
		card = strstr(buf,"SCSI");
		if ( card ) {
			card2 = strchr(card, '(');
			if ( card2 ) {
				found=1;
				card2[-1] = 0;
				card3 = strchr(card,':')+1;
				printf(" %s\n", card3);
			}
		}
	}
	if ( !found ) {
		printf("  Could not find your SCSI card.\n\n");
		return;
	}

	printf("\n  SCSI Devices:\n\n");
	for (i = 0; i < sizeof(devices) / sizeof(char *); i++) {
		show_info(devices[i]);
	}

	puts("");
}
