//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************

// -------------------------------------------------------------------
//
// File name:     G4OpticalMaterialProperties.hh
//
// Author:        Daren Sawkey
//
// Creation date: 05.13.2021
//
// Description: Contains material properties for optical materials. These
// may be used in a user application instead of defining one's own. In user
// code, write for example:
// G4MaterialPropertiesTable* mpt = new G4MaterialPropertiesTable();
// mpt->AddProperty(keyname, materialname);
// where keyname is the key (e.g. "RINDEX") and materialname is the
// name of the material (e.g. "Air")
// The vectors are to be in order of increasing energy (as required by
// G4PhysicsVector. If wavelength is specified rather than energy,
// specify it in decreasing value and call ConvertToEnergy().
// -------------------------------------------------------------------

#ifndef G4OPTICALMATERIALPROPERTIES
#define G4OPTICALMATERIALPROPERTIES

#include "G4MaterialPropertyVector.hh"

#include <CLHEP/Units/PhysicalConstants.h>

#include <vector>

namespace G4OpticalMaterialProperties
{
void ConvertToEnergy(std::vector<G4double>& wavelength)
{
  for (auto& val : wavelength) {
    val = CLHEP::h_Planck * CLHEP::c_light / (val / 1000.);
  }
}

G4MaterialPropertyVector* GetRefractiveIndex(const G4String& material)
{
  G4MaterialPropertyVector* v = nullptr;
  if (material == "Air") {
    // P. E. Ciddor. Refractive index of air: new equations for the visible
    // and near infrared, Appl. Optics 35, 1566-1573 (1996) via
    // refractiveindex.info
    // clang-format off
    std::vector<G4double> wl = {
      1.69,   1.675,  1.661,  1.646,  1.632,  1.617,  1.602,  1.588,  1.573,
      1.559,  1.544,  1.529,  1.515,  1.5,    1.486,  1.471,  1.456,  1.442,
      1.427,  1.413,  1.398,  1.383,  1.369,  1.354,  1.34,   1.325,  1.31,
      1.296,  1.281,  1.267,  1.252,  1.237,  1.223,  1.208,  1.194,  1.179,
      1.164,  1.15,   1.135,  1.121,  1.106,  1.091,  1.077,  1.062,  1.048,
      1.033,  1.018,  1.004,  0.9892, 0.9746, 0.96,   0.9454, 0.9308, 0.9162,
      0.9016, 0.887,  0.8724, 0.8578, 0.8432, 0.8286, 0.814,  0.7994, 0.7848,
      0.7702, 0.7556, 0.741,  0.7264, 0.7118, 0.6972, 0.6826, 0.668,  0.6534,
      0.6388, 0.6242, 0.6096, 0.595,  0.5804, 0.5658, 0.5512, 0.5366, 0.522,
      0.5074, 0.4928, 0.4782, 0.4636, 0.449,  0.4344, 0.4198, 0.4052, 0.3906,
      0.376,  0.3614, 0.3468, 0.3322, 0.3176, 0.303,  0.2884, 0.2738, 0.2592,
      0.2446, 0.23
    };

    std::vector<G4double> ri = {
      1.0002731583221, 1.0002731680358, 1.0002731773414, 1.0002731875774,
      1.0002731973879, 1.0002732081845, 1.0002732192871, 1.000273229936,
      1.000273241664,  1.0002732529184, 1.0002732653197, 1.0002732780896,
      1.0002732903537, 1.0002733038787, 1.0002733168752, 1.0002733312162,
      1.000273346005,  1.0002733602287, 1.000273375938,  1.0002733910563,
      1.0002734077641, 1.0002734250218, 1.0002734416467, 1.000273460038,
      1.000273477767,  1.0002734973936, 1.0002735177028, 1.0002735373021,
      1.0002735590237, 1.0002735800022, 1.0002736032705, 1.0002736273968,
      1.0002736507265, 1.0002736766354, 1.0002737017104, 1.0002737295823,
      1.0002737585479, 1.0002737866205, 1.0002738178689, 1.0002738481835,
      1.0002738819616, 1.0002739171561, 1.0002739513533, 1.0002739895199,
      1.0002740266464, 1.0002740681302, 1.0002741114819, 1.0002741537295,
      1.0002742003794, 1.0002742485227, 1.0002742989071, 1.0002743516746,
      1.0002744069782, 1.0002744649838, 1.000274525871,  1.0002745898346,
      1.0002746570862, 1.0002747278561, 1.0002748023951, 1.0002748809769,
      1.0002749639005, 1.0002750514935, 1.0002751441149, 1.0002752421591,
      1.0002753460601, 1.0002754562966, 1.0002755733975, 1.0002756979483,
      1.0002758305987, 1.0002759720715, 1.0002761231724, 1.0002762848018,
      1.0002764579689, 1.0002766438078, 1.0002768435966, 1.0002770587802,
      1.0002772909975, 1.0002775421136, 1.0002778142591, 1.0002781098769,
      1.0002784317799, 1.0002787832208, 1.000279167979,  1.0002795904676,
      1.0002800558675, 1.0002805702966, 1.0002811410247, 1.000281776749,
      1.0002824879531, 1.0002832873764, 1.0002841906396, 1.0002852170881,
      1.0002863909483, 1.000287742941,  1.000289312581,  1.000291151532,
      1.0002933286388, 1.000295937727,  1.0002991101754, 1.0003030361805,
      1.0003080029552
    };
    // clang-format on
    ConvertToEnergy(wl);
    v = new G4MaterialPropertyVector(wl, ri);
  }
  else if (material == "Water") {
    // M. Daimon and A. Masumura. Measurement of the refractive index of
    // distilled water from the near-infrared region to the ultraviolet
    // region, Appl. Opt. 46, 3811-3820 (2007) via refractiveindex.info
    // clang-format off
    std::vector<G4double> wl = {
      1.129,  1.12,   1.11,   1.101,  1.091,  1.082,  1.072,  1.063,  1.053,
      1.044,  1.034,  1.025,  1.015,  1.006,  0.9964, 0.987,  0.9775, 0.968,
      0.9585, 0.9491, 0.9396, 0.9301, 0.9207, 0.9112, 0.9017, 0.8923, 0.8828,
      0.8733, 0.8638, 0.8544, 0.8449, 0.8354, 0.826,  0.8165, 0.807,  0.7976,
      0.7881, 0.7786, 0.7691, 0.7597, 0.7502, 0.7407, 0.7313, 0.7218, 0.7123,
      0.7029, 0.6934, 0.6839, 0.6744, 0.665,  0.6555, 0.646,  0.6366, 0.6271,
      0.6176, 0.6082, 0.5987, 0.5892, 0.5797, 0.5703, 0.5608, 0.5513, 0.5419,
      0.5324, 0.5229, 0.5135, 0.504,  0.4945, 0.485,  0.4756, 0.4661, 0.4566,
      0.4472, 0.4377, 0.4282, 0.4188, 0.4093, 0.3998, 0.3903, 0.3809, 0.3714,
      0.3619, 0.3525, 0.343,  0.3335, 0.3241, 0.3146, 0.3051, 0.2956, 0.2862,
      0.2767, 0.2672, 0.2578, 0.2483, 0.2388, 0.2294, 0.2199, 0.2104, 0.2009,
      0.1915, 0.182
    };

    std::vector<G4double> ri = {
      1.3235601610672, 1.3236962786529, 1.3238469492274, 1.3239820826015,
      1.3241317601229, 1.3242660923031, 1.3244149850321, 1.3245487081924,
      1.3246970353146, 1.3248303521764, 1.3249783454392, 1.3251114708334,
      1.3252593763883, 1.3253925390161, 1.3255346928953, 1.3256740639273,
      1.3258151661284, 1.3259565897464, 1.326098409446,  1.3262392023332,
      1.32638204417,   1.3265255240887, 1.3266682080154, 1.3268132228682,
      1.3269591507928, 1.32710453999,   1.3272525883205, 1.3274018651452,
      1.3275524865531, 1.3277029655807, 1.3278566311755, 1.3280120256415,
      1.328167625867,  1.3283268916356, 1.3284883366632, 1.3286503921034,
      1.3288166823394, 1.3289856845931, 1.3291575989438, 1.3293307783594,
      1.3295091314406, 1.329691073075,  1.3298748828499, 1.3300647424335,
      1.330259008797,  1.3304558735667, 1.3306598562207, 1.3308692454666,
      1.3310844250714, 1.3313034432243, 1.3315313994219, 1.3317664745307,
      1.3320065870964, 1.3322573970809, 1.3325169923974, 1.3327831408348,
      1.3330622051201, 1.3333521716563, 1.3336538750639, 1.3339648469612,
      1.334292688017,  1.3346352438404, 1.3349898436519, 1.3353653263299,
      1.3357594410975, 1.3361692982684, 1.3366053508081, 1.3370652823778,
      1.3375512404603, 1.3380600434506, 1.3386051585073, 1.3391843066628,
      1.3397941348754, 1.34045134693,   1.3411539035636, 1.341898413271,
      1.3427061376724, 1.3435756703017, 1.3445141685829, 1.3455187528254,
      1.3466202523109, 1.3478194943997, 1.3491150472655, 1.350549622307,
      1.3521281492629, 1.3538529543346, 1.3557865447701, 1.3579431129972,
      1.3603615197762, 1.3630595401556, 1.3661548299831, 1.3696980785677,
      1.3737440834249, 1.3785121412586, 1.3841454790718, 1.3908241012126,
      1.399064758142,  1.4093866965284, 1.422764121467,  1.4407913910231,
      1.4679465862259
    };
    // clang-format on
    ConvertToEnergy(wl);
    v = new G4MaterialPropertyVector(wl, ri);
  }
  else if (material == "PMMA") {
    // G. Beadie, M. Brindza, R. A. Flynn, A. Rosenberg, and J. S. Shirk.
    // Refractive index measurements of poly(methyl methacrylate) (PMMA) from
    // 0.4-1.6 μm, Appl. Opt. 54, F139-F143 (2015) via refractiveindex.info
    // clang-format off
    std::vector<G4double> wl = {
      1.620, 1.608, 1.596, 1.584, 1.572, 1.560, 1.548, 1.536, 1.524, 1.512,
      1.500, 1.488, 1.476, 1.464, 1.452, 1.440, 1.428, 1.416, 1.404, 1.392,
      1.380, 1.368, 1.356, 1.344, 1.332, 1.320, 1.308, 1.296, 1.284, 1.272,
      1.260, 1.248, 1.236, 1.224, 1.212, 1.200, 1.188, 1.176, 1.164, 1.152,
      1.140, 1.128, 1.116, 1.104, 1.092, 1.080, 1.068, 1.056, 1.044, 1.032,
      1.020, 1.008, 0.996, 0.984, 0.972, 0.960, 0.948, 0.936, 0.924, 0.912,
      0.900, 0.888, 0.876, 0.864, 0.852, 0.840, 0.828, 0.816, 0.804, 0.792,
      0.780, 0.768, 0.756, 0.744, 0.732, 0.720, 0.708, 0.696, 0.684, 0.672,
      0.660, 0.648, 0.636, 0.624, 0.612, 0.600, 0.588, 0.576, 0.564, 0.552,
      0.540, 0.528, 0.516, 0.504, 0.492, 0.480, 0.468, 0.456, 0.444, 0.432,
      0.420
    };

    std::vector<G4double> ri = {
      1.4805336370194, 1.4805958419707, 1.4806572042606, 1.4807177801532,
      1.4807776264012, 1.4808368002755, 1.4808953595962, 1.4809533627658,
      1.4810108688022, 1.4810679373753, 1.4811246288434, 1.4811810042925,
      1.4812371255769, 1.4812930553619, 1.4813488571684, 1.4814045954196,
      1.4814603354903, 1.4815161437579, 1.4815720876568, 1.4816282357347,
      1.4816846577124, 1.4817414245457, 1.4817986084914, 1.4818562831765,
      1.4819145236704, 1.4819734065613, 1.4820330100373, 1.4820934139706,
      1.4821547000072, 1.4822169516613, 1.4822802544152, 1.4823446958243,
      1.4824103656291, 1.4824773558724, 1.4825457610252, 1.4826156781186,
      1.482687206885,  1.4827604499074, 1.4828355127788, 1.4829125042715,
      1.4829915365184, 1.4830727252059, 1.483156189781,  1.4832420536723,
      1.4833304445284, 1.4834214944731, 1.483515340381,  1.4836121241751,
      1.483711993148,  1.4838151003112, 1.4839216047736, 1.4840316721549,
      1.4841454750362, 1.4842631934547, 1.4843850154458, 1.484511137641,
      1.4846417659288, 1.4847771161864, 1.4849174150935, 1.4850629010407,
      1.4852138251453, 1.4853704523922, 1.4855330629191, 1.4857019534694,
      1.4858774390395, 1.486059854753,  1.4862495580004, 1.4864469308883,
      1.4866523830528, 1.4868663548994, 1.4870893213462, 1.4873217961577,
      1.4875643369775, 1.4878175511839, 1.4880821027184, 1.4883587200625,
      1.4886482055708, 1.4889514464045, 1.4892694273501, 1.4896032458562,
      1.4899541296675, 1.4903234574922, 1.4907127831856, 1.491123863976,
      1.4915586932831, 1.4920195386642, 1.4925089853408, 1.4930299855674,
      1.4935859137206, 1.4941806263051, 1.494818524906,  1.4955046181841,
      1.4962445758678, 1.4970447626402, 1.4979122317605, 1.4988546454557,
      1.4998800687986, 1.5009965515508, 1.5022113611478, 1.5035296480805,
      1.5049521933717
    };
    // clang-format on

    ConvertToEnergy(wl);
    v = new G4MaterialPropertyVector(wl, ri);
  }
  else if (material == "Fused Silica") {
    // 1) I. H. Malitson. Interspecimen comparison of the refractive index of
    // fused silica, J. Opt. Soc. Am. 55, 1205-1208 (1965)  2) C. Z. Tan.
    // Determination of refractive index of silica glass for infrared
    // wavelengths by IR spectroscopy, J. Non-Cryst. Solids 223, 158-163 (1998)
    // Sellmeier formula is reported in Ref. 1 for the 0.21-3.71 um wavelength
    // range. Ref. 2 verifies the validity of the formula up to 6.7 um.
    // via refractiveindex.info
    // clang-format off
    std::vector<G4double> wl = {
      6.700,  6.472,  6.252,  6.039,  5.833,  5.635,  5.443,  5.258,  5.079,
      4.906,  4.739,  4.578,  4.422,  4.271,  4.126,  3.986,  3.850,  3.719,
      3.592,  3.470,  3.352,  3.238,  3.128,  3.021,  2.918,  2.819,  2.723,
      2.630,  2.541,  2.454,  2.371,  2.290,  2.212,  2.137,  2.064,  1.994,
      1.926,  1.861,  1.797,  1.736,  1.677,  1.620,  1.565,  1.512,  1.460,
      1.410,  1.362,  1.316,  1.271,  1.228,  1.186,  1.146,  1.107,  1.069,
      1.033,  0.9976, 0.9636, 0.9308, 0.8992, 0.8686, 0.8390, 0.8104, 0.7829,
      0.7562, 0.7305, 0.7056, 0.6816, 0.6584, 0.6360, 0.6143, 0.5934, 0.5732,
      0.5537, 0.5349, 0.5167, 0.4991, 0.4821, 0.4657, 0.4498, 0.4345, 0.4197,
      0.4055, 0.3917, 0.3783, 0.3655, 0.3530, 0.3410, 0.3294, 0.3182, 0.3074,
      0.2969, 0.2868, 0.2770, 0.2676, 0.2585, 0.2497, 0.2412, 0.2330, 0.2251,
      0.2174, 0.2100
    };

    std::vector<G4double> ri = {
      1.1596494139777, 1.1973256716307, 1.2280888354422, 1.2537289561387,
      1.2753723963511, 1.2937460280032, 1.3096384003386, 1.3234105439689,
      1.3354823573874, 1.3461171232165, 1.3555262189157, 1.36388143366,
      1.3713701305288, 1.3780997735118, 1.3841208059058, 1.3895553417944,
      1.3945035722002, 1.3989819997456, 1.4030708962299, 1.4067782146466,
      1.4101696832452, 1.4132741569442, 1.416117293379,  1.4187459519934,
      1.4211544161909, 1.4233613684394, 1.4254044329951, 1.427296093753,
      1.4290287283802, 1.4306516985468, 1.4321372109342, 1.433529881253,
      1.4348196176837, 1.4360139618555, 1.4371349603433, 1.4381729429417,
      1.43914806608,   1.4400509385522, 1.4409133669095, 1.4417121733672,
      1.4424645759259, 1.4431739285381, 1.4438434020915, 1.4444759883488,
      1.4450861470109, 1.445664578157,  1.4462138519129, 1.4467363524093,
      1.4472455774929, 1.4477322458328, 1.4482096590065, 1.448668310738,
      1.4491214616538, 1.4495710901504, 1.4500069615101, 1.450447735732,
      1.4508853971281, 1.4513240787777, 1.4517653834747, 1.4522138205731,
      1.4526712743322, 1.4531396102638, 1.4536188536247, 1.4541161650842,
      1.4546298754538, 1.4551660299221, 1.4557246986958, 1.4563104080175,
      1.4569256013294, 1.4575758068817, 1.4582607881284, 1.4589865613939,
      1.4597562854765, 1.4605730794883, 1.4614449911601, 1.4623764385944,
      1.4633719346282, 1.4644360310913, 1.465580829975,  1.4668048202486,
      1.4681218218832, 1.4695286500209, 1.4710525123802, 1.4727046797948,
      1.4744682820342, 1.4763951298847, 1.4784676522483, 1.4807144415912,
      1.4831504333467, 1.4857914366574, 1.488683281387,  1.4918215034661,
      1.495262719426,  1.498999218542,  1.5031009629039, 1.5076095872199,
      1.5125721155558, 1.5180417677275, 1.5240789072975, 1.530846431063,
      1.5383576204905
    };
    // clang-format on

    ConvertToEnergy(wl);
    v = new G4MaterialPropertyVector(wl, ri);
  }

  else {
    G4ExceptionDescription ed;
    ed << "Material " << material << " not found.";
    G4Exception("G4OpticalMaterialProperties::GetRefractiveIndex", "mat400", FatalException, ed);
    return nullptr;
  }

  return v;
}

G4MaterialPropertyVector* GetProperty(const G4String& key, const G4String& mat)
{
  if (key == "RINDEX") {
    return GetRefractiveIndex(mat);
  }

  G4ExceptionDescription ed;
  ed << "Property name " << key << " doesn't exist in optical properties data table.";
  G4Exception("G4OpticalMaterialProperties::GetProperty", "mat401", FatalException, ed);
  return nullptr;
}

}  // namespace G4OpticalMaterialProperties

#endif
