﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cleanroomsml/CleanRoomsMLRequest.h>
#include <aws/cleanroomsml/CleanRoomsML_EXPORTS.h>
#include <aws/cleanroomsml/model/AudienceGenerationJobDataSource.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CleanRoomsML {
namespace Model {

/**
 */
class StartAudienceGenerationJobRequest : public CleanRoomsMLRequest {
 public:
  AWS_CLEANROOMSML_API StartAudienceGenerationJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartAudienceGenerationJob"; }

  AWS_CLEANROOMSML_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the audience generation job.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  StartAudienceGenerationJobRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the configured audience model that is used
   * for this audience generation job.</p>
   */
  inline const Aws::String& GetConfiguredAudienceModelArn() const { return m_configuredAudienceModelArn; }
  inline bool ConfiguredAudienceModelArnHasBeenSet() const { return m_configuredAudienceModelArnHasBeenSet; }
  template <typename ConfiguredAudienceModelArnT = Aws::String>
  void SetConfiguredAudienceModelArn(ConfiguredAudienceModelArnT&& value) {
    m_configuredAudienceModelArnHasBeenSet = true;
    m_configuredAudienceModelArn = std::forward<ConfiguredAudienceModelArnT>(value);
  }
  template <typename ConfiguredAudienceModelArnT = Aws::String>
  StartAudienceGenerationJobRequest& WithConfiguredAudienceModelArn(ConfiguredAudienceModelArnT&& value) {
    SetConfiguredAudienceModelArn(std::forward<ConfiguredAudienceModelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The seed audience that is used to generate the audience.</p>
   */
  inline const AudienceGenerationJobDataSource& GetSeedAudience() const { return m_seedAudience; }
  inline bool SeedAudienceHasBeenSet() const { return m_seedAudienceHasBeenSet; }
  template <typename SeedAudienceT = AudienceGenerationJobDataSource>
  void SetSeedAudience(SeedAudienceT&& value) {
    m_seedAudienceHasBeenSet = true;
    m_seedAudience = std::forward<SeedAudienceT>(value);
  }
  template <typename SeedAudienceT = AudienceGenerationJobDataSource>
  StartAudienceGenerationJobRequest& WithSeedAudience(SeedAudienceT&& value) {
    SetSeedAudience(std::forward<SeedAudienceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether the seed audience is included in the audience generation output.</p>
   */
  inline bool GetIncludeSeedInOutput() const { return m_includeSeedInOutput; }
  inline bool IncludeSeedInOutputHasBeenSet() const { return m_includeSeedInOutputHasBeenSet; }
  inline void SetIncludeSeedInOutput(bool value) {
    m_includeSeedInOutputHasBeenSet = true;
    m_includeSeedInOutput = value;
  }
  inline StartAudienceGenerationJobRequest& WithIncludeSeedInOutput(bool value) {
    SetIncludeSeedInOutput(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the collaboration that contains the audience generation
   * job.</p>
   */
  inline const Aws::String& GetCollaborationId() const { return m_collaborationId; }
  inline bool CollaborationIdHasBeenSet() const { return m_collaborationIdHasBeenSet; }
  template <typename CollaborationIdT = Aws::String>
  void SetCollaborationId(CollaborationIdT&& value) {
    m_collaborationIdHasBeenSet = true;
    m_collaborationId = std::forward<CollaborationIdT>(value);
  }
  template <typename CollaborationIdT = Aws::String>
  StartAudienceGenerationJobRequest& WithCollaborationId(CollaborationIdT&& value) {
    SetCollaborationId(std::forward<CollaborationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the audience generation job.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  StartAudienceGenerationJobRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional metadata that you apply to the resource to help you categorize
   * and organize them. Each tag consists of a key and an optional value, both of
   * which you define.</p> <p>The following basic restrictions apply to tags:</p>
   * <ul> <li> <p>Maximum number of tags per resource - 50.</p> </li> <li> <p>For
   * each resource, each tag key must be unique, and each tag key can have only one
   * value.</p> </li> <li> <p>Maximum key length - 128 Unicode characters in
   * UTF-8.</p> </li> <li> <p>Maximum value length - 256 Unicode characters in
   * UTF-8.</p> </li> <li> <p>If your tagging schema is used across multiple services
   * and resources, remember that other services may have restrictions on allowed
   * characters. Generally allowed characters are: letters, numbers, and spaces
   * representable in UTF-8, and the following characters: + - = . _ : / @.</p> </li>
   * <li> <p>Tag keys and values are case sensitive.</p> </li> <li> <p>Do not use
   * aws:, AWS:, or any upper or lowercase combination of such as a prefix for keys
   * as it is reserved for AWS use. You cannot edit or delete tag keys with this
   * prefix. Values can have this prefix. If a tag value has aws as its prefix but
   * the key does not, then Clean Rooms ML considers it to be a user tag and will
   * count against the limit of 50 tags. Tags with only the key prefix of aws do not
   * count against your tags per resource limit.</p> </li> </ul>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  StartAudienceGenerationJobRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  StartAudienceGenerationJobRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_configuredAudienceModelArn;
  bool m_configuredAudienceModelArnHasBeenSet = false;

  AudienceGenerationJobDataSource m_seedAudience;
  bool m_seedAudienceHasBeenSet = false;

  bool m_includeSeedInOutput{false};
  bool m_includeSeedInOutputHasBeenSet = false;

  Aws::String m_collaborationId;
  bool m_collaborationIdHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CleanRoomsML
}  // namespace Aws
