﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudformation/CloudFormationRequest.h>
#include <aws/cloudformation/CloudFormation_EXPORTS.h>
#include <aws/cloudformation/model/RegistryType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudFormation {
namespace Model {

/**
 */
class DescribeTypeRequest : public CloudFormationRequest {
 public:
  AWS_CLOUDFORMATION_API DescribeTypeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeType"; }

  AWS_CLOUDFORMATION_API Aws::String SerializePayload() const override;

 protected:
  AWS_CLOUDFORMATION_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The kind of extension.</p> <p>Conditional: You must specify either
   * <code>TypeName</code> and <code>Type</code>, or <code>Arn</code>.</p>
   */
  inline RegistryType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(RegistryType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline DescribeTypeRequest& WithType(RegistryType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the extension.</p> <p>Conditional: You must specify either
   * <code>TypeName</code> and <code>Type</code>, or <code>Arn</code>.</p>
   */
  inline const Aws::String& GetTypeName() const { return m_typeName; }
  inline bool TypeNameHasBeenSet() const { return m_typeNameHasBeenSet; }
  template <typename TypeNameT = Aws::String>
  void SetTypeName(TypeNameT&& value) {
    m_typeNameHasBeenSet = true;
    m_typeName = std::forward<TypeNameT>(value);
  }
  template <typename TypeNameT = Aws::String>
  DescribeTypeRequest& WithTypeName(TypeNameT&& value) {
    SetTypeName(std::forward<TypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the extension.</p> <p>Conditional: You must
   * specify either <code>TypeName</code> and <code>Type</code>, or
   * <code>Arn</code>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  DescribeTypeRequest& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a specific version of the extension. The version ID is the value at
   * the end of the Amazon Resource Name (ARN) assigned to the extension version when
   * it is registered.</p> <p>If you specify a <code>VersionId</code>,
   * <code>DescribeType</code> returns information about that specific extension
   * version. Otherwise, it returns information about the default extension
   * version.</p>
   */
  inline const Aws::String& GetVersionId() const { return m_versionId; }
  inline bool VersionIdHasBeenSet() const { return m_versionIdHasBeenSet; }
  template <typename VersionIdT = Aws::String>
  void SetVersionId(VersionIdT&& value) {
    m_versionIdHasBeenSet = true;
    m_versionId = std::forward<VersionIdT>(value);
  }
  template <typename VersionIdT = Aws::String>
  DescribeTypeRequest& WithVersionId(VersionIdT&& value) {
    SetVersionId(std::forward<VersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The publisher ID of the extension publisher.</p> <p>Extensions provided by
   * Amazon Web Services are not assigned a publisher ID.</p>
   */
  inline const Aws::String& GetPublisherId() const { return m_publisherId; }
  inline bool PublisherIdHasBeenSet() const { return m_publisherIdHasBeenSet; }
  template <typename PublisherIdT = Aws::String>
  void SetPublisherId(PublisherIdT&& value) {
    m_publisherIdHasBeenSet = true;
    m_publisherId = std::forward<PublisherIdT>(value);
  }
  template <typename PublisherIdT = Aws::String>
  DescribeTypeRequest& WithPublisherId(PublisherIdT&& value) {
    SetPublisherId(std::forward<PublisherIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version number of a public third-party extension.</p>
   */
  inline const Aws::String& GetPublicVersionNumber() const { return m_publicVersionNumber; }
  inline bool PublicVersionNumberHasBeenSet() const { return m_publicVersionNumberHasBeenSet; }
  template <typename PublicVersionNumberT = Aws::String>
  void SetPublicVersionNumber(PublicVersionNumberT&& value) {
    m_publicVersionNumberHasBeenSet = true;
    m_publicVersionNumber = std::forward<PublicVersionNumberT>(value);
  }
  template <typename PublicVersionNumberT = Aws::String>
  DescribeTypeRequest& WithPublicVersionNumber(PublicVersionNumberT&& value) {
    SetPublicVersionNumber(std::forward<PublicVersionNumberT>(value));
    return *this;
  }
  ///@}
 private:
  RegistryType m_type{RegistryType::NOT_SET};
  bool m_typeHasBeenSet = false;

  Aws::String m_typeName;
  bool m_typeNameHasBeenSet = false;

  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  Aws::String m_versionId;
  bool m_versionIdHasBeenSet = false;

  Aws::String m_publisherId;
  bool m_publisherIdHasBeenSet = false;

  Aws::String m_publicVersionNumber;
  bool m_publicVersionNumberHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
