#
# Copyright 2003 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

from wxPython.wx import wxPoint
from math import pi, sin, cos

# PostScript style 2D transformation matrix

# mathematically a 3-by-3 matrix:
#
#  a  b  0
#  c  d  0
#  tx ty 1
#
#
# we represent this as a 6-element list:
#   0  1  2  3  4  5 
# [ a  b  c  d tx ty ]
#
# translation by (tx, ty) is described by
#
#   1  0  0
#   0  1  0
#  tx ty  1
#
#
# scaling by sx, sy is described by
#
#  sx  0  0
#   0 sy  0
#   0  0  1
#
#
# rotations counterclockwise about the origin by an angle A
# is described by:
#
#  cos(A) sin(A) 0
# -sin(A) cos(A) 0
#    0     0     1
#
# these transformations are applied like this:
#
# newCTM = transform x originalCTM


def _matrixmult (a,b):
    c = [ 0, 0, 0, 0, 0, 0 ]
    c[0] = a[0]*b[0] + a[1]*b[2]
    c[1] = a[0]*b[1] + a[1]*b[3]
    c[2] = a[2]*b[0] + a[3]*b[2]
    c[3] = a[2]*b[1] + a[3]*b[3]
    c[4] = a[4]*b[0] + a[5]*b[2] + b[4]
    c[5] = a[4]*b[1] + a[5]*b[3] + b[5]
    return c

class tmatrix (object):
    def __init__ (self):
        self.ctm = [ 1, 0, 0, 1, 0, 0 ] # identity matrix

    def map_xy (self, x, y):
        m = self.ctm
        # x' = a*x + c*y + tx
        xprime = m[0]*x + m[2]*y + m[4]
        # y' = b*x + d*y + ty
        yprime = m[1]*x + m[3]*y + m[5]
        return (xprime, yprime)

    def map_tuple (self, t):
        m = self.ctm
        x = t[0]
        y = t[1]
        # x' = a*x + c*y + tx
        xprime = m[0]*x + m[2]*y + m[4]
        # y' = b*x + d*y + ty
        yprime = m[1]*x + m[3]*y + m[5]
        return (min (xprime, 32767), yprime)

    def map_point (self, point):
        (x, y) = self.map_xy (point.x, point.y)
        # the min is to keep nasty wrap-around from occuring when converting
        # from floats bigger than 32767...
        # FIXME this just fixes the problem we're seeing.  In general this should
        #   saturate x and y before conversion
        return wxPoint(min (x, 32767), y)
    
    def map_point_list (self, list_of_points):
        # return map (self.map_point, list_of_points)
        return map (self.map_tuple, list_of_points)
    
    def translate (self, dx, dy):
        self.ctm = _matrixmult ([1, 0, 0, 1, dx, dy], self.ctm)

    def scale (self, sx, sy):
        self.ctm = _matrixmult ([sx, 0, 0, sy, 0, 0], self.ctm)

    def rotate (self, angle_in_degrees):
        radians = angle_in_degrees * (pi / 180)
        sinA = sin (radians)
        cosA = cos (radians)
        self.ctm = _matrixmult([cosA, sinA, -sinA, cosA, 0, 0], self.ctm)

    def concat (self, tmat):            
        self.ctm = _matrixmult (tmat.ctm, self.ctm) # FIXME check order
        
    
