/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRFHFSKDEMOD_H_
#define _VRFHFSKDEMOD_H_

// need to manually adjust threshold
#define FH_FSK_THRESHOLD 5000

#include <VrSigProc.h>
#include <VrHoppingComplexFIRfilter.h>

template<class iType> 
class VrFHFSKDemod : public VrSigProc<iType,bit> {
protected:
  VrHoppingComplexFIRfilter<char>* upStreamMod;
  int bitLength;
  int dataRate;
  int seqLen;
  int currSeqNum;
  int* hopSequence;
  int bitsPerHop;
  int syncStatus;
  int bitsDecoded;
  int hopCount;
  int bitsPerSyncCheck;
  virtual void initialize();
public:   
  virtual void work(int n);  
  VrFHFSKDemod(VrHoppingComplexFIRfilter<char>*,int,int*,int,int);
};

template<class iType> void
VrFHFSKDemod<iType>::work(int n)
{
  static int lostsync = 0;
  static int bitsBetweenSync = 0;

  static int below_threshold = 0;
  iType prevVal, inputVal;
  float a,b,x,y;
  float imaginary;
  int thresh_count=0;
  int zero_count = 0;
  int step_count = 0;
  static int bitsSinceSyncCheck = 0;

  while (n > 0) {
    if (!syncStatus) {
      
      float base = 0.0;
      do {
	incReadPtr(bitLength);

	inputVal = inputRead(0);
	a = real(inputVal);
	b = imag(inputVal);

	if ((a*a+b*b) > FH_FSK_THRESHOLD) {
	  thresh_count++;
	  prevVal = inputRead(-1);
	  x = real(prevVal);
	  y = imag(prevVal);
	  base = x*b - a*y;
	  if (base < 0) zero_count++;
	  else zero_count = 0;
	}
	else {
	  thresh_count = 0;
	  zero_count = 0;
	}
	//	printf("%f\n",abs(a*a+b*b));
      }	while (thresh_count < 5);

      imaginary = base;
      if (base > 0) {
	while (imaginary > 0) {
	  incReadPtr(1);
	  prevVal = inputVal;
	  inputVal = inputRead();
	  x = a;
	  y = b;
	  a = real(inputVal);
	  b = imag(inputVal);
	  imaginary = x*b - a*y;
	}
      }
      else {
	while (imaginary < 0) {
	  step_count++;
	  incReadPtr(1);
	  prevVal = inputVal;
	  inputVal = inputRead();
	  x = a;
	  y = b;
	  a = real(inputVal);
	  b = imag(inputVal);
	  imaginary = x*b - a*y;
	}     
      }
      // found bit boundary, output the zeros that have been previously saved 
      int zeros = zero_count + (step_count/bitLength);
      if (zeros > 0) {
	if (zeros==14) zeros=15; /** stupid hack, sometimes not catching all
				   15 of the zeros in training seq.
				   Same hack in VrFHFSKDemod.h **/
	if (zeros > 15) zeros = 15; /** don't want to overload output buffer*/

	for (int i=0;i<zeros;i++)
	  outputWrite(0);
	outputWrite(1);
	n -= zeros+1;
	bitsDecoded+=zeros+1;
	bitsBetweenSync+=zeros+1;
      }
      
      thresh_count = 0;
      zero_count = 0;
      step_count = 0;

      incReadPtr(3*bitLength/2);  // move to center of a bit

      syncStatus = 1;      
      printf("got sync, zeros = %d\n",zeros);
    }
    
    /** main loop, synched up and free running **/
    
    /** need to only look at img part to see if the freq is + or - **/
    prevVal = inputRead(-1);
    inputVal = inputRead(0);
    
    // float multiply
    a = real(inputVal);
    b = imag(inputVal);
    x = real(prevVal);
    y = imag(prevVal);
    
    imaginary = x*b - a*y;
    //    printf("%f %f \n",y/(x+.0000001),b/(a+.0000001));
    //    printf("%f %f %f %f %f\n",a*a+b*b,a,b,x,y);
    if ((a*a+b*b) < FH_FSK_THRESHOLD) {
      below_threshold++;
      //      printf("losing sync: %d %f\n",below_threshold,abs(a*a+b*b));
    }
    else below_threshold = 0;

    if (below_threshold >= 3) {
      below_threshold = 0;
      syncStatus=0;
      printf("lost sync (%d), total bits = %d, sending %d nulls\n",++lostsync,
	     bitsBetweenSync,n);
      bitsBetweenSync = 0;
      /*  need to send NULL bits to signal end/loss of transmission */
      bit* nullbits = (bit*)new bit[n];
      memset(nullbits,-1,n);
      outputWrite(nullbits,n);
      n=0;
      bitsDecoded = 0;
      hopCount = 0;
      upStreamMod->changeChannel(hopSequence[0]);
      printf("reset to %d\n",hopSequence[hopCount]);
    }
    else
      {
	if (imaginary > 0) {
	  //	  printf("1 (%d) : %f %f %f\n",bitsDecoded,abs(x*x+y*y),abs(a*a+b*b),
	  //		 imaginary);
	  outputWrite(1); // positive freq
	}
	else {
	  //	  printf("0 (%d) : %f %f %f\n",bitsDecoded,abs(x*x+y*y),abs(a*a+b*b),
	  //		 imaginary);
	  outputWrite(0); // negative freq
	}
	n--;
	bitsDecoded++;
	bitsSinceSyncCheck++;
	bitsBetweenSync++;
      }

    // make sure we're still synchronized
    if ((bitsSinceSyncCheck >= bitsPerSyncCheck) && (imaginary > 0)) {
      /* if it's time to check and previous bit was a 1
	 to check sync, need a bit transition (0-1) or (1-0). 
	   (0-0),(1-1) are bad.  Since (0-0) probably occurs more often
	   than (1-1), (e.g. long strings of 0's are more likely than 1's)
	   just look for (1-0) */
      for(int i=0;i<bitLength;i++) {
	incReadPtr(1);
	prevVal = inputVal;
	inputVal = inputRead();
	x = real(prevVal);
	y = imag(prevVal);
	a = real(inputVal);
	b = imag(inputVal);
	imaginary = x*b - a*y;
	if (imaginary < 0) {  // found (1-0) transition
	  //	  printf("syncCheck: %d\n",i);
	  bitsSinceSyncCheck = 0;  // reset counter
	  incReadPtr(bitLength/2);  // advance to middle of bit
	  break;
	}
      }
    } else
      incReadPtr(bitLength);  // not checking sync , advance forward one bit

    // check to see if hopping is needed
    if (bitsDecoded == bitsPerHop) {  // hop!
      bitsDecoded = 0;
      hopCount++;
      if (hopCount == seqLen) hopCount = 0;
      upStreamMod->changeChannel(hopSequence[hopCount]);
      //      printf("hopped to %d\n",hopSequence[hopCount]);
    }

  }
  return;
}

template<class iType> void
VrFHFSKDemod<iType>::initialize() {
  int sf = getInputSamplingFrequencyN(0);
  bitLength = sf / dataRate;
  setHistory (2);
  printf("bitLength = %d\n",bitLength);
  if ((bitLength * dataRate) != sf) {
    dataRate = sf/bitLength;
    cout << "Warning: Adjusting data rate to: " << dataRate << endl;
  }
}

template<class iType> 
VrFHFSKDemod<iType>::VrFHFSKDemod(VrHoppingComplexFIRfilter<char>* u,
				  int seq_len, int* seq, int bit_rate, int bph)
  :dataRate(bit_rate),seqLen(seq_len),currSeqNum(0),
   bitsPerHop(bph),syncStatus(0),bitsDecoded(0),hopCount(0),
   bitsPerSyncCheck(bit_rate/10)
{
  upStreamMod = u;
  hopSequence = seq;
  setOutputSize(16);
}

#endif




