--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_wads_in(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wads_in';
BEGIN
  PERFORM create_partitioned_table(
     'CREATE TABLE %I '
     '(wad_in_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
     ',wad_id BYTEA PRIMARY KEY CHECK (LENGTH(wad_id)=24)'
     ',origin_exchange_url TEXT NOT NULL'
     ',amount taler_amount NOT NULL'
     ',arrival_time INT8 NOT NULL'
     ',UNIQUE (wad_id, origin_exchange_url)'
     ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wad_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Incoming exchange-to-exchange wad wire transfers'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Unique identifier of the wad, part of the wire transfer subject'
    ,'wad_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Base URL of the originating URL, also part of the wire transfer subject'
    ,'origin_exchange_url'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Actual amount that was received by our exchange'
    ,'amount'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the wad was received'
    ,'arrival_time'
    ,table_name
    ,partition_suffix
  );
END $$;


CREATE FUNCTION constrain_table_wads_in(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wads_in';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_in_serial_id_key'
    ' UNIQUE (wad_in_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_wad_is_origin_exchange_url_key'
    ' UNIQUE (wad_id, origin_exchange_url) '
  );
END $$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wads_in'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wads_in'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
