#include <stdlib.h>

#include <qfileinfo.h>
#include <qlayout.h>
#include <qscrollview.h>
#include <qscrollbar.h>
#include <qtimer.h>
#include <qpainter.h>
#include <qprinter.h>
#include <qprintdialog.h>


#include <kaboutdialog.h>
#include <kaccel.h>
#include <kaction.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kconfigdialog.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kfilterbase.h>
#include <kfilterdev.h>
#include <kglobal.h>
#include <kinstance.h>
#include <kio/job.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kparts/componentfactory.h>
#include <kparts/genericfactory.h>
#include <kprogress.h>
#include <kstandarddirs.h>
#include <kstdaction.h>
#include <ktempfile.h>
#include <ktrader.h>
#include <kinputdialog.h>


#include <stdlib.h>
#include <unistd.h>
#include <math.h>

#include "centeringScrollview.h"
#include "documentRenderer.h"
#include "kviewpart.moc"
#include "pageSize.h"
#include "pageSizeDialog.h"

#include "optionDialogGUIWidget_base.h"

#include "kvsprefs.h"

#define MULTIPAGE_VERSION 1

typedef KParts::GenericFactory<KViewPart> KViewPartFactory;
K_EXPORT_COMPONENT_FACTORY(kviewerpart, KViewPartFactory)

KViewPart::KViewPart(QWidget *parentWidget, const char *widgetName, QObject *parent,
                     const char *name, const QStringList& args)
  : KViewPart_Iface(parent, name), showSidebar(0), saveAction(0), multiPage(0),
    multiPageLibrary(QString::null), aboutDialog(0)
{
  KGlobal::locale()->insertCatalogue("kviewshell");

  tmpUnzipped = 0L;
  pageChangeIsConnected = false;
  setInstance(KViewPartFactory::instance());

  watch = KDirWatch::self();
  connect(watch, SIGNAL(dirty(const QString&)), this, SLOT(fileChanged(const QString&)));
  watch->startScan();

  mainWidget = new QWidget(parentWidget, widgetName);
  mainWidget->setFocusPolicy(QWidget::StrongFocus);
  setWidget(mainWidget);

  mainLayout = new QHBoxLayout(mainWidget, 0, 0);

  // create the displaying part

  // Search for service
  KTrader::OfferList offers;

  if (!args.isEmpty())
  {
    // If a default MimeType is specified try to load a MultiPage supporting it.
    QString defaultMimeType = args.first();
    offers = KTrader::self()->query(
        QString::fromLatin1("KViewShell/MultiPage" ),
        QString("([X-KDE-MultiPageVersion] == %1) and "
                "([X-KDE-MimeTypes] == '%2')").arg(MULTIPAGE_VERSION).arg(defaultMimeType));
  }

  // If no default MimeType is given or no MultiPage has been found, try to load the Empty MultiPage.
  if (offers.isEmpty())
  {
    offers = KTrader::self()->query(
        QString::fromLatin1("KViewShell/MultiPage" ),
        QString("([X-KDE-MultiPageVersion] == %1) and "
                "([X-KDE-EmptyMultiPage] == 1)").arg(MULTIPAGE_VERSION));
  }

  // If still no MultiPage has been found, report an error and abort.
  if (offers.isEmpty())
  {
    KMessageBox::error(parentWidget, i18n("<qt>No MultiPage found.</qt>"));
    return;
  }

  KService::Ptr service = offers.first();
  kdDebug() << service->library() << endl;

  // Try to load the multiPage
  multiPage = static_cast<KMultiPage*>(
      KParts::ComponentFactory::createInstanceFromService<KParts::ReadOnlyPart>(service, mainWidget,
        service->name().utf8()) );

  // If the loading of the MultiPage failed report and error and abort.
  if (!multiPage)
  {
    KMessageBox::error(parentWidget, i18n("<qt>Loading of library <b>%1</b> failed.</qt>").arg(service->library().utf8()));
     return;
  }
  // Remember the name of the library.
  multiPageLibrary = service->library();
  mainLayout->addWidget(multiPage->widget());

  // settings menu
  showSidebar = new KToggleAction (i18n("Show &Sidebar"), "show_side_panel", 0, this,
                                   SLOT(slotShowSidebar()), actionCollection(), "show_sidebar");
  showSidebar->setCheckedState(i18n("Hide &Sidebar"));
  watchAct = new KToggleAction(i18n("&Watch File"), 0, 0, 0, actionCollection(), "watch_file");
  scrollbarHandling = new KToggleAction (i18n("Show Scrollbars"), 0, 0, 0, actionCollection(), "scrollbarHandling");
  scrollbarHandling->setCheckedState(i18n("Hide Scrollbars"));

  // View modes
  QStringList viewModes;
  viewModes.append(i18n("Single Page"));
  viewModes.append(i18n("Continuous"));
  viewModes.append(i18n("Continuous - Facing"));
  viewModes.append(i18n("Overview"));
  viewModeAction = new KSelectAction (i18n("View Mode"), 0, 0, 0, actionCollection(), "viewmode");
  viewModeAction->setItems(viewModes);

  // Orientation menu
  QStringList orientations;
  orientations.append(i18n("Portrait"));
  orientations.append(i18n("Landscape"));
  orientation = new KSelectAction (i18n("Preferred &Orientation"), 0, 0, 0, actionCollection(), "view_orientation");
  orientation->setItems(orientations);
  connect(orientation, SIGNAL(activated (int)), &userRequestedPaperSize, SLOT(setOrientation(int)));

  // Zoom Menu
  zoom_action = new KSelectAction (i18n("&Zoom"), 0, 0, 0, actionCollection(), "view_zoom");
  zoom_action->setComboWidth(80);
  zoom_action->setEditable(true);
  zoom_action->setItems(_zoomVal.zoomNames());
  connect (&_zoomVal, SIGNAL(zoomNamesChanged(const QStringList &)), zoom_action, SLOT(setItems(const QStringList &)));
  connect (&_zoomVal, SIGNAL(valNoChanged(int)), zoom_action, SLOT(setCurrentItem(int)));
  connect (&_zoomVal, SIGNAL(zoomNameChanged(const QString &)), this, SIGNAL(zoomChanged(const QString &)) );
  connect (zoom_action, SIGNAL(activated(const QString &)), this, SLOT(setZoomValue(const QString &)));
  _zoomVal.setZoomValue(1.0); // should not be necessary @@@@
  emit(zoomChanged("100%"));

  // Paper Size Menu
  media = new KSelectAction (i18n("Preferred Paper &Size"), 0, 0, 0, actionCollection(), "view_media");
  QStringList items = userRequestedPaperSize.pageSizeNames();
  items.prepend(i18n("Custom Size..."));
  media->setItems(items);
  connect (media, SIGNAL(activated(int)), this, SLOT(slotMedia(int)));

  useDocumentSpecifiedSize = new KToggleAction(i18n("&Use Document Specified Paper Size"), 0, this, SLOT(slotShowSidebar()), 
					       actionCollection(), "view_use_document_specified_size");

  // Zoom Actions
  zoomInAct = KStdAction::zoomIn (this, SLOT(zoomIn()), actionCollection());
  zoomOutAct = KStdAction::zoomOut(this, SLOT(zoomOut()), actionCollection());

  fitPageAct = new KToggleAction(i18n("&Fit to Page"), Key_P,
                                 actionCollection(), "view_fit_to_page");
  fitWidthAct = new KToggleAction(i18n("Fit to Page &Width"), Key_W,
                                  actionCollection(), "view_fit_to_width");
  fitHeightAct = new KToggleAction(i18n("Fit to Page &Height"), Key_H,
                                   actionCollection(), "view_fit_to_height");

  fitPageAct -> setExclusiveGroup("view_fit");
  fitWidthAct -> setExclusiveGroup("view_fit");
  fitHeightAct -> setExclusiveGroup("view_fit");

  connect(fitPageAct, SIGNAL(toggled(bool)), this, SLOT(enableFitToPage(bool)));
  connect(fitWidthAct, SIGNAL(toggled(bool)), this, SLOT(enableFitToWidth(bool)));
  connect(fitHeightAct, SIGNAL(toggled(bool)), this, SLOT(enableFitToHeight(bool)));

  // go menu
  backAct = KStdAction::prior(this, SLOT(mp_prevPage()), actionCollection());
  forwardAct = KStdAction::next(this, SLOT(mp_nextPage()), actionCollection());
  startAct = KStdAction::firstPage(this, SLOT(mp_firstPage()), actionCollection());
  endAct = KStdAction::lastPage(this, SLOT(mp_lastPage()), actionCollection());
  gotoAct = KStdAction::gotoPage(this, SLOT(goToPage()), actionCollection());
  gotoAct->setShortcut("CTRL+G");

  readUpAct = new KAction(i18n("Read Up Document"), "up", SHIFT+Key_Space, this, SLOT(mp_readUp()), actionCollection(), "go_read_up");
  readDownAct = new KAction(i18n("Read Down Document"), "down", Key_Space, this, SLOT(mp_readDown()), actionCollection(), "go_read_down");

  printAction = KStdAction::print(this, SLOT(slotPrint()), actionCollection());

  saveAsAction = KStdAction::saveAs(this, SLOT(mp_slotSave()), actionCollection());

  // history action
  backAction = new KAction(i18n("&Back"), "1leftarrow", 0, 
                   this, SLOT(mp_doGoBack()), actionCollection(), "go_back");
  forwardAction = new KAction(i18n("&Forward"), "1rightarrow", 0, 
                      this, SLOT(mp_doGoForward()), actionCollection(), "go_forward");

  backAction->setEnabled(false);
  forwardAction->setEnabled(false);


  settingsAction = KStdAction::preferences(this, SLOT(doSettings()), actionCollection());

  aboutAction = new KAction(i18n("About KViewShell"), "kviewshell", 0, this,
                            SLOT(aboutKViewShell()), actionCollection(), "help_about_kviewshell");
  // keyboard accelerators
  accel = new KAccel(mainWidget);
  accel->insert(I18N_NOOP("Scroll Up"), Key_Up, this, SLOT(mp_scrollUp()));
  accel->insert(I18N_NOOP("Scroll Down"), Key_Down, this, SLOT(mp_scrollDown()));
  accel->insert(I18N_NOOP("Scroll Left"), Key_Left, this, SLOT(mp_scrollLeft()));
  accel->insert(I18N_NOOP("Scroll Right"), Key_Right, this, SLOT(mp_scrollRight()));

  accel->insert(I18N_NOOP("Scroll Up Page"), SHIFT+Key_Up, this, SLOT(mp_scrollUpPage()));
  accel->insert(I18N_NOOP("Scroll Down Page"), SHIFT+Key_Down, this, SLOT(mp_scrollDownPage()));
  accel->insert(I18N_NOOP("Scroll Left Page"), SHIFT+Key_Left, this, SLOT(mp_scrollLeftPage()));
  accel->insert(I18N_NOOP("Scroll Right Page"), SHIFT+Key_Right, this, SLOT(mp_scrollRightPage()));

  accel->readSettings();
  readSettings();

  m_extension = new KViewPartExtension(this);

  setXMLFile("kviewerpart.rc");
  initializeMultiPage();

  // The page size dialog is constructed on first usage -- saves some
  // memory when not used.
  _pageSizeDialog = 0;

  multiPage->setNumberOfPages(0);
  checkActions();
  viewModeAction->setCurrentItem(KVSPrefs::viewMode());
}

KViewPart::~KViewPart()
{
  writeSettings();
  delete multiPage;
  delete tmpUnzipped;
}


void KViewPart::initializeMultiPage()
{
  // Paper Size handling
  multiPage->pageCache.setUseDocumentSpecifiedSize(useDocumentSpecifiedSize->isChecked());
  multiPage->pageCache.setUserPreferredSize(userRequestedPaperSize);
  connect(&userRequestedPaperSize, SIGNAL(sizeChanged(simplePageSize)), &(multiPage->pageCache), SLOT(setUserPreferredSize(simplePageSize)));
  connect(useDocumentSpecifiedSize, SIGNAL(toggled(bool)), &(multiPage->pageCache), SLOT(setUseDocumentSpecifiedSize(bool)));


  connect(scrollbarHandling, SIGNAL(toggled(bool)),  multiPage, SLOT(slotShowScrollbars(bool)));
  
  // connect to the multi page view
  connect( this, SIGNAL(scrollbarStatusChanged(bool)), multiPage, SLOT(slotShowScrollbars(bool)));
  connect( multiPage, SIGNAL(pageInfo(int, int)), this, SLOT(pageInfo(int, int)) );
  connect( multiPage, SIGNAL(askingToCheckActions()), this, SLOT(checkActions()) );
  connect( multiPage, SIGNAL( started( KIO::Job * ) ), this, SIGNAL( started( KIO::Job * ) ) );
  connect( multiPage, SIGNAL( completed() ), this, SIGNAL( completed() ) );
  connect( multiPage, SIGNAL( canceled( const QString & ) ), this, SIGNAL( canceled( const QString & ) ) );
  connect( multiPage, SIGNAL( setStatusBarText( const QString& ) ), this, SLOT( setStatusBarTextFromMultiPage( const QString& ) ) );
  connect( multiPage->scrollView(), SIGNAL( wheelEventReceived(QWheelEvent *)), this,   SLOT(wheelEvent(QWheelEvent *)));

  // change the viewmode
  connect(viewModeAction, SIGNAL(activated (int)), multiPage, SLOT(setViewMode(int)));

  // Update zoomlevel on viewmode changes
  connect(multiPage, SIGNAL(viewModeChanged()), this, SLOT(updateZoomLevel()));
  
  // navigation history
  connect(multiPage->history(), SIGNAL(backItem(bool)), backAction, SLOT(setEnabled(bool)));
  connect(multiPage->history(), SIGNAL(forwardItem(bool)), forwardAction, SLOT(setEnabled(bool)));


  // Allow saving of the document, if the multipage supports it.
  delete saveAction;
  saveAction = 0;
  if (multiPage->isReadWrite())
    saveAction = KStdAction::save(this, SLOT(mp_slotSave_defaultFilename()), actionCollection());

  // allow parts to have a GUI, too :-)
  // (will be merged automatically)
  insertChildClient( multiPage );
}

void KViewPart::slotStartFitTimer()
{
  fitTimer.start(100, true);
}


QString KViewPart::pageSizeDescription(void)
{
  PageNumber nr = multiPage->currentPageNumber();
  if (!nr.isValid())
    return QString::null;
  simplePageSize ss = multiPage->pageCache.sizeOfPage(nr);
  if (!ss.isValid())
    return QString::null;
  pageSize s(ss);

  QString size = " ";
  if (s.formatNumber() == -1) {
    if (KGlobal::locale()-> measureSystem() == KLocale::Metric)
      size += QString("%1x%2 mm").arg(s.width_in_mm(), 0, 'f', 0).arg(s.height_in_mm(), 0, 'f', 0);
    else 
      size += QString("%1x%2 in").arg(s.width_in_mm()/25.4, 0, 'g', 2).arg(s.height_in_mm()/25.4, 0, 'g', 2);
  } else {
    size += s.formatName() + "/";
    if (s.getOrientation() == 0)
      size += i18n("portrait");
    else
      size += i18n("landscape");
  }
  return size+" ";
}


void KViewPart::restoreDocument(const KURL &url, int page)
{
  if (openURL(url))
    multiPage->gotoPage(page);
}


void KViewPart::saveDocumentRestoreInfo(KConfig* config)
{
  config->writePathEntry("URL", url().url());
  if (multiPage->numberOfPages() > 0)
    config->writeEntry("Page", multiPage->currentPageNumber());
}


void KViewPart::slotFileOpen()
{
  QString ff;

  // Search for service
  KTrader::OfferList offers = KTrader::self()->query(
      QString::fromLatin1("KViewShell/MultiPage"),
      QString("([X-KDE-MultiPageVersion] == %1)").arg(MULTIPAGE_VERSION)
  );

  // Build filter based on the capabilities of the found multiPages.
  if (!offers.isEmpty())
  {
    KTrader::OfferList::ConstIterator iterator = offers.begin();
    KTrader::OfferList::ConstIterator end = offers.end();

    for (; iterator != end; ++iterator)
    {
      KService::Ptr service = *iterator;
      ff += service->property("X-KDE-MimeTypes").toString() + " ";
    }
  }

  KURL url = KFileDialog::getOpenURL(QString::null, ff);

  if (!url.isEmpty())
    openURL(url);
}

QStringList KViewPart::fileFormats()
{
  // Compile a list of the supported filename patterns

  // First we build a list of the mimetypes which are supported by the
  // currently installed KMultiPage-Plugins.
  QStringList supportedMimeTypes;
  QStringList supportedPattern;

  // Search for service
  KTrader::OfferList offers = KTrader::self()->query(
      QString::fromLatin1("KViewShell/MultiPage"),
      QString("([X-KDE-MultiPageVersion] == %1)").arg(MULTIPAGE_VERSION)
  );

  if (!offers.isEmpty())
  {
    KTrader::OfferList::ConstIterator iterator = offers.begin();
    KTrader::OfferList::ConstIterator end = offers.end();

    for (; iterator != end; ++iterator)
    {
      KService::Ptr service = *iterator;
      QString mimeType = service->property("X-KDE-MimeTypes").toString();
      supportedMimeTypes << mimeType;

      QStringList pattern = KMimeType::mimeType(mimeType)->patterns();
      while(!pattern.isEmpty())
      {
        supportedPattern.append(pattern.front().stripWhiteSpace());
        pattern.pop_front();
      }
    }
  }

  // The kviewshell is also able to read compressed files and to
  // uncompress them on the fly. Thus, we modify the list of supported
  // file formats which we obtain from the multipages to include
  // compressed files like "*.dvi.gz". We add "*.dvi.bz2" if support
  // for bzip2 is compiled into KDE.

  // Check if this version of KDE supports bzip2
  bool bzip2Available = (KFilterBase::findFilterByMimeType( "application/x-bzip2" ) != 0L);

  QStringList compressedPattern;

  for(QStringList::Iterator it = supportedPattern.begin(); it != supportedPattern.end(); ++it )
  {
    if ((*it).find(".gz", -3) == -1) // Paranoia safety check
      compressedPattern.append(*it + ".gz");

    if ((bzip2Available) && ((*it).find(".bz2", -4) == -1)) // Paranoia safety check
      compressedPattern.append(*it + ".bz2");
  }

  while (!compressedPattern.isEmpty())
  {
    supportedPattern.append(compressedPattern.front());
    compressedPattern.pop_front();
  }

  kdDebug(1223) << "Supported Pattern: " << supportedPattern << endl;

  return supportedPattern;
}


void KViewPart::slotSetFullPage(bool fullpage)
{
  if (multiPage)
    multiPage->slotSetFullPage(fullpage);
  else
    kdError(4300) << "KViewPart::slotSetFullPage() called without existing multipage" << endl;

  if (fullpage == false)
    slotShowSidebar();
}


void KViewPart::slotShowSidebar()
{
  bool show = showSidebar->isChecked();
  multiPage->slotShowSidebar(show);
}


bool KViewPart::openFile()
{
  KURL tmpFileURL;

  // We try to be error-tolerant about filenames. If the user calls us
  // with something like "test", and we are using the DVI-part, we'll
  // also look for "testdvi" and "test.dvi".
  QFileInfo fi(m_file);
  m_file = fi.absFilePath();

  if (!fi.exists())
  {
    QStringList supportedPatterns = fileFormats();
    QStringList endings;

    for (QStringList::Iterator it = supportedPatterns.begin(); it != supportedPatterns.end(); ++it)
    {
      // Only consider patterns starting with "*."
      if ((*it).find("*.") == 0)
      {
        // Remove first Letter from string
        endings.append((*it).mid(2).stripWhiteSpace());
      }
    }
    kdDebug(1223) << "Supported Endings: " << endings << endl;

    // Now try to append the endings with and without "." to the given filename,
    // and see if that gives a existing file.
    for (QStringList::Iterator it = endings.begin(); it != endings.end(); ++it)
    {
      fi.setFile(m_file+(*it));
      if (fi.exists())
      {
        m_file = m_file+(*it);
        break;
      }
      fi.setFile(m_file+"."+(*it));
      if (fi.exists())
      {
        m_file = m_file+"."+(*it);
        break; 
      }
    }

    // If we still have not found a file. Show an error message and return.
    if (!fi.exists())
    {
      KMessageBox::error(mainWidget, i18n("<qt>File <nobr><strong>%1</strong></nobr> does not exist.</qt>").arg(m_file));
      emit setStatusBarText(QString::null);
      return false;
    }
    m_url.setPath(QFileInfo(m_file).absFilePath());
  }

  // Set the window caption now, before we do any uncompression and generation of temporary files.
  tmpFileURL.setPath(m_file);
  emit setStatusBarText(i18n("Loading '%1'...").arg(tmpFileURL.prettyURL()));
  emit setWindowCaption( tmpFileURL.prettyURL() ); // set Window caption WITHOUT the reference part!

  // Check if the file is compressed
  KMimeType::Ptr mimetype = KMimeType::findByPath( m_file );

  if (( mimetype->name() == "application/x-gzip" ) || ( mimetype->name() == "application/x-bzip2" ) ||
      ( mimetype->parentMimeType() == "application/x-gzip" ) ||
      ( mimetype->parentMimeType() == "application/x-bzip2" ))
  {
    // The file is compressed. Make a temporary file, and store an uncompressed version there...
    if (tmpUnzipped != 0L)  // Delete old temporary file
      delete tmpUnzipped;

    tmpUnzipped = new KTempFile;
    if (tmpUnzipped == 0L) 
    {
      KMessageBox::error(mainWidget, i18n("<qt><strong>File Error!</strong> Could not create "
                         "temporary file.</qt>"));
      emit setWindowCaption(QString::null);
      emit setStatusBarText(QString::null);
      return false;
    }
    tmpUnzipped->setAutoDelete(true);
    if(tmpUnzipped->status() != 0)
    {
      KMessageBox::error(mainWidget, i18n("<qt><strong>File Error!</strong> Could not create temporary file "
                         "<nobr><strong>%1</strong></nobr>.</qt>").arg(strerror(tmpUnzipped->status())));
      emit setWindowCaption(QString::null);
      emit setStatusBarText(QString::null);
      return false;
    }

    QIODevice* filterDev;
    if (( mimetype->parentMimeType() == "application/x-gzip" ) ||
        ( mimetype->parentMimeType() == "application/x-bzip2" ))
      filterDev = KFilterDev::deviceForFile(m_file, mimetype->parentMimeType());
    else
      filterDev = KFilterDev::deviceForFile(m_file);
    if (filterDev == 0L)
    {
      emit setWindowCaption(QString::null);
      emit setStatusBarText(QString::null);
      return false;
    }
    if(!filterDev->open(IO_ReadOnly))
    {
      KMessageBox::detailedError(mainWidget, i18n("<qt><strong>File Error!</strong> Could not open the file "
          "<nobr><strong>%1</strong></nobr> for uncompression. "
          "The file will not be loaded.</qt>").arg(m_file),
          i18n("<qt>This error typically occurs if you do not have enough permissions to read the file. "
          "You can check ownership and permissions if you right-click on the file in the Konqueror "
          "file manager and then choose the 'Properties' menu.</qt>"));
      emit setWindowCaption(QString::null);
      delete filterDev;
      emit setStatusBarText(QString::null);
      return false;
    }

    KProgressDialog* prog = new KProgressDialog(0L, "uncompress-progress",
        i18n("Uncompressing..."),
        i18n("<qt>Uncompressing the file <nobr><strong>%1</strong></nobr>. Please wait.</qt>").arg(m_file));

    prog->progressBar()->setTotalSteps((int) fi.size()/1024);
    prog->progressBar()->setProgress(0);
    prog->setMinimumDuration(250);

    QByteArray buf(1024);
    int read = 0, wrtn = 0;

    while ((read = filterDev->readBlock(buf.data(), buf.size())) > 0)
    {
      kapp->processEvents();
      if (prog->wasCancelled())
        break;
      prog->progressBar()->advance(1);

      wrtn = tmpUnzipped->file()->writeBlock(buf.data(), read);
      if(read != wrtn)
        break;
    }
    delete filterDev;
    delete prog;

    if (prog->wasCancelled()) {
      emit setStatusBarText(QString::null);
      return false;
    }

    if ((read != 0) || (tmpUnzipped->file()->size() == 0))
    {
      KMessageBox::detailedError(mainWidget, i18n("<qt><strong>File Error!</strong> Could not uncompress "
          "the file <nobr><strong>%1</strong></nobr>. The file will not be loaded.</qt>").arg( m_file ),
          i18n("<qt>This error typically occurs if the file is corrupt. "
          "If you want to be sure, try to decompress the file manually using command-line tools.</qt>"));
      emit setWindowCaption(QString::null);
      emit setStatusBarText(QString::null);
      return false;
    }
    tmpUnzipped->close();

    m_file = tmpUnzipped->name();
  }

  // Now call the openURL-method of the multipage and give it an URL
  // pointing to the downloaded file.
  tmpFileURL.setPath(m_file);
  // Pass the reference part of the URL through to the multipage
  tmpFileURL.setRef(m_url.ref());

  mimetype = KMimeType::findByURL(tmpFileURL);

  // Search for service
  KTrader::OfferList offers = KTrader::self()->query(
      QString::fromLatin1("KViewShell/MultiPage" ),
  QString("([X-KDE-MultiPageVersion] == %1) and "
          "([X-KDE-MimeTypes] == '%2')").arg(MULTIPAGE_VERSION).arg(mimetype->name()));

  kdDebug(1223) << "MimeType: " << mimetype->name() << endl;

  if (offers.isEmpty())
  {
    KMessageBox::detailedError(mainWidget, 
			       i18n("<qt><strong>File Type Error!</strong> Could not load the file "
				    "<nobr><strong>%1</strong></nobr>.</qt>").arg(m_file),
			       i18n("<qt>No MultiPage supporting mimetype %1  found.</qt>").arg(mimetype->name()));
    emit setWindowCaption(QString::null);
    emit setStatusBarText(QString::null);
    return false;
  }

  KService::Ptr service = offers.first();
  kdDebug() << service->library() << endl;

  // The the new multiPage is different then the currently loaded one.
  if (service->library() != multiPageLibrary)
  {
    // Remove the old MultiPage.
    mainLayout->remove(multiPage->widget());
    kdDebug(1223) << endl << "Delete MultiPage" << endl << endl;
    removeChildClient(multiPage);
    delete multiPage;

    // Try to load the multiPage
    multiPage = static_cast<KMultiPage*>(
        KParts::ComponentFactory::createInstanceFromService<KParts::ReadOnlyPart>(service, mainWidget,
        service->name().utf8()) );

    mainLayout->addWidget(multiPage->widget());
    multiPage->widget()->show();
    initializeMultiPage();
    readSettings();
  }

  // Load the URL
  bool r = multiPage->openURL(tmpFileURL);

  if (r) {
    // Add the file to the watchlist
    watch->addFile( m_file );
    // Set the multipage to the current viewmode.
    // This also generates the page widgets.
    multiPage->setViewMode(viewModeAction->currentItem());
    // Notify the ViewShell about the newly opened file.
    emit fileOpened();
  } else {
    m_url = QString::null;
    emit setWindowCaption(QString::null);
  }

  checkActions();
  emit zoomChanged(QString("%1%").arg((int)(_zoomVal.value()*100.0+0.5)));
  emit setStatusBarText(QString::null);
  return r;
}


void KViewPart::reload(void)
{
  multiPage->reload();
}


void KViewPart::fileChanged(const QString &file)
{
  if (file == m_file && watchAct->isChecked())
    multiPage->reload();
}


bool KViewPart::closeURL()
{
  if( watch && !m_file.isEmpty() )
    watch->removeFile( m_file );

  KParts::ReadOnlyPart::closeURL();

  multiPage->closeURL();

  m_url = QString::null;

  multiPage->setNumberOfPages(0);
  checkActions();

  emit setWindowCaption("");

  return true;
}


void KViewPart::slotMedia(int id)
{
  // If the user has chosen one of the 'known' paper sizes, set the
  // user requested paper size to that value. Via signals and slots,
  // this will update the menus, and also the GUI, if necessary.
  if (id > 1) {
    userRequestedPaperSize.setPageSize(media->currentText());
    return;
  }
  
  // If the user has chosen "Custom paper size..", show the paper size
  // dialog. Construct it, if necessary. The paper size dialog will
  // know the address of userRequestedPaperSize and change this
  // member, if the user clicks ok/accept. The signal/slot mechanism
  // will then make sure that the necessary updates in the GUI are
  // done.
  if (_pageSizeDialog == 0) {
    _pageSizeDialog = new pageSizeDialog(mainWidget, &userRequestedPaperSize);
    if (_pageSizeDialog == 0) {
      kdError(4300) << "Could not construct the page size dialog!" << endl;
      return;
    }
  }

  // Reset the "preferred paper size" menu. We don't want to have the
  // "custom paper size" check if the user aborts the dialog.
  checkActions();

  // Set or update the paper size dialog to show the currently
  // selected value.
  _pageSizeDialog->setPageSize(userRequestedPaperSize.serialize());
  _pageSizeDialog->show();
}


void KViewPart::pageInfo(int numpages, int currentpage)
{
  // ATTN: The string here must be the same as in setPage() below
  QString pageString = i18n("Page %1 of %2").arg(currentpage).arg(numpages);
  if (pageChangeIsConnected) {
    emit pageChanged(pageString);
    emit sizeChanged(pageSizeDescription());
  } else
    emit setStatusBarText(pageString);

  checkActions();
}


void KViewPart::goToPage()
{
  bool ok = false;
  int p = KInputDialog::getInteger(i18n("Go to Page"), i18n("Page:"), 
                                   multiPage->currentPageNumber(), 1, multiPage->numberOfPages(),
                                   1 /*step*/, &ok, mainWidget, "gotoDialog");
  if (ok)
    multiPage->gotoPage(p);
}


void KViewPart::disableZoomFit()
{
  if (fitPageAct -> isChecked())
  {
    fitPageAct -> setChecked(false);
    enableFitToPage(false);
  }
  else if(fitWidthAct -> isChecked())
  {  
    fitWidthAct -> setChecked(false);
    enableFitToWidth(false);
  }
  else if (fitHeightAct -> isChecked())
  {
    fitHeightAct -> setChecked(false);
    enableFitToHeight(false);
  }
}

void KViewPart::zoomIn()
{
  disableZoomFit();

  float oldVal = _zoomVal.value();
  float newVal = _zoomVal.zoomIn();

  if (oldVal != newVal)
    _zoomVal.setZoomValue(multiPage->setZoom(_zoomVal.zoomIn()));
}


void KViewPart::zoomOut()
{
  disableZoomFit();

  float oldVal = _zoomVal.value();
  float newVal = _zoomVal.zoomOut();

  if (oldVal != newVal)
    _zoomVal.setZoomValue(multiPage->setZoom(_zoomVal.zoomOut()));
}

void KViewPart::updateZoomLevel()
{
  if (fitPageAct->isChecked())
  {
    fitToPage();
  }
  else if (fitWidthAct->isChecked())
  {
    fitToWidth();
  }
  else if (fitHeightAct->isChecked())
  {
    fitToHeight();
  }
  else
  {
    // Manuell Zoom
  }
}

void KViewPart::enableFitToPage(bool enable)
{
  if (enable)
  {
    fitToPage();
    connect(multiPage->scrollView(), SIGNAL(viewSizeChanged(QSize)), 
            this, SLOT(slotStartFitTimer()));
    connect(&fitTimer, SIGNAL(timeout()), SLOT(fitToPage()));
  }
  else
  {
    disconnect(multiPage->scrollView(), SIGNAL(viewSizeChanged(QSize)),
               this, SLOT(slotStartFitTimer()));
    disconnect(&fitTimer, SIGNAL(timeout()), this, SLOT(fitToPage()));
  }
}

void KViewPart::enableFitToWidth(bool enable)
{
  if (enable)
  {
    fitToWidth();
    connect(multiPage->scrollView(), SIGNAL(viewSizeChanged(QSize)), 
            this, SLOT(slotStartFitTimer()));
    connect(&fitTimer, SIGNAL(timeout()), SLOT(fitToWidth()));
  }
  else
  {
    disconnect(multiPage->scrollView(), SIGNAL(viewSizeChanged(QSize)),
               this, SLOT(slotStartFitTimer()));
    disconnect(&fitTimer, SIGNAL(timeout()), this, SLOT(fitToWidth()));
  }
}

void KViewPart::enableFitToHeight(bool enable)
{
  if (enable)
  {
    fitToHeight();
    connect(multiPage->scrollView(), SIGNAL(viewSizeChanged(QSize)), 
            this, SLOT(slotStartFitTimer()));
    connect(&fitTimer, SIGNAL(timeout()), SLOT(fitToHeight()));
  }
  else
  {
    disconnect(multiPage->scrollView(), SIGNAL(viewSizeChanged(QSize)),
               this, SLOT(slotStartFitTimer()));
    disconnect(&fitTimer, SIGNAL(timeout()), this, SLOT(fitToHeight()));
  }
}


void KViewPart::fitToPage()
{
  double z = QMIN(multiPage->calculateFitToHeightZoomValue(), 
                  multiPage->calculateFitToWidthZoomValue());

  // Check if the methods returned usable values. Values that are not
  // within the limits indicate that fitting to width or height is
  // currently not possible (e.g. because no document is
  // loaded). In that case, we abort.
  if ((z < ZoomLimits::MinZoom/1000.0) || (z > ZoomLimits::MaxZoom/1000.0))
    return;

  _zoomVal.setZoomValue(multiPage->setZoom(z));
}


void KViewPart::fitToHeight()
{
  double z = multiPage->calculateFitToHeightZoomValue();

  // Check if the method returned a usable value. Values that are not
  // within the limits indicate that fitting to height is currently
  // not possible (e.g. because no document is loaded). In that case,
  // we abort.
  if ((z < ZoomLimits::MinZoom/1000.0) || (z > ZoomLimits::MaxZoom/1000.0))
    return;

  _zoomVal.setZoomValue(multiPage->setZoom(z));
}


void KViewPart::fitToWidth()
{
  double z = multiPage->calculateFitToWidthZoomValue();
  
  // Check if the method returned a usable value. Values that are not
  // within the limits indicate that fitting to width is currently not
  // possible (e.g. because no document is loaded). In that case, we
  // abort.
  if ((z < ZoomLimits::MinZoom/1000.0) || (z > ZoomLimits::MaxZoom/1000.0))
    return;
  
  _zoomVal.setZoomValue(multiPage->setZoom(z));
}


void KViewPart::setZoomValue(const QString &sval)
{
  disableZoomFit();

  float fval = _zoomVal.value();
  _zoomVal.setZoomValue(sval);
  if (fval != _zoomVal.value())
    _zoomVal.setZoomValue(multiPage->setZoom(_zoomVal.value()));
}


void KViewPart::checkActions()
{
  int currentPage = multiPage->currentPageNumber();
  int numberOfPages = multiPage->numberOfPages();

  bool doc = !url().isEmpty();

  documentRenderer *r = multiPage->getRenderer();
  useDocumentSpecifiedSize->setEnabled( (r != 0) && r->hasSpecifiedPageSizes() );

  if (multiPage->scrollView()->overviewMode())
  {
    int visiblePages = multiPage->scrollView()->getNrRows() *
                       multiPage->scrollView()->getNrColumns();
    // firstVisiblePage is the smallest currently shown pagenumber.
    int firstVisiblePage = currentPage - (currentPage % visiblePages);

    backAct->setEnabled(doc && currentPage >= visiblePages);
    forwardAct->setEnabled(doc && firstVisiblePage <= numberOfPages - visiblePages);

    startAct->setEnabled(doc && firstVisiblePage > 1);
    endAct->setEnabled(doc && firstVisiblePage + visiblePages < numberOfPages);
  }
  else
  {
    backAct->setEnabled(doc && currentPage > 1);
    forwardAct->setEnabled(doc && currentPage < numberOfPages);

    startAct->setEnabled(doc && currentPage > 1);
    endAct->setEnabled(doc && currentPage < numberOfPages);
  }

  gotoAct->setEnabled(doc && numberOfPages > 1);
  readDownAct->setEnabled(doc);
  readUpAct->setEnabled(doc);

  zoomInAct->setEnabled(doc);
  zoomOutAct->setEnabled(doc);

  fitPageAct->setEnabled(doc);
  fitHeightAct->setEnabled(doc);
  fitWidthAct->setEnabled(doc);

  media->setEnabled(doc);
  orientation->setEnabled(doc);

  printAction->setEnabled(doc);

  if (multiPage->isReadWrite())
    saveAction->setEnabled(multiPage->isModified());

  saveAsAction->setEnabled(doc);

  if (userRequestedPaperSize.formatNumber() != -1) {
    orientation->setCurrentItem(userRequestedPaperSize.getOrientation());
    orientation->setEnabled(true);
    media->setCurrentItem(userRequestedPaperSize.formatNumber()+1);
  } else {
    orientation->setEnabled(false);
    media->setCurrentItem(userRequestedPaperSize.formatNumber()-1);
  }
}


void KViewPart::wheelEvent(QWheelEvent *e)
{
  QScrollBar *sb = multiPage->scrollView()->verticalScrollBar();
  if (sb == 0)
    return;

  // Zoom in/out
  if (e->state() & ControlButton)
  {
    if (e->delta() < 0)
      zoomOut();
    else
      zoomIn();
    return;
  }

  Q_INT32 pxl = -(e->delta()*sb->lineStep())/60;
  if (pxl == 0)
  {
    if (e->delta() > 0)
      pxl = -1;
    else
      pxl = 1;
  }
 
  // Faster scrolling
  if (e->state() & ShiftButton)
    pxl *= 10;
  multiPage->scroll(pxl);
}


void KViewPart::slotPrint()
{
  QStringList pages;
  QValueList<int> selectedPageNo = multiPage->selectedPages();

  QValueList<int>::iterator it;
  for ( it = selectedPageNo.begin(); it != selectedPageNo.end(); ++it )
    pages.append(QString::number((*it)-1));

  multiPage->print(pages, multiPage->currentPageNumber());
}


void KViewPart::readSettings()
{
  showSidebar->setChecked(KVSPrefs::pageMarks());
  slotShowSidebar();

  watchAct->setChecked(KVSPrefs::watchFile());

  // The value 'fitToPage' has several meanings: 1 is 'fit to page
  // width', 2 is 'fit to page height', 3 is 'fit to page'. Other
  // values indicate 'no fit to page'. Note: at the time this code is
  // executed, the methods fitToWidth(), etc., do not work well at all
  // (perhaps some data is not initialized yet)? For that reason, we
  // do not call these methods, and load the last zoom-value from the
  // configuration file below. The hope is that this value is not
  // terribly wrong. If the user doesn't like it, it suffices to
  // resize the window just a bit...
  switch(KVSPrefs::fitToPage()) {
  case KVSPrefs::EnumFitToPage::KVS_FitToPage:
    fitPageAct->setChecked(true);
    enableFitToPage(true);
    break;
  case KVSPrefs::EnumFitToPage::KVS_FitToPageWidth:
    fitWidthAct->setChecked(true);
    enableFitToWidth(true);
    break;
  case KVSPrefs::EnumFitToPage::KVS_FitToPageHeight:
    fitHeightAct->setChecked(true);
    enableFitToHeight(true);
    break;
  }
  
  // Read zoom value. Even if 'fitToPage' has been set above, there is
  // no widget available right now, so setting a good default value
  // from the configuration file is perhaps not a bad idea.
  float _zoom = KVSPrefs::zoom();
  if ( (_zoom < ZoomLimits::MinZoom/1000.0) || (_zoom > ZoomLimits::MaxZoom/1000.0)) {
    kdWarning(4300) << "Illeagal zoom value of " << _zoom*100.0 << "% found in the preferences file. Setting zoom to 100%." << endl;
    _zoom = 1.0;
  }
  _zoomVal.setZoomValue(multiPage->setZoom(_zoom));
  
  // Read Paper Size. and orientation. The GUI is updated
  // automatically by the signals/slots mechanism whenever
  // userRequestedPaperSize is changed.
  userRequestedPaperSize.setPageSize(KVSPrefs::paperFormat());

  // Check if scrollbars should be shown.
  bool sbstatus = KVSPrefs::scrollbars();
  scrollbarHandling->setChecked(sbstatus);
  emit scrollbarStatusChanged(sbstatus);

  // Check if document specified paper sizes should be shown. We do
  // not need to take any action here, because this method is called
  // only in the constructor of the KViewPart when no document is loaded.
  useDocumentSpecifiedSize->setChecked(KVSPrefs::useDocumentSpecifiedSize());
}


void KViewPart::writeSettings()
{
  // if loading the KPart failed - just exit now
  if (!showSidebar)
    return;
  KVSPrefs::setPageMarks(showSidebar->isChecked());
  KVSPrefs::setWatchFile(watchAct->isChecked());
  KVSPrefs::setZoom(_zoomVal.value());
  KVSPrefs::setPaperFormat(userRequestedPaperSize.serialize());
  KVSPrefs::setScrollbars(scrollbarHandling->isChecked());
  KVSPrefs::setUseDocumentSpecifiedSize(useDocumentSpecifiedSize->isChecked());
  if (fitPageAct->isChecked())
    KVSPrefs::setFitToPage(KVSPrefs::EnumFitToPage::KVS_FitToPage);
  else if(fitWidthAct->isChecked())
    KVSPrefs::setFitToPage(KVSPrefs::EnumFitToPage::KVS_FitToPageWidth);
  else if (fitHeightAct->isChecked())
    KVSPrefs::setFitToPage(KVSPrefs::EnumFitToPage::KVS_FitToPageHeight);
  else
    KVSPrefs::setFitToPage(KVSPrefs::EnumFitToPage::KVS_DontFit);
  
  KVSPrefs::writeConfig();
}


void KViewPart::connectNotify ( const char *sig )
{
  if (QString(sig).contains("pageChanged"))
    pageChangeIsConnected = true;
}


void KViewPart::setStatusBarTextFromMultiPage( const QString &msg )
{
  if (msg.isEmpty())
  {
    if (pageChangeIsConnected)
      emit setStatusBarText(QString::null);
    else
    {
      int currentPage = multiPage->currentPageNumber();
      int numberOfPages = multiPage->numberOfPages();
      emit setStatusBarText(i18n("Page %1 of %2").arg(currentPage + 1).arg(numberOfPages));
    }
  }
  else
    emit setStatusBarText(msg);
}

KAboutData* KViewPart::createAboutData()
{
  return new KAboutData("kviewerpart", I18N_NOOP("Document Viewer Part"),
                        "0.4", I18N_NOOP(""),
                        KAboutData::License_GPL,
                        I18N_NOOP("Copyright (c) 2005 Wilfried Huss"));
}

void KViewPart::aboutKViewShell()
{
  if (aboutDialog == 0)
  {
    // Create Dialog
    aboutDialog = new KAboutDialog(multiPage->scrollView(), "about_kviewshell");
    aboutDialog->setTitle(I18N_NOOP("KViewShell"));
    aboutDialog->setVersion("0.4");
    aboutDialog->setAuthor("Matthias Hoelzer-Kluepfel", QString::null, QString::null,
                           I18N_NOOP("Original Author"));

    aboutDialog->addContributor("Matthias Hoelzer-Kluepfel", "mhk@caldera.de", QString::null,
                                I18N_NOOP("Framework"));
    aboutDialog->addContributor("David Sweet", "dsweet@kde.org", "http://www.chaos.umd.edu/~dsweet",
                                I18N_NOOP("Former KGhostView Maintainer"));
    aboutDialog->addContributor("Mark Donohoe", QString::null, QString::null,
                                I18N_NOOP("KGhostView Author"));
    aboutDialog->addContributor("Markku Hihnala", QString::null, QString::null,
                                I18N_NOOP("Navigation widgets"));
    aboutDialog->addContributor("David Faure", QString::null, QString::null,
                                I18N_NOOP("Basis for shell"));
    aboutDialog->addContributor("Daniel Duley", QString::null, QString::null,
                                I18N_NOOP("Port to KParts"));
    aboutDialog->addContributor("Espen Sand", QString::null, QString::null,
                                I18N_NOOP("Dialog boxes"));
    aboutDialog->addContributor("Stefan Kebekus", "kebekus@kde.org", QString::null,
                                I18N_NOOP("DCOP-Interface, major improvements"));
    aboutDialog->addContributor("Wilfried Huss", "Wilfried.Huss@gmx.at", QString::null,
                                I18N_NOOP("Interface enhancements"));
  }
  aboutDialog->show();
}

void KViewPart::doSettings()
{
  if (KConfigDialog::showDialog("kviewshell_config"))
    return;

  KConfigDialog* configDialog = new KConfigDialog(multiPage->scrollView(), "kviewshell_config", KVSPrefs::self());

  optionDialogGUIWidget_base* guiWidget = new optionDialogGUIWidget_base(multiPage->scrollView());
  configDialog->addPage(guiWidget, i18n("User Interface"), "view_choose");

  multiPage->addConfigDialogs(configDialog);

  connect(configDialog, SIGNAL(settingsChanged()), this, SLOT(preferencesChanged()));
  configDialog->show();
}

void KViewPart::preferencesChanged()
{
  multiPage->setViewMode(viewModeAction->currentItem());
  multiPage->slotShowThumbnails(KVSPrefs::showThumbnails());
}

void KViewPart::partActivateEvent( KParts::PartActivateEvent *ev )
{
  QApplication::sendEvent( multiPage, ev );
}


void KViewPart::guiActivateEvent( KParts::GUIActivateEvent *ev )
{
  QApplication::sendEvent( multiPage, ev );
}


KViewPartExtension::KViewPartExtension(KViewPart *parent)
  : KParts::BrowserExtension( parent, "KViewPartExtension")
{
}


// KMultiPage Interface
void KViewPart::mp_prevPage()
{
  multiPage->prevPage();
}

void KViewPart::mp_nextPage()
{
  multiPage->nextPage();
}

void KViewPart::mp_firstPage()
{
  multiPage->firstPage();
}

void KViewPart::mp_lastPage()
{
  multiPage->lastPage();
}


void KViewPart::mp_readUp()
{
  multiPage->readUp();
}

void KViewPart::mp_readDown()
{
  multiPage->readDown();
}


void KViewPart::mp_scrollUp()
{
  multiPage->scrollUp();
}

void KViewPart::mp_scrollDown()
{
  multiPage->scrollDown();
}

void KViewPart::mp_scrollLeft()
{
  multiPage->scrollLeft();
}

void KViewPart::mp_scrollRight()
{
  multiPage->scrollRight();
}

void KViewPart::mp_scrollUpPage()
{
  multiPage->scrollUpPage();
}

void KViewPart::mp_scrollDownPage()
{
  multiPage->scrollDownPage();
}

void KViewPart::mp_scrollLeftPage()
{
  multiPage->scrollLeftPage();
}

void KViewPart::mp_scrollRightPage()
{
  multiPage->scrollRightPage();
}


void KViewPart::mp_slotSave()
{
  multiPage->slotSave();
}

void KViewPart::mp_slotSave_defaultFilename()
{
  multiPage->slotSave_defaultFilename();
}


void KViewPart::mp_doGoBack()
{
  multiPage->doGoBack();
}

void KViewPart::mp_doGoForward()
{
  multiPage->doGoForward();
}
