/*
 * METALBASE 5.1
 *
 * Released January 1st, 1993 by Huan-Ti [ t-richj@microsoft.com ]
 *
 */

#ifndef PARSE_H
#define PARSE_H

#ifndef MBASE_H
#include <mbase.h>
#endif


/*
 * MAX_BUF, defined below, is the size of the read-buffer used during
 * file reading.  Because of the method used, this doubles as the maximum
 * line- and word-length; if this is too small for any parsing you need to
 * do, define MAX_BUF higher when compiling.
 *
 */

#define MAX_BUF   512
#define MAX_LINE  128


/*
 * File parsing is done on a word-by-word basis; words being delimited by
 * whitespace, or changes between the type of string; that is, a word will
 * continue as long as it is made of just letters and numbers, or just special
 * characters (:;,!@#*& etc).  Grouping characters (parentheses, brackets
 * and braces) are returned as separate words, as are C and C++ comment
 * strings.  Quotes are honored (but not returned); and if fpInit() is called
 * with TRUE as the second argument, non-quoted words (from fpWord) and non-
 * quoted sections of lines (from fpLine) will be mapped to lower-case.
 *
 * In order to parse the data in a file, use the following operations:
 *
 *    FParse *fp;
 *
 *    if ((fp = fpInit ("filename", FALSE)) == NULL)  abort
 *
 *    for ( ; ! fpEOF(fp); )
 *       {
 *       printf ("WORD IS '%s'\n", fpWord (fp));
 *       }
 *
 * fpBackup() will place the previously-read word back, so it will be read
 * again the next time fpWord() is called.  In contrast, fpRewind() will
 * reset the file to the very beginning, as if you'd just called fpInit().
 *
 */

typedef struct
   {
   int   fh;               /* Filehandle    ( if fpInit() )         */
   char *str;              /* Source string ( if fpInitStr() )      */
   bool  fLower;           /* Make all non-quoted words lower-case? */
   bool  fEOF;             /* TRUE if EOF has been reached          */
   bool  fQuoted;          /* TRUE if last word was quoted          */
   bool  fLastEOF;         /* TRUE if EOF reached at last position  */
   char  word[MAX_LINE+1]; /* Return buffer                         */
   char  buf[MAX_BUF+1];   /* Read buffer                           */
   int   nBuf;             /* Number of characters in read buffer   */
   int   nLines;           /* Current line number (1==first)        */
   int   nLast;            /* Number of lines at lastpos            */
   long  lastpos;          /* Position before we read the last word */
   long  curpos;           /* Current position (after buf[])        */
   } FParse;


/*
 * MACROS ---------------------------------------------------------------------
 *
 */

#define fpLastno(fp)  ( fp->nLast )
#define fpLineno(fp)  ( fp->nLines )
#define fpQuoted(fp)  ( fp->fQuoted )
#define fpEOF(fp)     ( fp->fEOF && (fp->nBuf == 0) )

#define fpRewind(fp)  fpSetPos (fp, 0L, 1)
#define fpGetPos(fp)  ( fp->curpos )

/*
 * FUNCTION PROTOTYPES --------------------------------------------------------
 *
 */

   FParse    *fpInit         XARGS( (charptr, bool) );
   FParse    *fpInitStr      XARGS( (charptr, bool) );
   void       fpClose        XARGS( (FParse *) );
   charptr    fpWord         XARGS( (FParse *) );
   charptr    fpLine         XARGS( (FParse *) );
   bool       fpBackup       XARGS( (FParse *) );
   void       fpSetPos       XARGS( (FParse *, long, int) );


#endif  /* PARSE_H */

