
#include <string.h>
#include <glib.h>
#include <pan/base/gnksa.h>
#include <pan/base/pan-glib-extensions.h>

typedef struct 
{
	int expected_retval;
	const char * address;
}
AddressCheck;

typedef struct
{
	int len;
	const char * in;
	const char * expected_out;
}
ReferencesCheck;

static gint
test_generate_references (void)
{
	gint test = 0;
	const gchar * refs;
	const gchar * msg_id;
	const gchar * expected;
	gchar * pch;

	++test;
	refs =  "<9uh0rq$8g9ua$1@ID-41667.news.dfncis.de> <9596705.7QZiUf9aFP@aschiller.easynet.de> <Xns916D78745865Dvg@newsf1.volker-gringmuth.de> <3C111F64.D2AABC41@babsi.de> <pan.2001.12.07.21.44.36.620796.4981@gmx.de> <pan.2001.12.08.21.19.07.420400.7547@babsi.de> <pan.2001.12.08.21.30.14.714578.7547@babsi.de>";
	msg_id = "<pan.2001.12.08.22.06.46.245566.1891@gmx.de>";
	pch = gnksa_generate_references (refs, msg_id);
	expected = "<9uh0rq$8g9ua$1@ID-41667.news.dfncis.de> <9596705.7QZiUf9aFP@aschiller.easynet.de> <Xns916D78745865Dvg@newsf1.volker-gringmuth.de> <3C111F64.D2AABC41@babsi.de> <pan.2001.12.07.21.44.36.620796.4981@gmx.de> <pan.2001.12.08.21.19.07.420400.7547@babsi.de> <pan.2001.12.08.21.30.14.714578.7547@babsi.de> <pan.2001.12.08.22.06.46.245566.1891@gmx.de>";
	if (strcmp (pch, expected))
		return test;

	g_message ("PASS gnksa_generate_references");
	return 0;
}

static ReferencesCheck references [] =
{
	{
		998, /*  gnksa cutoff */

		"<gnksa_pan-0.8.0_001@lull.org>"
		" <gnksa_pan-0.8.0_002@lull.org> <gnksa_pan-0.8.0_003@lull.org>"
		" <gnksa_pan-0.8.0_004@lull.org> <gnksa_pan-0.8.0_005@lull.org>"
		" <gnksa_pan-0.8.0_006@lull.org> <gnksa_pan-0.8.0_007@lull.org>"
		" <gnksa_pan-0.8.0_008@lull.org> <gnksa_pan-0.8.0_009@lull.org>"
		" <gnksa_pan-0.8.0_010@lull.org> <gnksa_pan-0.8.0_011@lull.org>"
		" <gnksa_pan-0.8.0_012@lull.org> <gnksa_pan-0.8.0_013@lull.org>"
		" <gnksa_pan-0.8.0_014@lull.org> <gnksa_pan-0.8.0_015@lull.org>"
		" <gnksa_pan-0.8.0_016@lull.org> <gnksa_pan-0.8.0_017@lull.org>"
		" <gnksa_pan-0.8.0_018@lull.org> <gnksa_pan-0.8.0_019@lull.org>"
		" <gnksa_pan-0.8.0_020@lull.org> <gnksa_pan-0.8.0_021@lull.org>"
		" <gnksa_pan-0.8.0_022@lull.org> <gnksa_pan-0.8.0_023@lull.org>"
		" <gnksa_pan-0.8.0_024@lull.org> <gnksa_pan-0.8.0_025@lull.org>"
		" <gnksa_pan-0.8.0_026@lull.org> <gnksa_pan-0.8.0_027@lull.org>"
		" <gnksa_pan-0.8.0_028@lull.org> <gnksa_pan-0.8.0_029@lull.org>"
		" <gnksa_pan-0.8.0_030@lull.org> <gnksa_pan-0.8.0_031@lull.org>"
  		" <gnksa_pan-0.8.0_035.12345@lull.org>",

		"<gnksa_pan-0.8.0_001@lull.org>"
		" <gnksa_pan-0.8.0_002@lull.org> <gnksa_pan-0.8.0_003@lull.org>"
		" <gnksa_pan-0.8.0_004@lull.org> <gnksa_pan-0.8.0_005@lull.org>"
		" <gnksa_pan-0.8.0_006@lull.org> <gnksa_pan-0.8.0_007@lull.org>"
		" <gnksa_pan-0.8.0_008@lull.org> <gnksa_pan-0.8.0_009@lull.org>"
		" <gnksa_pan-0.8.0_010@lull.org> <gnksa_pan-0.8.0_011@lull.org>"
		" <gnksa_pan-0.8.0_012@lull.org> <gnksa_pan-0.8.0_013@lull.org>"
		" <gnksa_pan-0.8.0_014@lull.org> <gnksa_pan-0.8.0_015@lull.org>"
		" <gnksa_pan-0.8.0_016@lull.org> <gnksa_pan-0.8.0_017@lull.org>"
		" <gnksa_pan-0.8.0_018@lull.org> <gnksa_pan-0.8.0_019@lull.org>"
		" <gnksa_pan-0.8.0_020@lull.org> <gnksa_pan-0.8.0_021@lull.org>"
		" <gnksa_pan-0.8.0_022@lull.org> <gnksa_pan-0.8.0_023@lull.org>"
		" <gnksa_pan-0.8.0_024@lull.org> <gnksa_pan-0.8.0_025@lull.org>"
		" <gnksa_pan-0.8.0_026@lull.org> <gnksa_pan-0.8.0_027@lull.org>"
		" <gnksa_pan-0.8.0_028@lull.org> <gnksa_pan-0.8.0_029@lull.org>"
		" <gnksa_pan-0.8.0_030@lull.org> <gnksa_pan-0.8.0_031@lull.org>"
  		" <gnksa_pan-0.8.0_035.12345@lull.org>"
	},
	{
		998, /*  gnksa cutoff */

		"<gnksa_pan-0.8.0_001@lull.org>"
		" <gnksa_pan-0.8.1_001@lull.org>"
		" <gnksa_pan-0.8.0_002@lull.org> <gnksa_pan-0.8.0_003@lull.org>"
		" <gnksa_pan-0.8.0_004@lull.org> <gnksa_pan-0.8.0_005@lull.org>"
		" <gnksa_pan-0.8.0_006@lull.org> <gnksa_pan-0.8.0_007@lull.org>"
		" <gnksa_pan-0.8.0_008@lull.org> <gnksa_pan-0.8.0_009@lull.org>"
		" <gnksa_pan-0.8.0_010@lull.org> <gnksa_pan-0.8.0_011@lull.org>"
		" <gnksa_pan-0.8.0_012@lull.org> <gnksa_pan-0.8.0_013@lull.org>"
		" <gnksa_pan-0.8.0_014@lull.org> <gnksa_pan-0.8.0_015@lull.org>"
		" <gnksa_pan-0.8.0_016@lull.org> <gnksa_pan-0.8.0_017@lull.org>"
		" <gnksa_pan-0.8.0_018@lull.org> <gnksa_pan-0.8.0_019@lull.org>"
		" <gnksa_pan-0.8.0_020@lull.org> <gnksa_pan-0.8.0_021@lull.org>"
		" <gnksa_pan-0.8.0_022@lull.org> <gnksa_pan-0.8.0_023@lull.org>"
		" <gnksa_pan-0.8.0_024@lull.org> <gnksa_pan-0.8.0_025@lull.org>"
		" <gnksa_pan-0.8.0_026@lull.org> <gnksa_pan-0.8.0_027@lull.org>"
		" <gnksa_pan-0.8.0_028@lull.org> <gnksa_pan-0.8.0_029@lull.org>"
		" <gnksa_pan-0.8.0_030@lull.org> <gnksa_pan-0.8.0_031@lull.org>"
  		" <gnksa_pan-0.8.0_035.12345@lull.org>",

		"<gnksa_pan-0.8.0_001@lull.org>"
		" <gnksa_pan-0.8.0_002@lull.org> <gnksa_pan-0.8.0_003@lull.org>"
		" <gnksa_pan-0.8.0_004@lull.org> <gnksa_pan-0.8.0_005@lull.org>"
		" <gnksa_pan-0.8.0_006@lull.org> <gnksa_pan-0.8.0_007@lull.org>"
		" <gnksa_pan-0.8.0_008@lull.org> <gnksa_pan-0.8.0_009@lull.org>"
		" <gnksa_pan-0.8.0_010@lull.org> <gnksa_pan-0.8.0_011@lull.org>"
		" <gnksa_pan-0.8.0_012@lull.org> <gnksa_pan-0.8.0_013@lull.org>"
		" <gnksa_pan-0.8.0_014@lull.org> <gnksa_pan-0.8.0_015@lull.org>"
		" <gnksa_pan-0.8.0_016@lull.org> <gnksa_pan-0.8.0_017@lull.org>"
		" <gnksa_pan-0.8.0_018@lull.org> <gnksa_pan-0.8.0_019@lull.org>"
		" <gnksa_pan-0.8.0_020@lull.org> <gnksa_pan-0.8.0_021@lull.org>"
		" <gnksa_pan-0.8.0_022@lull.org> <gnksa_pan-0.8.0_023@lull.org>"
		" <gnksa_pan-0.8.0_024@lull.org> <gnksa_pan-0.8.0_025@lull.org>"
		" <gnksa_pan-0.8.0_026@lull.org> <gnksa_pan-0.8.0_027@lull.org>"
		" <gnksa_pan-0.8.0_028@lull.org> <gnksa_pan-0.8.0_029@lull.org>"
		" <gnksa_pan-0.8.0_030@lull.org> <gnksa_pan-0.8.0_031@lull.org>"
  		" <gnksa_pan-0.8.0_035.12345@lull.org>"
	},
	{
		1024,

		"<gnksa.01@lull.org> <lull.org> <gnksa.03> <@lull.org>"
		" <gnksa.05@> <gnksa.06@lull.org> <>"
		" <gnksa.08@lull.org> <gnksa.09@lull.org"
		" <gnksa.10@lull.org> gnksa.11@lull.org>"
		" <gnksa.12@@lull.org> <gnksa.13@lull.org>.14@lull.org>"
		" gnksa.15@lull.org <gnksa.16@lull.org>",

		"<gnksa.01@lull.org> <gnksa.06@lull.org>"
		" <gnksa.08@lull.org> <gnksa.10@lull.org>"
		" <gnksa.13@lull.org>"
		" <gnksa.16@lull.org>"
	},
	{
		1024,

		"<a@b.a> <d@f.uk> <postmaster@l.uk> <n@o.uk> <@bar.uk>"
		" <foo@bar.com><baz@xyzzy.org> <foo@> <foo@"
		" <blah@trala> <blah@trala.org>",

		"<a@b.a> <d@f.uk> <n@o.uk> <foo@bar.com>"
		" <baz@xyzzy.org> <blah@trala> <blah@trala.org>"
	},
	{
		36,
		"<a@b.uk> <d@f.uk> <h@i.uk> <k@l.uk> <n@o.uk> <q@r.uk>",
		"<a@b.uk> <k@l.uk> <n@o.uk> <q@r.uk>"
	},
	{
		35,
		"<a@b.uk> <d@f.uk> <h@i.uk> <k@l.uk> <n@o.uk> <q@r.uk>",
		"<a@b.uk> <k@l.uk> <n@o.uk> <q@r.uk>"
	},
	{
		34,
		"<a@b.uk> <d@f.uk> <h@i.uk> <k@l.uk> <n@o.uk> <q@r.uk>",
		"<a@b.uk> <n@o.uk> <q@r.uk>"
	},
	{
		32,
		"<a@b.uk> <d@f.uk> <h@i.uk> <k@l.uk> <n@o.uk> <q@r.uk>",
		"<a@b.uk> <n@o.uk> <q@r.uk>",
	},
	{
		27,
		"<a@b.uk> <d@f.uk> <h@i.uk> <k@l.uk> <n@o.uk> <q@r.uk>",
		"<a@b.uk> <n@o.uk> <q@r.uk>",
	},
	{
		26,
		"<a@b.uk> <d@f.uk> <h@i.uk> <k@l.uk> <n@o.uk> <q@r.uk>",
		"<a@b.uk> <n@o.uk> <q@r.uk>",
	},
	{
		25,
		"<a@b.uk> <d@f.uk> <h@i.uk> <k@l.uk> <n@o.uk> <q@r.uk>",
		"<a@b.uk> <q@r.uk>",
	},
	/* make sure that twisted domains like [10.0.0.4] don't get stripped.
	   see comments on pan/base/gnksa.c gnksa_check_message_id() for details. */
	{       
		998,

		"<B8CE15E0.2DBE%frederic.platzer@wanadoo.fr> "
		"<1f9yc83.l59f37ehygn4N%florian@****nachon.net> "
		"<1f9yzfg.1dxhomt17221j4N%mannucci@wild-works.net> "
		"<1fa2kwz.14tkecl3qg8uaN%moi@mapetitentreprise.com> "
		"<1fa7mt7.1t7eiqu1scmm0iN@[10.0.0.4]> "
		"<enlevezca.michel.depeige-E0EA3B.15534806042002@news.wanadoo.fr>",

		"<B8CE15E0.2DBE%frederic.platzer@wanadoo.fr> "
		"<1f9yc83.l59f37ehygn4N%florian@****nachon.net> "
		"<1f9yzfg.1dxhomt17221j4N%mannucci@wild-works.net> "
		"<1fa2kwz.14tkecl3qg8uaN%moi@mapetitentreprise.com> "
		"<1fa7mt7.1t7eiqu1scmm0iN@[10.0.0.4]> "
		"<enlevezca.michel.depeige-E0EA3B.15534806042002@news.wanadoo.fr>"
	}

};

static AddressCheck addresses [] =
{
	{ GNKSA_ILLEGAL_PLAIN_PHRASE, "Charles S. Kerr <charles@foo.com>" },
	{ GNKSA_OK, "\"Charles S. Kerr\" <charles@foo.com>" },
	{ GNKSA_OK, "charles kerr <charles@superpimp.org>" },
	{ GNKSA_OK, "Charles \"Pan Programmer\" Kerr <charles@superpimp.org>" },
	{ GNKSA_OK, "Charles \"Likes, to, put, commas, in, quoted, strings\" Kerr <charles@superpimp.org>" },
	{ GNKSA_OK, "\"Charles Kerr, Pan Programmer\" <charles@superpimp.org>" },
	{ GNKSA_ILLEGAL_PLAIN_PHRASE, "Charles Kerr, Pan Programmer <charles@superpimp.org>" },
	{ GNKSA_INVALID_DOMAIN, "charles kerr <charles>" },
	{ GNKSA_OK, "charles@superpimp.org" },
	{ GNKSA_OK, "charles@superpimp.org (Charles Kerr)" },
	{ GNKSA_SINGLE_DOMAIN, "Charles Kerr <charles@org>" },
	{ GNKSA_SINGLE_DOMAIN, "Charles Kerr <@org>" },
	{ GNKSA_OK, "Charles Kerr <charles@[127.0.0.1]>" },
	{ GNKSA_BAD_DOMAIN_LITERAL, "Charles Kerr <charles@[127..0.1]>" },
	{ GNKSA_BAD_DOMAIN_LITERAL, "Charles Kerr <charles@[127...1]>" },
	{ GNKSA_BAD_DOMAIN_LITERAL, "Charles Kerr <charles@[127.0.0.]>" },
	{ GNKSA_ZERO_LENGTH_LABEL, "<charles@pimp.org.>" },
	{ GNKSA_ILLEGAL_LABEL_HYPHEN, "<charles@pimp-.org>" },
	{ GNKSA_OK, "<charles@pimp.org>" },
	{ GNKSA_OK, "Charles <charles@pimp.asf.fa>" },
	{ GNKSA_OK, "Charles <charles@pimp.asf.uk>" },
	{ GNKSA_ZERO_LENGTH_LABEL, "<charles@>" },
	{ GNKSA_LOCALPART_MISSING, "<@pimp.org>" },
	{ GNKSA_OK, "Charles Kerr <charles@skywalker.ecn.ou.edu>" },
	{ GNKSA_LPAREN_MISSING, "Charles Kerr" },
	{ GNKSA_OK, "looniii@aol.com (Looniii)" },
	{ GNKSA_OK, "Eric <scare.crow@oz.land>" }
};

int
main (void)
{
	/* test addresses */
	if (1)
	{
		gint i;
		gint qty = sizeof(addresses) / sizeof(addresses[0]);
		for (i=0; i!=qty; ++i) {
			const gint retval = gnksa_check_from (addresses[i].address, TRUE);
			if (retval!=addresses[i].expected_retval) return i+1;
		}
		g_message ("PASS email address parsing");
	}

	/* test trimming */
	if (1)
	{
		gint i;
		gint qty = sizeof(references) / sizeof(references[0]);
		for (i=0; i!=qty; ++i) {
			gchar * pch = gnksa_trim_references_to_len (references[i].in, references[i].len);
			if (strcmp(references[i].expected_out, pch)) return 100 + i+1;
			g_free (pch);
		}
		g_message ("PASS trimming references strings");
	}


	/* test message-id generation */
	if (1)
	{
		gchar * id1;
		gchar * id2;

		id1 = gnksa_generate_message_id_from_email_addr ("<foo@bar.com>");
		id2 = gnksa_generate_message_id_from_email_addr ("<foo@bar.com>");
		if (!is_nonempty_string(id1)) return 1;
		if (!is_nonempty_string(id2)) return 1;
		g_free (id1);
		g_free (id2);

		id1 = gnksa_generate_message_id_from_email_addr ("Joe <joe@bar.com>");
		g_message ("[%s]", id1);
		if (!is_nonempty_string(id1)) return 1;
		if (!strstr(id1,"bar")) return 1;
		g_free (id1);

		id1 = gnksa_generate_message_id_from_email_addr ("zzz.com");
		g_message ("[%s]", id1);
		if (!is_nonempty_string(id1)) return 1;
		if (!strstr(id1,"zzz")) return 1;
		g_free (id1);

		id1 = gnksa_generate_message_id_from_email_addr ("@bar.com>");
		g_message ("[%s]", id1);
		if (!is_nonempty_string(id1)) return 1;
		if (!strstr(id1,"bar")) return 1;
		g_free (id1);

		g_message ("PASS gnksa_generate_message_id_from_email_addr");
	}

	if (1)
	{
		gint i = test_generate_references ();
		if (i != 0)
			return i;
	}

	g_message ("All GNKSA tests passed");

	return 0;
}
