#ifndef __KMHEADERS
#define __KMHEADERS

#include <qwidget.h>
#include <qstrlist.h>
#include <klistview.h>
#include <kfoldertree.h>
#include <qmemarray.h>
#include <qmap.h>
#include <qdragobject.h>
#include <qdict.h>
#include "kmmessage.h"
#include "kmime_util.h"
#include <kpopupmenu.h>

class KMFolder;
class KMMessage;
class KMMsgBase;
class KMMainWin;
class QPalette;
class KMHeaderItem;
class QPixmap;
class QIconSet;
class QDateTime;

typedef QPtrList<KMMsgBase> KMMessageList;
typedef QMap<int,KMFolder*> KMMenuToFolder;

/** The widget that shows the contents of folders */
#define KMHeadersInherited KListView
class KMHeaders : public KListView
{
  Q_OBJECT
  friend class KMHeaderItem; // For easy access to the pixmaps

public:
  KMHeaders(KMMainWin *owner, QWidget *parent=0, const char *name=0);
  virtual ~KMHeaders();

  /** A new folder has been selected update the list of headers shown */
  virtual void setFolder(KMFolder *, bool jumpToFirst = false);

  /** Return the folder whose message headers are being displayed */
  KMFolder* folder(void) { return mFolder; }

  /** read the config file and update nested state if necessary */
  void refreshNestedState(void);

  /** Set current message. If id<0 then the first message is shown,
    if id>count() the last message is shown. */
  virtual void setCurrentMsg(int msgId);

  /** Get a list of all items in the current thread */
  QPtrList<QListViewItem> currentThread() const;

  /** Set all messages in the current thread to status @p status */
  virtual void setThreadStatus(KMMsgStatus status);

  /** The following methods process the message in the folder with
    the given msgId, or if no msgId is given all selected
    messages are processed. */
  virtual void setMsgStatus(KMMsgStatus status, int msgId=-1);
  virtual void setMsgRead(int msgId=-1);
  virtual void deleteMsg(int msgId=-1);
  virtual void applyFiltersOnMsg(int msgId=-1);
  virtual void undo();
  virtual bool canUndo() const;
  virtual void resendMsg();
  virtual void prepareMove( KMMsgBase **curMsg, int *contentX, int *contentY );
  virtual void finalizeMove( KMMsgBase *curMsg, int contentX, int contentY );

  /** If destination is 0 then the messages are deleted, otherwise
    they are moved to this folder. */
  virtual void moveMsgToFolder(KMFolder* destination, int msgId=-1);

  /** Messages are duplicated and added to given folder.
      If aMsg is set this one will be written to the destination folder. */
  virtual void copyMsgToFolder(KMFolder* destination,
                               int msgId=-1,
                               KMMessage* aMsg = 0);

 /** Returns list of selected messages or a list with the message with
    the given Id if msgId >= 0. Do not delete the returned list. */
  virtual KMMessageList* selectedMsgs(int msgId=-1);

  /** Returns message with given id or current message if no
    id is given. First call with msgId==-1 returns first
    selected message, subsequent calls with no argument
    return the following selected messages. */
  KMMessage* getMsg (int msgId=-2);

  /** Returns index of message returned by last getMsg() call */
  int indexOfGetMsg (void) const { return getMsgIndex; }

  /** Returns pointer to owning main window. */
  KMMainWin* owner(void) const { return mOwner; }

  /** PaintInfo pointer */
  const KPaintInfo *paintInfo(void) const { return &mPaintInfo; }

  /** Read config options. */
  virtual void readConfig(void);

  /** Read color options and set palette. */
  virtual void readColorConfig(void);

  /** Refresh the list of message headers shown */
  virtual void reset(void);

  /** Scroll to show new mail */
  void showNewMail();

  /** Return the current message */
  virtual KMMessage* currentMsg();
  /** Return the current list view item */
  virtual KMHeaderItem* currentHeaderItem();
  /** Return the index of the message corresponding to the current item */
  virtual int currentItemIndex();
  /** Set the current item to the one corresponding to the given msg id */
  virtual void setCurrentItemByIndex( int msgIdx );
  /** Return the message id of the top most visible item */
  virtual int topItemIndex();
  /** Make the item corresponding to the message with the given id the
      top most visible item. */
  virtual void setTopItemByIndex( int aMsgIdx );
  virtual void setNestedOverride( bool override );
  /** Double force items to always be open */
  virtual void setOpen ( QListViewItem *, bool );

  int getNestingPolicy() const { return nestingPolicy; }
  /** Returns true if the current header list is threaded. */
  bool isThreaded() const {
    return mNested != mNestedOverride; // xor
  }

  /** Find next/prev unread message. Starts at currentItem() if startAt
    is unset. */
  virtual int findUnread(bool findNext, int startAt=-1, bool onlyNew = false, bool acceptCurrent = false);

  void highlightMessage(QListViewItem*, bool markitread);

  /** return a string relativ to the current time */
  static QString fancyDate( time_t otime );

  QFont dateFont;

  bool noRepaint;

  // filter events for popup
  bool eventFilter ( QObject *o, QEvent *e );

signals:
  /** emitted when the list view item corresponding to this message
      has been selected */
  virtual void selected(KMMessage *);
  /** emitted when the list view item corresponding to this message
      has been double clicked */
  virtual void activated(KMMessage *);
  /** emitted when we might be about to delete messages */
  virtual void maybeDeleting();

public slots:
  void workAroundQListViewLimitation();

  /** For when a list view item has been double clicked */
  void selectMessage(QListViewItem*);
  /** For nested message view, recusively add all children of a message */
  void recursivelyAddChildren( int i, KMHeaderItem *parent );
  /** For when a list view item has been selected */
  void highlightMessage(QListViewItem*);
  /** For when righ mouse button is pressed */
  void slotRMB();
  /** Refresh list view item corresponding to the messae with the given id */
  void msgHeaderChanged(int msgId);
  /** For when the list of messages in a folder has changed */
  void msgChanged();
  /** For when the message with the given message id has been added to a folder */
  void msgAdded(int);
  /** For when the message with the given id has been removed for a folder */
  void msgRemoved(int,QString);
  /** Make the next header visible scrolling if necessary */
  void nextMessage();
  /** Same as nextMessage() but don't clear the current selection */
  void selectNextMessage();
  /** Make the previous header visible scrolling if necessary */
  void prevMessage();
  /** Same as prevMessage() but don't clear the current selection */
  void selectPrevMessage();
  /** Make the nextUnread message header visible scrolling if necessary */
  void nextUnreadMessage(bool acceptCurrent = false);
  /** Make the previous message header visible scrolling if necessary */
  void prevUnreadMessage();
  /** Don't show a drag cursor */
  void slotNoDrag();
  /** timer function to set the current time regularly */
  void resetCurrentTime();

  /** Expands (@p expand == true) or collapses (@p expand == false)
      the current thread. */
  void slotExpandOrCollapseThread( bool expand );
  /** Expands (@p expand == true) or collapses (@p expand == false)
      all threads */
  void slotExpandOrCollapseAllThreads( bool expand );

  virtual void ensureCurrentItemVisible();

  /** switch size-column */
  void slotToggleSizeColumn();

  /** Provide information about number of messages in a folder */
  void setFolderInfoStatus();

protected:
  static QPixmap *pixNew, *pixUns, *pixDel, *pixOld, *pixRep, *pixSent,
    *pixQueued, *pixFwd, *pixFlag,
    *pixFullySigned, *pixPartiallySigned, *pixUndefinedSigned,
    *pixFullyEncrypted, *pixPartiallyEncrypted, *pixUndefinedEncrypted,
      *pixFiller, *pixEncryptionProblematic,
      *pixSignatureProblematic;

  /** Look for color changes */
  virtual bool event(QEvent *e);

  /** Overridden to support backing pixmap */
  virtual void paintEmptyArea( QPainter * p, const QRect & rect );

  /** Ensure the current item is visible */
  void makeHeaderVisible();

  /** Auxillary method to findUnread */
  void findUnreadAux( KMHeaderItem*&, bool &, bool, bool );

  /** Returns message index of first selected message of the messages
    where the message with the given id is in. This for finding the correct
    message that shall be the current message after move/delete of multiple
    messages. */
  virtual int firstSelectedMsg() const;

  /** Read per-folder config options and apply them. */
  virtual void readFolderConfig(void);

  /** Write per-folder config options. */
  virtual void writeFolderConfig(void);

  /** Write global config options. */
  virtual void writeConfig(void);

  /** Handle shift and control selection */
  virtual void contentsMousePressEvent(QMouseEvent*);
  virtual void contentsMouseReleaseEvent(QMouseEvent* e);
  virtual void keyPressEvent( QKeyEvent * e );

  /** Unselect all items except one */
  virtual void clearSelectionExcept( QListViewItem *exception );

  /** Select all items in list from begin to end, return FALSE
     if end occurs before begin in the list */
  virtual bool shiftSelection( QListViewItem *begin, QListViewItem *end );

  /** Called when a header is clicked */
  virtual void setSorting( int column, bool ascending = TRUE);

  /** To initiate a drag operation */
  void contentsMouseMoveEvent( QMouseEvent *e );

protected slots:
  /** Move messages corresponding to the selected items to the folder
      corresponding to the given menuId */
  virtual void moveSelectedToFolder( int menuId );
  /** Same thing but copy */
  virtual void copySelectedToFolder( int menuId );
  /** Apply the filter Rules to a single message */
  virtual int slotFilterMsg( KMMessage * );
  /** dirties the sort order */
  void dirtySortOrder(int);
  /** show context menu */
  void rightButtonPressed( QListViewItem *, const QPoint &, int );

private:
  /** Is equivalent to clearing the list and inserting an item for
      each message in the current folder */
  virtual void updateMessageList(bool set_selection=FALSE);

  /** Currently associated folder */
  KMFolder* mFolder;
  /** The KMMainWin for status bar updates */
  KMMainWin* mOwner;
  /** Top most visible item */
  int mTopItem;
  /** Current item */
  int mCurrentItem;
  /** Map messages ids into KMHeaderItems */
  QMemArray<KMHeaderItem*> mItems;
  QDict< KMHeaderItem > mIdTree;
  QDict< KMHeaderItem > mPhantomIdTree;
  QDict< QValueList< int > > mTree;
  QDict< bool > mTreeSeen;
  QDict< bool > mTreeToplevel;
  bool mNested, mNestedOverride;
  int nestingPolicy;

  /** These must replaced by something better! */
  static bool mTrue, mFalse;

  /** are we currently showing the size field? */
  bool showingSize;
  /** Updated as side effect of KMHeaders::getMsg */
  int getMsgIndex;
  /** ditto */
  bool getMsgMulti;
  /** ditto */
  KMHeaderItem* getMsgItem;
  /** @ref KMHeaders::selectedMsgs isn't reentrant */
  KMMessageList mSelMsgBaseList;
  QPtrList<KMMessage> mSelMsgList;
  KMHeaderItem* mPrevCurrent;

  /** For shift selection */
  QListViewItem *beginSelection, *endSelection;
  /** Current colours and backing pixmap */
  KPaintInfo mPaintInfo;

  int mSortCol;
  bool mSortDescending;

  /** Icons shown in header */
  static QIconSet *up, *down;
  /** Map menu id into a folder */
  KMMenuToFolder mMenuToFolder;

  /** Drag and drop support */
  bool mousePressed;
  /** ditto */
  QPoint presspos;

  struct {
      uint ascending : 1;
      uint dirty : 1;
      short column;
      uint fakeSort : 1;
      uint removed : 1;
  } mSortInfo;
  void appendUnsortedItem(KMHeaderItem *);
  bool writeSortOrder();
  bool readSortOrder(bool set_selection=FALSE);

  KMime::DateFormatter mDate;
  /** value of config key Behaviour/LoopOnGotoUnread */
  bool mLoopOnGotoUnread;
  bool mJumpToUnread;

  /** popup to switch columns */
  KPopupMenu* mPopup;
  int mSizeColumn;
};

#endif
